// Copyright 2017-2023, Nicholas Sharp and the Polyscope contributors. https://polyscope.run

#include "polyscope/render/color_maps.h"

#include "polyscope/render/colormap_defs.h"
#include "polyscope/render/engine.h"

#include "imgui.h"
#include "polyscope/polyscope.h"

namespace polyscope {

void loadColorMap(std::string cmapName, std::string filename) { render::engine->loadColorMap(cmapName, filename); }

namespace render {

// ImGUI helper to select a colormap. Returns true if the selection changed
bool buildColormapSelector(std::string& cm, std::string fieldName) {
  bool changed = false;

  ImGui::PushItemWidth(125 * options::uiScale);

  if (ImGui::BeginCombo(fieldName.c_str(), cm.c_str())) {
    for (auto& c : render::engine->colorMaps) {
      if (ImGui::Selectable(c->name.c_str(), c->name == cm)) {
        changed = true;
        cm = c->name;
      }
    }
    ImGui::EndCombo();
  }

  ImGui::PopItemWidth();

  return changed;
};


// === color map data buffers

// clang-format off

const std::vector<glm::vec3> CM_GRAY =
{{5.119113838889112e-07,2.0522701956616554e-06,5.982577941045683e-06},{5.119113838889112e-07,2.0522701956616554e-06,5.982577941045683e-06},{0.0002598440803609316,0.00026259440245361075,0.00027706228396051196},{0.0002598440803609316,0.00026259440245361075,0.00027706228396051196},{0.0008656156394278344,0.0008570638031131798,0.0008794207369001088},{0.0008656156394278344,0.0008570638031131798,0.0008794207369001088},{0.001769403238430108,0.0017403680581581408,0.0017686070963996048},{0.001769403238430108,0.0017403680581581408,0.0017686070963996048},{0.002949472160671299,0.002891903379202927,0.002924180394823832},{0.002949472160671299,0.002891903379202927,0.002924180394823832},{0.004392323531342204,0.004298904740810658,0.004333431047933434},{0.004392323531342204,0.004298904740810658,0.004333431047933434},{0.00608855311595789,0.005952519318106651,0.005987527380406318},{0.00608855311595789,0.005952519318106651,0.005987527380406318},{0.008031191715623246,0.00784622292261931,0.00787995029696465},{0.008031191715623246,0.00784622292261931,0.00787995029696465},{0.010214875235272093,0.00997502608955014,0.010005706087016208},{0.010214875235272093,0.00997502608955014,0.010005706087016208},{0.012635372261218893,0.012335021633001353,0.012360877171488911},{0.012635372261218893,0.012335021633001353,0.012360877171488911},{0.015289290743098349,0.014923103611522015,0.014942342882390302},{0.015289290743098349,0.014923103611522015,0.014942342882390302},{0.018173884067540134,0.01773678153559083,0.017747594833936085},{0.018173884067540134,0.01773678153559083,0.017747594833936085},{0.02128691657088564,0.02077405157770593,0.02077460893832101},{0.02128691657088564,0.02077405157770593,0.02077460893832101},{0.02462656669738433,0.024033303900485702,0.024021753317628538},{0.02462656669738433,0.024033303900485702,0.024021753317628538},{0.028191355111633225,0.027513253929353176,0.027487720010651605},{0.028191355111633225,0.027513253929353176,0.027487720010651605},{0.031980089978436425,0.031212890095399095,0.03117147304311679},{0.031980089978436425,0.031212890095399095,0.03117147304311679},{0.03599182442355884,0.03513143325966101,0.035072208100447595},{0.03599182442355884,0.03513143325966101,0.035072208100447595},{0.04022582286584988,0.03926830463975649,0.03918932064337804},{0.04022582286584988,0.03926830463975649,0.03918932064337804},{0.044490079205597936,0.0435012249064101,0.04340640025397465},{0.044490079205597936,0.0435012249064101,0.04340640025397465},{0.04870367428407673,0.047678466927454234,0.047567902696042295},{0.052873620993883326,0.05181334368492382,0.051687053963736816},{0.052873620993883326,0.05181334368492382,0.051687053963736816},{0.057003147469548036,0.055909012523011246,0.05576700147480037},{0.057003147469548036,0.055909012523011246,0.05576700147480037},{0.061095137234846185,0.05996829329212761,0.05981055680351737},{0.061095137234846185,0.05996829329212761,0.05981055680351737},{0.06515218007540821,0.06399371820125457,0.0638202452197574},{0.06515218007540821,0.06399371820125457,0.0638202452197574},{0.06917661353844734,0.06798757248509635,0.06779834611081526},{0.06917661353844734,0.06798757248509635,0.06779834611081526},{0.07317055710313111,0.07195192788781538,0.07174692627103735},{0.07317055710313111,0.07195192788781538,0.07174692627103735},{0.07713594055593873,0.07588867046670467,0.07566786755046528},{0.07713594055593873,0.07588867046670467,0.07566786755046528},{0.08107452773767657,0.07979952385914274,0.0795628899969453},{0.08107452773767657,0.07979952385914274,0.0795628899969453},{0.08498793655960166,0.08368606889245686,0.08343357136470589},{0.08498793655960166,0.08368606889245686,0.08343357136470589},{0.0888776559863606,0.08754976022063193,0.08728136366834671},{0.0888776559863606,0.08754976022063193,0.08728136366834671},{0.0927450605335162,0.09139194052486757,0.09110760731542752},{0.0927450605335162,0.09139194052486757,0.09110760731542752},{0.09659142271361737,0.09521385270343014,0.09491354324016602},{0.09659142271361737,0.09521385270343014,0.09491354324016602},{0.10041792377751321,0.09901665039071003,0.09870032337586748},{0.10041792377751321,0.09901665039071003,0.09870032337586748},{0.10422566303005368,0.10280140707916763,0.10246901973797143},{0.10422566303005368,0.10280140707916763,0.10246901973797143},{0.10801566594658954,0.10656912406613869,0.10622063233826792},{0.10801566594658954,0.10656912406613869,0.10622063233826792},{0.11178889127514621,0.11032073740675324,0.10995609611040333},{0.11178889127514621,0.11032073740675324,0.10995609611040333},{0.11554623727620819,0.11405712402191859,0.1136762869947188},{0.11554623727620819,0.11405712402191859,0.1136762869947188},{0.11928854722572543,0.11777910708450154,0.11738202730482386},{0.11928854722572543,0.11777910708450154,0.11738202730482386},{0.12301661428577407,0.12148746078607905,0.12107409047767947},{0.12301661428577407,0.12148746078607905,0.12107409047767947},{0.12673118583015935,0.1251829145698134,0.12475320529226316},{0.13043296729827977,0.12886615690131367,0.12842005962827954},{0.13043296729827977,0.12886615690131367,0.12842005962827954},{0.13412262563913602,0.13253783863813123,0.132075303825237},{0.13412262563913602,0.13253783863813123,0.132075303825237},{0.13780079239794768,0.13619857604929606,0.13571955369302932},{0.13780079239794768,0.13619857604929606,0.13571955369302932},{0.14146806649004226,0.13984895352866278,0.13935339321756454},{0.14146806649004226,0.13984895352866278,0.13935339321756454},{0.14512501670019584,0.14348952603947387,0.14297737699866214},{0.14512501670019584,0.14348952603947387,0.14297737699866214},{0.14877218394019315,0.14712082132223708,0.14659203245216168},{0.14877218394019315,0.14712082132223708,0.14659203245216168},{0.15241008329281713,0.15074334189355915,0.15019786180375175},{0.15241008329281713,0.15074334189355915,0.15019786180375175},{0.1560392058666605,0.15435756685982158,0.15379534389829613},{0.1560392058666605,0.15435756685982158,0.15379534389829613},{0.15966002048290173,0.15796395356641008,0.15738493584527577},{0.15966002048290173,0.15796395356641008,0.15738493584527577},{0.1632729752124384,0.1615629391005102,0.16096707451828082},{0.1632729752124384,0.1615629391005102,0.16096707451828082},{0.16687849877942876,0.1651549416631864,0.16454217792420187},{0.16687849877942876,0.1651549416631864,0.16454217792420187},{0.17047700184528455,0.16874036182449317,0.16811064645581555},{0.17047700184528455,0.16874036182449317,0.16811064645581555},{0.17406887818543398,0.17231958367368572,0.17167286403977897},{0.17406887818543398,0.17231958367368572,0.17167286403977897},{0.17765450576970573,0.17589297587514247,0.17522919919060903},{0.17765450576970573,0.17589297587514247,0.17522919919060903},{0.18123424775588307,0.17946089263936252,0.17878000597996957},{0.18123424775588307,0.17946089263936252,0.17878000597996957},{0.1848084534049017,0.1830236746173137,0.1823256249295161},{0.1848084534049017,0.1830236746173137,0.1823256249295161},{0.18837745892516847,0.1865816497254701,0.18586638383460624},{0.18837745892516847,0.1865816497254701,0.18586638383460624},{0.19194158825267513,0.190135133908052,0.18940259852537042},{0.19194158825267513,0.190135133908052,0.18940259852537042},{0.19550115377282457,0.1936844318422727,0.19293457357092483},{0.19550115377282457,0.1936844318422727,0.19293457357092483},{0.1990564569892651,0.1972298375917655,0.1964626029318854},{0.20260778914445177,0.2007716352128197,0.1999869705657933},{0.20260778914445177,0.2007716352128197,0.1999869705657933},{0.2061554317961845,0.2043100993175656,0.2035079509895849},{0.2061554317961845,0.2043100993175656,0.2035079509895849},{0.20969965735391027,0.20784549559783222,0.2070258098028131},{0.20969965735391027,0.20784549559783222,0.2070258098028131},{0.21324072957821566,0.2113780813130166,0.21054080417495308},{0.21324072957821566,0.2113780813130166,0.21054080417495308},{0.2167789040465809,0.21490810574497904,0.2140531832997964},{0.2167789040465809,0.21490810574497904,0.2140531832997964},{0.22031442858818173,0.2184358106226793,0.21756318881964226},{0.22031442858818173,0.2184358106226793,0.21756318881964226},{0.22384754369024062,0.22196143051900868,0.22107105522173387},{0.22384754369024062,0.22196143051900868,0.22107105522173387},{0.22737848287820317,0.22548519322203997,0.22457701020915594},{0.22737848287820317,0.22548519322203997,0.22457701020915594},{0.23090747307179832,0.22900732008271107,0.2280812750482047},{0.23090747307179832,0.22900732008271107,0.2280812750482047},{0.2344347349188553,0.23252802634076847,0.23158406489405264},{0.2344347349188553,0.23252802634076847,0.23158406489405264},{0.23796048310857054,0.2360475214306363,0.2350855890963675},{0.23796048310857054,0.2360475214306363,0.2350855890963675},{0.24148492666578375,0.23956600926872218,0.2385860514863974},{0.24148492666578375,0.23956600926872218,0.2385860514863974},{0.24500826922766467,0.24308368852354312,0.24208565064689802},{0.24500826922766467,0.24308368852354312,0.24208565064689802},{0.24853070930410476,0.24660075286992894,0.2455845801661603},{0.24853070930410476,0.24660075286992894,0.2455845801661603},{0.2520524405229937,0.250117391228459,0.2490830288772885},{0.2520524405229937,0.250117391228459,0.2490830288772885},{0.2555736518614572,0.25363378799118214,0.2525811810837814},{0.2555736518614572,0.25363378799118214,0.2525811810837814},{0.2590945278640408,0.25715012323459174,0.25607921677238027},{0.2590945278640408,0.25715012323459174,0.25607921677238027},{0.26261524884875664,0.26066657292073886,0.2595773118140703},{0.26261524884875664,0.26066657292073886,0.2595773118140703},{0.2661359911018203,0.2641833090872994,0.2630756381540484},{0.2696569270618427,0.2677005000273448,0.26657436399140455},{0.2696569270618427,0.2677005000273448,0.26657436399140455},{0.2731782254941878,0.27121831045950634,0.27007365394920596},{0.2731782254941878,0.27121831045950634,0.27007365394920596},{0.2767000516561452,0.27473690168916753,0.27357366923561866},{0.2767000516561452,0.27473690168916753,0.27357366923561866},{0.2802225674535131,0.27825643176127346,0.2770745677966528},{0.2802225674535131,0.27825643176127346,0.2770745677966528},{0.2837459315891544,0.28177705560529565,0.2805765044610716},{0.2837459315891544,0.28177705560529565,0.2805765044610716},{0.2872702997040329,0.28529892517285693,0.2840796310779651},{0.2872702997040329,0.28529892517285693,0.2840796310779651},{0.29079582451120206,0.2888221895684777,0.2875840966474506},{0.29079582451120206,0.2888221895684777,0.2875840966474506},{0.2943226559231934,0.2923469951738719,0.2910900474449293},{0.2943226559231934,0.2923469951738719,0.2910900474449293},{0.2978509411732042,0.29587348576619366,0.29459762713929627},{0.2978509411732042,0.29587348576619366,0.29459762713929627},{0.30138082493046714,0.2994018026306029,0.29810697690547344},{0.30138082493046714,0.2994018026306029,0.29810697690547344},{0.304912449410152,0.3029320846674895,0.3016182355316066},{0.304912449410152,0.3029320846674895,0.3016182355316066},{0.30844595447811946,0.30646446849468284,0.3051315395212463},{0.30844595447811946,0.30646446849468284,0.3051315395212463},{0.3119814777508493,0.3099990885449341,0.30864702319080917},{0.3119814777508493,0.3099990885449341,0.30864702319080917},{0.31551915469079933,0.31353607715895315,0.3121648187625949},{0.31551915469079933,0.31353607715895315,0.3121648187625949},{0.3190591186974825,0.3170755646742604,0.31568505645361644},{0.3190591186974825,0.3170755646742604,0.31568505645361644},{0.322601501194492,0.32061767951008274,0.31920786456048234},{0.322601501194492,0.32061767951008274,0.31920786456048234},{0.3261464317127154,0.32416254824853064,0.3227333695405576},{0.3261464317127154,0.32416254824853064,0.3227333695405576},{0.3296940379699382,0.3277102957122579,0.3262616960896098},{0.3296940379699382,0.3277102957122579,0.3262616960896098},{0.3332444459470546,0.33126104503880144,0.32979296721613593},{0.3332444459470546,0.33126104503880144,0.32979296721613593},{0.3367977799610607,0.3348149177517828,0.3333273043125561},{0.34035416273500496,0.33837203382914927,0.33686482722343924},{0.34035416273500496,0.33837203382914927,0.33686482722343924},{0.34391371546506766,0.3419325117686051,0.34040565431092945},{0.34391371546506766,0.3419325117686051,0.34040565431092945},{0.3474765578849235,0.34549646865039063,0.3439499025175156},{0.3474765578849235,0.34549646865039063,0.3439499025175156},{0.3510428083275172,0.3490640201975498,0.34749768742628845},{0.3510428083275172,0.3490640201975498,0.34749768742628845},{0.35461258378441374,0.35263528083381307,0.3510491233188204},{0.35461258378441374,0.35263528083381307,0.3510491233188204},{0.35818599996282147,0.3562103637392294,0.3546043232307861},{0.35818599996282147,0.3562103637392294,0.3546043232307861},{0.36176317134043134,0.359789380903655,0.35816339900544786},{0.36176317134043134,0.359789380903655,0.35816339900544786},{0.3653442112181758,0.3633724431782132,0.3617264613451084},{0.3653442112181758,0.3633724431782132,0.3617264613451084},{0.36892923177100456,0.366959660324833,0.3652936198606414},{0.36892923177100456,0.366959660324833,0.3652936198606414},{0.37251834409679924,0.3705511410639528,0.36886498311919264},{0.37251834409679924,0.3705511410639528,0.36886498311919264},{0.37611165826349635,0.3741469931204934,0.37244065869014525},{0.37611165826349635,0.3741469931204934,0.37244065869014525},{0.37970928335452314,0.3777473232681791,0.3760207531894346},{0.37970928335452314,0.3777473232681791,0.3760207531894346},{0.38331132751262736,0.38135223737229057,0.37960537232229685},{0.38331132751262736,0.38135223737229057,0.37960537232229685},{0.38691789798217374,0.38496184043092785,0.38319462092452544},{0.38691789798217374,0.38496184043092785,0.38319462092452544},{0.39052910114998884,0.3885762366148534,0.38678860300230933},{0.39052910114998884,0.3885762366148534,0.38678860300230933},{0.39414504258482247,0.39219552930598844,0.39038742177072233},{0.39414504258482247,0.39219552930598844,0.39038742177072233},{0.39776582707548697,0.39581982113462366,0.3939911796909281},{0.39776582707548697,0.39581982113462366,0.3939911796909281},{0.4013915586677524,0.3994492140154045,0.3975999785061618},{0.4013915586677524,0.3994492140154045,0.3975999785061618},{0.405022340700029,0.4030838091821594,0.4012139192765472},{0.405022340700029,0.4030838091821594,0.4012139192765472},{0.4086582758379367,0.4067237072216088,0.40483310241280485},{0.41229946610776325,0.4103690081060248,0.4084576277089013},{0.41229946610776325,0.4103690081060248,0.4084576277089013},{0.4159460129289094,0.41401981122487985,0.41208759437369297},{0.4159460129289094,0.41401981122487985,0.41208759437369297},{0.4195980171453375,0.417676215415535,0.4157231010616072},{0.4195980171453375,0.417676215415535,0.4157231010616072},{0.4232555790560795,0.4213383189930122,0.4193642459024059},{0.4232555790560795,0.4213383189930122,0.4193642459024059},{0.4269187984448541,0.42500621977889325,0.4230111265300785},{0.4269187984448541,0.42500621977889325,0.4230111265300785},{0.43058777460882475,0.4286800151293829,0.4266638401108987},{0.43058777460882475,0.4286800151293829,0.4266638401108987},{0.43426260638653963,0.43235980196258117,0.4303224833706848},{0.43426260638653963,0.43235980196258117,0.4303224833706848},{0.4379433921850948,0.43604567678499145,0.4339871526213026},{0.4379433921850948,0.43604567678499145,0.4339871526213026},{0.4416302300065532,0.43973773571730596,0.4376579437864427},{0.4416302300065532,0.43973773571730596,0.4376579437864427},{0.4453232174736505,0.44343607451950195,0.4413349524267044},{0.4453232174736505,0.44343607451950195,0.4413349524267044},{0.4490224518548273,0.44714078861527135,0.44501827376402053},{0.4490224518548273,0.44714078861527135,0.44501827376402053},{0.4527280300886078,0.4508519731158264,0.4487080027054508},{0.4527280300886078,0.4508519731158264,0.4487080027054508},{0.4564400488073677,0.4545697228430991,0.4524042338663719},{0.4564400488073677,0.4545697228430991,0.4524042338663719},{0.460158604360502,0.4582941323523669,0.45610706159309317},{0.460158604360502,0.4582941323523669,0.45610706159309317},{0.4638837928370361,0.462025295954328,0.45981657998492237},{0.4638837928370361,0.462025295954328,0.45981657998492237},{0.46761571008769504,0.46576330773665214,0.46353288291570477},{0.46761571008769504,0.46576330773665214,0.46353288291570477},{0.4713544517464567,0.4695082615850323,0.4672560640548638},{0.4713544517464567,0.4695082615850323,0.4672560640548638},{0.47510011325161067,0.47326025120375664,0.4709862168879611},{0.47510011325161067,0.47326025120375664,0.4709862168879611},{0.47885278986635954,0.4770193701358209,0.4747234347367984},{0.48261257669895335,0.4807857117826079,0.4784678107790839},{0.48261257669895335,0.4807857117826079,0.4784678107790839},{0.48637956872241056,0.48455936942314876,0.48221943806768125},{0.48637956872241056,0.48455936942314876,0.48221943806768125},{0.4901538607938191,0.4883404362329858,0.48597840954945876},{0.4901538607938191,0.4883404362329858,0.48597840954945876},{0.4939355476732588,0.49212900530265485,0.4897448180837611},{0.4939355476732588,0.49212900530265485,0.4897448180837611},{0.497724724042333,0.4959251696558081,0.49351875646051496},{0.497724724042333,0.4959251696558081,0.49351875646051496},{0.5015214845223622,0.4997290222669866,0.4973003174179899},{0.5015214845223622,0.4997290222669866,0.4973003174179899},{0.5053259236922266,0.5035406560790654,0.5010895936602286},{0.5053259236922266,0.5035406560790654,0.5010895936602286},{0.5091381361058909,0.5073601640203813,0.5048866778741612},{0.5091381361058909,0.5073601640203813,0.5048866778741612},{0.5129582163096104,0.5111876390215643,0.5086916627464181},{0.5129582163096104,0.5111876390215643,0.5086916627464181},{0.5167862588588553,0.5150231740320771,0.5125046409798591},{0.5167862588588553,0.5150231740320771,0.5125046409798591},{0.520622358334941,0.5188668620364875,0.5163257053098251},{0.520622358334941,0.5188668620364875,0.5163257053098251},{0.5244666093614033,0.5227187960704766,0.5201549485201331},{0.5244666093614033,0.5227187960704766,0.5201549485201331},{0.5283191066201067,0.5265790692366017,0.5239924634588228},{0.5283191066201067,0.5265790692366017,0.5239924634588228},{0.5321799448671195,0.5304477747198241,0.5278383430536652},{0.5321799448671195,0.5304477747198241,0.5278383430536652},{0.5360492189483509,0.5343250058028127,0.5316926803274503},{0.5360492189483509,0.5343250058028127,0.5316926803274503},{0.5399270238149753,0.5382108558810341,0.5355555684130593},{0.5399270238149753,0.5382108558810341,0.5355555684130593},{0.5438134545386418,0.5421054184776435,0.5394271005683371},{0.5438134545386418,0.5421054184776435,0.5394271005683371},{0.5477086063264959,0.546008787258179,0.5433073701907696},{0.5477086063264959,0.546008787258179,0.5433073701907696},{0.5516125745359979,0.5499210560450836,0.547196470831983},{0.5516125745359979,0.5499210560450836,0.547196470831983},{0.5555254546895829,0.5538423188320419,0.5510944962120673},{0.5594473424891353,0.5577726697981703,0.55500154023374},{0.5594473424891353,0.5577726697981703,0.55500154023374},{0.5633783338303158,0.5617122033220392,0.558917696996353},{0.5633783338303158,0.5617122033220392,0.558917696996353},{0.5673185248167354,0.5656610139955613,0.562843060809758},{0.5673185248167354,0.5656610139955613,0.562843060809758},{0.5712680117739896,0.569619196637736,0.5667777262080311},{0.5712680117739896,0.569619196637736,0.5667777262080311},{0.5752268912635539,0.5735868463082713,0.5707217879630734},{0.5752268912635539,0.5735868463082713,0.5707217879630734},{0.5791952600965621,0.5775640583210816,0.5746753410980865},{0.5791952600965621,0.5775640583210816,0.5746753410980865},{0.5831732153474625,0.5815509282576771,0.5786384809009386},{0.5831732153474625,0.5815509282576771,0.5786384809009386},{0.5871608543675724,0.5855475519804424,0.5826113029374229},{0.5871608543675724,0.5855475519804424,0.5826113029374229},{0.5911582747985198,0.5895540256458232,0.5865939030644199},{0.5911582747985198,0.5895540256458232,0.5865939030644199},{0.5951655745855984,0.5935704457174206,0.5905863774429682},{0.5951655745855984,0.5935704457174206,0.5905863774429682},{0.5991828519910354,0.5975969089789966,0.594588822551246},{0.5991828519910354,0.5975969089789966,0.594588822551246},{0.6032102056071761,0.601633512547412,0.5986013351974859},{0.6032102056071761,0.601633512547412,0.5986013351974859},{0.6072477343695865,0.6056803538854836,0.6026240125328051},{0.6072477343695865,0.6056803538854836,0.6026240125328051},{0.6112955375701088,0.6097375308147855,0.6066569520639827},{0.6112955375701088,0.6097375308147855,0.6066569520639827},{0.6153537148698354,0.6138051415283824,0.6107002516661683},{0.6153537148698354,0.6138051415283824,0.6107002516661683},{0.6194223663120388,0.6178832846035217,0.6147540095955469},{0.6194223663120388,0.6178832846035217,0.6147540095955469},{0.6235015923350435,0.621972059014272,0.6188183245019492},{0.6235015923350435,0.621972059014272,0.6188183245019492},{0.6275914937850678,0.6260715641441202,0.6228932954414248},{0.6275914937850678,0.6260715641441202,0.6228932954414248},{0.6316921719290114,0.6301818997985384,0.626979021888779},{0.6316921719290114,0.6301818997985384,0.626979021888779},{0.6358037284672281,0.6343031662175185,0.6310756037500811},{0.6399262655462556,0.6384354640880802,0.6351831413751425},{0.6399262655462556,0.6384354640880802,0.6351831413751425},{0.6440598857715363,0.6425788945567643,0.6393017355699825},{0.6440598857715363,0.6425788945567643,0.6393017355699825},{0.6482046922201169,0.6467335592421093,0.6434314876092732},{0.6482046922201169,0.6467335592421093,0.6434314876092732},{0.6523607884533357,0.6508995602471196,0.6475724992487806},{0.6523607884533357,0.6508995602471196,0.6475724992487806},{0.6565282785295068,0.6550770001717297,0.6517248727377952},{0.6565282785295068,0.6550770001717297,0.6517248727377952},{0.6607072670166061,0.6592659821252699,0.6558887108315696},{0.6607072670166061,0.6592659821252699,0.6558887108315696},{0.6648978590049535,0.6634666097389375,0.6600641168037573},{0.6648978590049535,0.6634666097389375,0.6600641168037573},{0.6691001601199109,0.6676789871782777,0.6642511944588598},{0.6691001601199109,0.6676789871782777,0.6642511944588598},{0.6733142765345892,0.6719032191556819,0.6684500481446936},{0.6733142765345892,0.6719032191556819,0.6684500481446936},{0.6775403149825813,0.6761394109429005,0.6726607827648713},{0.6775403149825813,0.6761394109429005,0.6726607827648713},{0.6817783827707052,0.6803876683835863,0.6768835037913099},{0.6817783827707052,0.6803876683835863,0.6768835037913099},{0.6860285877917918,0.6846480979058628,0.6811183172767649},{0.6860285877917918,0.6846480979058628,0.6811183172767649},{0.6902910385374814,0.6889208065349255,0.6853653298673966},{0.6902910385374814,0.6889208065349255,0.6853653298673966},{0.6945658441110834,0.6932059019056817,0.6896246488153769},{0.6945658441110834,0.6932059019056817,0.6896246488153769},{0.6988531142404553,0.6975034922754331,0.6938963819915324},{0.6988531142404553,0.6975034922754331,0.6938963819915324},{0.7031529592909377,0.7018136865365993,0.6981806378980351},{0.7031529592909377,0.7018136865365993,0.6981806378980351},{0.7074654902783347,0.7061365942294986,0.7024775256811436},{0.7074654902783347,0.7061365942294986,0.7024775256811436},{0.7117908188819498,0.7104723255551799,0.7067871551439977},{0.7117908188819498,0.7104723255551799,0.7067871551439977},{0.7161290574576848,0.714820991388308,0.7111096367594696},{0.7204803190511825,0.7191827032901243,0.7154450816830777},{0.7204803190511825,0.7191827032901243,0.7154450816830777},{0.7248447174110647,0.7235575735214593,0.7197936017659686},{0.7248447174110647,0.7235575735214593,0.7197936017659686},{0.7292223670022127,0.7279457150558283,0.7241553095679665},{0.7292223670022127,0.7279457150558283,0.7241553095679665},{0.7336133830191388,0.7323472415925961,0.728530318370699},{0.7336133830191388,0.7323472415925961,0.728530318370699},{0.7380178813994349,0.7367622675702209,0.7329187421908016},{0.7380178813994349,0.7367622675702209,0.7329187421908016},{0.7424359788373007,0.7411909081795812,0.7373206957932009},{0.7424359788373007,0.7411909081795812,0.7373206957932009},{0.7468677927971572,0.7456332793773957,0.7417362947044913},{0.7468677927971572,0.7456332793773957,0.7417362947044913},{0.7513134415273599,0.7500894978997243,0.7461656552263957},{0.7513134415273599,0.7500894978997243,0.7461656552263957},{0.7557730440739957,0.7545596812755733,0.7506088944493226},{0.7557730440739957,0.7545596812755733,0.7506088944493226},{0.760246720294789,0.7590439478405959,0.755066130266025},{0.760246720294789,0.7590439478405959,0.755066130266025},{0.7647345908731116,0.7635424167508956,0.7595374813853585},{0.7647345908731116,0.7635424167508956,0.7595374813853585},{0.7692367773320856,0.7680552079969388,0.764023067346146},{0.7692367773320856,0.7680552079969388,0.764023067346146},{0.7737534020488214,0.7725824424175787,0.768523008531156},{0.7737534020488214,0.7725824424175787,0.768523008531156},{0.7782845882687531,0.7771242417141915,0.7730374261811926},{0.7782845882687531,0.7771242417141915,0.7730374261811926},{0.782830460120099,0.7816807284649387,0.7775664424093063},{0.782830460120099,0.7816807284649387,0.7775664424093063},{0.7873911426284513,0.7862520261391467,0.782110180215128},{0.7873911426284513,0.7862520261391467,0.782110180215128},{0.7919667617314894,0.7908382591118159,0.7866687634993276},{0.7919667617314894,0.7908382591118159,0.7866687634993276},{0.7965574442938269,0.7954395526782646,0.7912423170782062},{0.7965574442938269,0.7954395526782646,0.7912423170782062},{0.8011633181219949,0.8000560330689073,0.7958309666984237},{0.8011633181219949,0.8000560330689073,0.7958309666984237},{0.8057845119795648,0.8046878274641713,0.8004348390518634},{0.8104211556024299,0.8093350640095595,0.8050540617906442},{0.8104211556024299,0.8093350640095595,0.8050540617906442},{0.8150733797142105,0.8139978718308641,0.8096887635422754},{0.8150733797142105,0.8139978718308641,0.8096887635422754},{0.8197413160418366,0.8186763810495289,0.8143390739249705},{0.8197413160418366,0.8186763810495289,0.8143390739249705},{0.8244250973312831,0.823370722798173,0.8190051235631107},{0.8244250973312831,0.823370722798173,0.8190051235631107},{0.8291248573634559,0.8280810292362736,0.8236870441028736},{0.8291248573634559,0.8280810292362736,0.8236870441028736},{0.8338407309702662,0.8328074335660154,0.8283849682280267},{0.8338407309702662,0.8328074335660154,0.8283849682280267},{0.8385728540508588,0.8375500700483111,0.8330990296758882},{0.8385728540508588,0.8375500700483111,0.8330990296758882},{0.843321363588025,0.8423090740189962,0.8378293632534638},{0.843321363588025,0.8423090740189962,0.8378293632534638},{0.848086397664791,0.8470845819052075,0.8425761048537627},{0.848086397664791,0.8470845819052075,0.8425761048537627},{0.8528680954812019,0.8518767312419356,0.8473393914722943},{0.8528680954812019,0.8518767312419356,0.8473393914722943},{0.8576665973712845,0.8566856606887784,0.8521193612237554},{0.8576665973712845,0.8566856606887784,0.8521193612237554},{0.8624820448202135,0.8615115100468785,0.8569161533589083},{0.8624820448202135,0.8615115100468785,0.8569161533589083},{0.8673145804816755,0.8663544202760625,0.8617299082816555},{0.8673145804816755,0.8663544202760625,0.8617299082816555},{0.8721643481954295,0.8712145335121869,0.8665607675663184},{0.8721643481954295,0.8712145335121869,0.8665607675663184},{0.8770314930050886,0.8760919930846851,0.871408873975121},{0.8770314930050886,0.8760919930846851,0.871408873975121},{0.8819161611761233,0.8809869435343266,0.8762743714758872},{0.8819161611761233,0.8809869435343266,0.8762743714758872},{0.8868185002140476,0.8858995306312075,0.8811574052599523},{0.8868185002140476,0.8858995306312075,0.8811574052599523},{0.8917386588828792,0.8908299013929443,0.8860581217602967},{0.8917386588828792,0.8908299013929443,0.8860581217602967},{0.8966767872237935,0.8957782041031092,0.8909766686699112},{0.8966767872237935,0.8957782041031092,0.8909766686699112},{0.9016330365740295,0.9007445883298949,0.8959131949603859},{0.9066075595860337,0.9057292049450164,0.9008678509007446},{0.9066075595860337,0.9057292049450164,0.9008678509007446},{0.9116005102468445,0.9107322061428585,0.9058407880765155},{0.9116005102468445,0.9107322061428585,0.9058407880765155},{0.9166120438977335,0.915753745459873,0.9108321594090564},{0.9166120438977335,0.915753745459873,0.9108321594090564},{0.921642317254111,0.9207939777942258,0.9158421191751267},{0.921642317254111,0.9207939777942258,0.9158421191751267},{0.9266914884256761,0.9258530594257129,0.9208708230267246},{0.9266914884256761,0.9258530594257129,0.9208708230267246},{0.9317597169368546,0.9309311480359354,0.9259184280111854},{0.9317597169368546,0.9309311480359354,0.9259184280111854},{0.9368471637474911,0.9360284027287524,0.9309850925915528},{0.9368471637474911,0.9360284027287524,0.9309850925915528},{0.9419539912738586,0.9411449840510026,0.9360709766672254},{0.9419539912738586,0.9411449840510026,0.9360709766672254},{0.9470803634098973,0.9462810540135232,0.9411762415948877},{0.9470803634098973,0.9462810540135232,0.9411762415948877},{0.9522264455488126,0.9514367761124457,0.9463010502097274},{0.9522264455488126,0.9514367761124457,0.9463010502097274},{0.957392404604907,0.9566123153508007,0.9514455668469503},{0.957392404604907,0.9566123153508007,0.9514455668469503},{0.9625784090357906,0.9618078382604079,0.9566099573635957},{0.9625784090357906,0.9618078382604079,0.9566099573635957},{0.9677846288648154,0.9670235129241013,0.9617943891606574},{0.9677846288648154,0.9670235129241013,0.9617943891606574},{0.973011235703915,0.972259508998241,0.9669990312055218},{0.973011235703915,0.972259508998241,0.9669990312055218},{0.978258402776717,0.9775159977355695,0.972224054054729},{0.978258402776717,0.9775159977355695,0.972224054054729},{0.983526304942001,0.982793152008386,0.9774696298770532},{0.983526304942001,0.982793152008386,0.9774696298770532},{0.9888151187175039,0.9880911463320615,0.9827359324769277},{0.9888151187175039,0.9880911463320615,0.9827359324769277},{0.9941250223040635,0.9934101568888957,0.9880231373182063},{0.9941250223040635,0.9934101568888957,0.9880231373182063},{0.9994561956101177,0.9987503615523224,0.9933314215482737},{0.9994561956101177,0.9987503615523224,0.9933314215482737}};

const std::vector<glm::vec3> CM_VIRIDIS =
{{0.267004,0.004874,0.329415},{0.267004,0.004874,0.329415},{0.26851,0.009605,0.335427},{0.26851,0.009605,0.335427},{0.269944,0.014625,0.341379},{0.269944,0.014625,0.341379},{0.271305,0.019942,0.347269},{0.271305,0.019942,0.347269},{0.272594,0.025563,0.353093},{0.272594,0.025563,0.353093},{0.273809,0.031497,0.358853},{0.273809,0.031497,0.358853},{0.274952,0.037752,0.364543},{0.274952,0.037752,0.364543},{0.276022,0.044167,0.370164},{0.276022,0.044167,0.370164},{0.277018,0.050344,0.375715},{0.277018,0.050344,0.375715},{0.277941,0.056324,0.381191},{0.277941,0.056324,0.381191},{0.278791,0.062145,0.386592},{0.278791,0.062145,0.386592},{0.279566,0.067836,0.391917},{0.279566,0.067836,0.391917},{0.280267,0.073417,0.397163},{0.280267,0.073417,0.397163},{0.280894,0.078907,0.402329},{0.280894,0.078907,0.402329},{0.281446,0.08432,0.407414},{0.281446,0.08432,0.407414},{0.281924,0.089666,0.412415},{0.281924,0.089666,0.412415},{0.282327,0.094955,0.417331},{0.282327,0.094955,0.417331},{0.282656,0.100196,0.42216},{0.282656,0.100196,0.42216},{0.28291,0.105393,0.426902},{0.28291,0.105393,0.426902},{0.283091,0.110553,0.431554},{0.283197,0.11568,0.436115},{0.283197,0.11568,0.436115},{0.283229,0.120777,0.440584},{0.283229,0.120777,0.440584},{0.283187,0.125848,0.44496},{0.283187,0.125848,0.44496},{0.283072,0.130895,0.449241},{0.283072,0.130895,0.449241},{0.282884,0.13592,0.453427},{0.282884,0.13592,0.453427},{0.282623,0.140926,0.457517},{0.282623,0.140926,0.457517},{0.28229,0.145912,0.46151},{0.28229,0.145912,0.46151},{0.281887,0.150881,0.465405},{0.281887,0.150881,0.465405},{0.281412,0.155834,0.469201},{0.281412,0.155834,0.469201},{0.280868,0.160771,0.472899},{0.280868,0.160771,0.472899},{0.280255,0.165693,0.476498},{0.280255,0.165693,0.476498},{0.279574,0.170599,0.479997},{0.279574,0.170599,0.479997},{0.278826,0.17549,0.483397},{0.278826,0.17549,0.483397},{0.278012,0.180367,0.486697},{0.278012,0.180367,0.486697},{0.277134,0.185228,0.489898},{0.277134,0.185228,0.489898},{0.276194,0.190074,0.493001},{0.276194,0.190074,0.493001},{0.275191,0.194905,0.496005},{0.275191,0.194905,0.496005},{0.274128,0.199721,0.498911},{0.274128,0.199721,0.498911},{0.273006,0.20452,0.501721},{0.273006,0.20452,0.501721},{0.271828,0.209303,0.504434},{0.270595,0.214069,0.507052},{0.270595,0.214069,0.507052},{0.269308,0.218818,0.509577},{0.269308,0.218818,0.509577},{0.267968,0.223549,0.512008},{0.267968,0.223549,0.512008},{0.26658,0.228262,0.514349},{0.26658,0.228262,0.514349},{0.265145,0.232956,0.516599},{0.265145,0.232956,0.516599},{0.263663,0.237631,0.518762},{0.263663,0.237631,0.518762},{0.262138,0.242286,0.520837},{0.262138,0.242286,0.520837},{0.260571,0.246922,0.522828},{0.260571,0.246922,0.522828},{0.258965,0.251537,0.524736},{0.258965,0.251537,0.524736},{0.257322,0.25613,0.526563},{0.257322,0.25613,0.526563},{0.255645,0.260703,0.528312},{0.255645,0.260703,0.528312},{0.253935,0.265254,0.529983},{0.253935,0.265254,0.529983},{0.252194,0.269783,0.531579},{0.252194,0.269783,0.531579},{0.250425,0.27429,0.533103},{0.250425,0.27429,0.533103},{0.248629,0.278775,0.534556},{0.248629,0.278775,0.534556},{0.246811,0.283237,0.535941},{0.246811,0.283237,0.535941},{0.244972,0.287675,0.53726},{0.244972,0.287675,0.53726},{0.243113,0.292092,0.538516},{0.243113,0.292092,0.538516},{0.241237,0.296485,0.539709},{0.241237,0.296485,0.539709},{0.239346,0.300855,0.540844},{0.237441,0.305202,0.541921},{0.237441,0.305202,0.541921},{0.235526,0.309527,0.542944},{0.235526,0.309527,0.542944},{0.233603,0.313828,0.543914},{0.233603,0.313828,0.543914},{0.231674,0.318106,0.544834},{0.231674,0.318106,0.544834},{0.229739,0.322361,0.545706},{0.229739,0.322361,0.545706},{0.227802,0.326594,0.546532},{0.227802,0.326594,0.546532},{0.225863,0.330805,0.547314},{0.225863,0.330805,0.547314},{0.223925,0.334994,0.548053},{0.223925,0.334994,0.548053},{0.221989,0.339161,0.548752},{0.221989,0.339161,0.548752},{0.220057,0.343307,0.549413},{0.220057,0.343307,0.549413},{0.21813,0.347432,0.550038},{0.21813,0.347432,0.550038},{0.21621,0.351535,0.550627},{0.21621,0.351535,0.550627},{0.214298,0.355619,0.551184},{0.214298,0.355619,0.551184},{0.212395,0.359683,0.55171},{0.212395,0.359683,0.55171},{0.210503,0.363727,0.552206},{0.210503,0.363727,0.552206},{0.208623,0.367752,0.552675},{0.208623,0.367752,0.552675},{0.206756,0.371758,0.553117},{0.206756,0.371758,0.553117},{0.204903,0.375746,0.553533},{0.204903,0.375746,0.553533},{0.203063,0.379716,0.553925},{0.201239,0.38367,0.554294},{0.201239,0.38367,0.554294},{0.19943,0.387607,0.554642},{0.19943,0.387607,0.554642},{0.197636,0.391528,0.554969},{0.197636,0.391528,0.554969},{0.19586,0.395433,0.555276},{0.19586,0.395433,0.555276},{0.1941,0.399323,0.555565},{0.1941,0.399323,0.555565},{0.192357,0.403199,0.555836},{0.192357,0.403199,0.555836},{0.190631,0.407061,0.556089},{0.190631,0.407061,0.556089},{0.188923,0.41091,0.556326},{0.188923,0.41091,0.556326},{0.187231,0.414746,0.556547},{0.187231,0.414746,0.556547},{0.185556,0.41857,0.556753},{0.185556,0.41857,0.556753},{0.183898,0.422383,0.556944},{0.183898,0.422383,0.556944},{0.182256,0.426184,0.55712},{0.182256,0.426184,0.55712},{0.180629,0.429975,0.557282},{0.180629,0.429975,0.557282},{0.179019,0.433756,0.55743},{0.179019,0.433756,0.55743},{0.177423,0.437527,0.557565},{0.177423,0.437527,0.557565},{0.175841,0.44129,0.557685},{0.175841,0.44129,0.557685},{0.174274,0.445044,0.557792},{0.174274,0.445044,0.557792},{0.172719,0.448791,0.557885},{0.172719,0.448791,0.557885},{0.171176,0.45253,0.557965},{0.171176,0.45253,0.557965},{0.169646,0.456262,0.55803},{0.168126,0.459988,0.558082},{0.168126,0.459988,0.558082},{0.166617,0.463708,0.558119},{0.166617,0.463708,0.558119},{0.165117,0.467423,0.558141},{0.165117,0.467423,0.558141},{0.163625,0.471133,0.558148},{0.163625,0.471133,0.558148},{0.162142,0.474838,0.55814},{0.162142,0.474838,0.55814},{0.160665,0.47854,0.558115},{0.160665,0.47854,0.558115},{0.159194,0.482237,0.558073},{0.159194,0.482237,0.558073},{0.157729,0.485932,0.558013},{0.157729,0.485932,0.558013},{0.15627,0.489624,0.557936},{0.15627,0.489624,0.557936},{0.154815,0.493313,0.55784},{0.154815,0.493313,0.55784},{0.153364,0.497,0.557724},{0.153364,0.497,0.557724},{0.151918,0.500685,0.557587},{0.151918,0.500685,0.557587},{0.150476,0.504369,0.55743},{0.150476,0.504369,0.55743},{0.149039,0.508051,0.55725},{0.149039,0.508051,0.55725},{0.147607,0.511733,0.557049},{0.147607,0.511733,0.557049},{0.14618,0.515413,0.556823},{0.14618,0.515413,0.556823},{0.144759,0.519093,0.556572},{0.144759,0.519093,0.556572},{0.143343,0.522773,0.556295},{0.143343,0.522773,0.556295},{0.141935,0.526453,0.555991},{0.141935,0.526453,0.555991},{0.140536,0.530132,0.555659},{0.139147,0.533812,0.555298},{0.139147,0.533812,0.555298},{0.13777,0.537492,0.554906},{0.13777,0.537492,0.554906},{0.136408,0.541173,0.554483},{0.136408,0.541173,0.554483},{0.135066,0.544853,0.554029},{0.135066,0.544853,0.554029},{0.133743,0.548535,0.553541},{0.133743,0.548535,0.553541},{0.132444,0.552216,0.553018},{0.132444,0.552216,0.553018},{0.131172,0.555899,0.552459},{0.131172,0.555899,0.552459},{0.129933,0.559582,0.551864},{0.129933,0.559582,0.551864},{0.128729,0.563265,0.551229},{0.128729,0.563265,0.551229},{0.127568,0.566949,0.550556},{0.127568,0.566949,0.550556},{0.126453,0.570633,0.549841},{0.126453,0.570633,0.549841},{0.125394,0.574318,0.549086},{0.125394,0.574318,0.549086},{0.124395,0.578002,0.548287},{0.124395,0.578002,0.548287},{0.123463,0.581687,0.547445},{0.123463,0.581687,0.547445},{0.122606,0.585371,0.546557},{0.122606,0.585371,0.546557},{0.121831,0.589055,0.545623},{0.121831,0.589055,0.545623},{0.121148,0.592739,0.544641},{0.121148,0.592739,0.544641},{0.120565,0.596422,0.543611},{0.120565,0.596422,0.543611},{0.120092,0.600104,0.54253},{0.119738,0.603785,0.5414},{0.119738,0.603785,0.5414},{0.119512,0.607464,0.540218},{0.119512,0.607464,0.540218},{0.119423,0.611141,0.538982},{0.119423,0.611141,0.538982},{0.119483,0.614817,0.537692},{0.119483,0.614817,0.537692},{0.119699,0.61849,0.536347},{0.119699,0.61849,0.536347},{0.120081,0.622161,0.534946},{0.120081,0.622161,0.534946},{0.120638,0.625828,0.533488},{0.120638,0.625828,0.533488},{0.12138,0.629492,0.531973},{0.12138,0.629492,0.531973},{0.122312,0.633153,0.530398},{0.122312,0.633153,0.530398},{0.123444,0.636809,0.528763},{0.123444,0.636809,0.528763},{0.12478,0.640461,0.527068},{0.12478,0.640461,0.527068},{0.126326,0.644107,0.525311},{0.126326,0.644107,0.525311},{0.128087,0.647749,0.523491},{0.128087,0.647749,0.523491},{0.130067,0.651384,0.521608},{0.130067,0.651384,0.521608},{0.132268,0.655014,0.519661},{0.132268,0.655014,0.519661},{0.134692,0.658636,0.517649},{0.134692,0.658636,0.517649},{0.137339,0.662252,0.515571},{0.137339,0.662252,0.515571},{0.14021,0.665859,0.513427},{0.14021,0.665859,0.513427},{0.143303,0.669459,0.511215},{0.143303,0.669459,0.511215},{0.146616,0.67305,0.508936},{0.150148,0.676631,0.506589},{0.150148,0.676631,0.506589},{0.153894,0.680203,0.504172},{0.153894,0.680203,0.504172},{0.157851,0.683765,0.501686},{0.157851,0.683765,0.501686},{0.162016,0.687316,0.499129},{0.162016,0.687316,0.499129},{0.166383,0.690856,0.496502},{0.166383,0.690856,0.496502},{0.170948,0.694384,0.493803},{0.170948,0.694384,0.493803},{0.175707,0.6979,0.491033},{0.175707,0.6979,0.491033},{0.180653,0.701402,0.488189},{0.180653,0.701402,0.488189},{0.185783,0.704891,0.485273},{0.185783,0.704891,0.485273},{0.19109,0.708366,0.482284},{0.19109,0.708366,0.482284},{0.196571,0.711827,0.479221},{0.196571,0.711827,0.479221},{0.202219,0.715272,0.476084},{0.202219,0.715272,0.476084},{0.20803,0.718701,0.472873},{0.20803,0.718701,0.472873},{0.214,0.722114,0.469588},{0.214,0.722114,0.469588},{0.220124,0.725509,0.466226},{0.220124,0.725509,0.466226},{0.226397,0.728888,0.462789},{0.226397,0.728888,0.462789},{0.232815,0.732247,0.459277},{0.232815,0.732247,0.459277},{0.239374,0.735588,0.455688},{0.239374,0.735588,0.455688},{0.24607,0.73891,0.452024},{0.24607,0.73891,0.452024},{0.252899,0.742211,0.448284},{0.259857,0.745492,0.444467},{0.259857,0.745492,0.444467},{0.266941,0.748751,0.440573},{0.266941,0.748751,0.440573},{0.274149,0.751988,0.436601},{0.274149,0.751988,0.436601},{0.281477,0.755203,0.432552},{0.281477,0.755203,0.432552},{0.288921,0.758394,0.428426},{0.288921,0.758394,0.428426},{0.296479,0.761561,0.424223},{0.296479,0.761561,0.424223},{0.304148,0.764704,0.419943},{0.304148,0.764704,0.419943},{0.311925,0.767822,0.415586},{0.311925,0.767822,0.415586},{0.319809,0.770914,0.411152},{0.319809,0.770914,0.411152},{0.327796,0.77398,0.40664},{0.327796,0.77398,0.40664},{0.335885,0.777018,0.402049},{0.335885,0.777018,0.402049},{0.344074,0.780029,0.397381},{0.344074,0.780029,0.397381},{0.35236,0.783011,0.392636},{0.35236,0.783011,0.392636},{0.360741,0.785964,0.387814},{0.360741,0.785964,0.387814},{0.369214,0.788888,0.382914},{0.369214,0.788888,0.382914},{0.377779,0.791781,0.377939},{0.377779,0.791781,0.377939},{0.386433,0.794644,0.372886},{0.386433,0.794644,0.372886},{0.395174,0.797475,0.367757},{0.395174,0.797475,0.367757},{0.404001,0.800275,0.362552},{0.412913,0.803041,0.357269},{0.412913,0.803041,0.357269},{0.421908,0.805774,0.35191},{0.421908,0.805774,0.35191},{0.430983,0.808473,0.346476},{0.430983,0.808473,0.346476},{0.440137,0.811138,0.340967},{0.440137,0.811138,0.340967},{0.449368,0.813768,0.335384},{0.449368,0.813768,0.335384},{0.458674,0.816363,0.329727},{0.458674,0.816363,0.329727},{0.468053,0.818921,0.323998},{0.468053,0.818921,0.323998},{0.477504,0.821444,0.318195},{0.477504,0.821444,0.318195},{0.487026,0.823929,0.312321},{0.487026,0.823929,0.312321},{0.496615,0.826376,0.306377},{0.496615,0.826376,0.306377},{0.506271,0.828786,0.300362},{0.506271,0.828786,0.300362},{0.515992,0.831158,0.294279},{0.515992,0.831158,0.294279},{0.525776,0.833491,0.288127},{0.525776,0.833491,0.288127},{0.535621,0.835785,0.281908},{0.535621,0.835785,0.281908},{0.545524,0.838039,0.275626},{0.545524,0.838039,0.275626},{0.555484,0.840254,0.269281},{0.555484,0.840254,0.269281},{0.565498,0.84243,0.262877},{0.565498,0.84243,0.262877},{0.575563,0.844566,0.256415},{0.575563,0.844566,0.256415},{0.585678,0.846661,0.249897},{0.585678,0.846661,0.249897},{0.595839,0.848717,0.243329},{0.606045,0.850733,0.236712},{0.606045,0.850733,0.236712},{0.616293,0.852709,0.230052},{0.616293,0.852709,0.230052},{0.626579,0.854645,0.223353},{0.626579,0.854645,0.223353},{0.636902,0.856542,0.21662},{0.636902,0.856542,0.21662},{0.647257,0.8584,0.209861},{0.647257,0.8584,0.209861},{0.657642,0.860219,0.203082},{0.657642,0.860219,0.203082},{0.668054,0.861999,0.196293},{0.668054,0.861999,0.196293},{0.678489,0.863742,0.189503},{0.678489,0.863742,0.189503},{0.688944,0.865448,0.182725},{0.688944,0.865448,0.182725},{0.699415,0.867117,0.175971},{0.699415,0.867117,0.175971},{0.709898,0.868751,0.169257},{0.709898,0.868751,0.169257},{0.720391,0.87035,0.162603},{0.720391,0.87035,0.162603},{0.730889,0.871916,0.156029},{0.730889,0.871916,0.156029},{0.741388,0.873449,0.149561},{0.741388,0.873449,0.149561},{0.751884,0.874951,0.143228},{0.751884,0.874951,0.143228},{0.762373,0.876424,0.137064},{0.762373,0.876424,0.137064},{0.772852,0.877868,0.131109},{0.772852,0.877868,0.131109},{0.783315,0.879285,0.125405},{0.783315,0.879285,0.125405},{0.79376,0.880678,0.120005},{0.79376,0.880678,0.120005},{0.804182,0.882046,0.114965},{0.814576,0.883393,0.110347},{0.814576,0.883393,0.110347},{0.82494,0.88472,0.106217},{0.82494,0.88472,0.106217},{0.83527,0.886029,0.102646},{0.83527,0.886029,0.102646},{0.845561,0.887322,0.099702},{0.845561,0.887322,0.099702},{0.85581,0.888601,0.097452},{0.85581,0.888601,0.097452},{0.866013,0.889868,0.095953},{0.866013,0.889868,0.095953},{0.876168,0.891125,0.09525},{0.876168,0.891125,0.09525},{0.886271,0.892374,0.095374},{0.886271,0.892374,0.095374},{0.89632,0.893616,0.096335},{0.89632,0.893616,0.096335},{0.906311,0.894855,0.098125},{0.906311,0.894855,0.098125},{0.916242,0.896091,0.100717},{0.916242,0.896091,0.100717},{0.926106,0.89733,0.104071},{0.926106,0.89733,0.104071},{0.935904,0.89857,0.108131},{0.935904,0.89857,0.108131},{0.945636,0.899815,0.112838},{0.945636,0.899815,0.112838},{0.9553,0.901065,0.118128},{0.9553,0.901065,0.118128},{0.964894,0.902323,0.123941},{0.964894,0.902323,0.123941},{0.974417,0.90359,0.130215},{0.974417,0.90359,0.130215},{0.983868,0.904867,0.136897},{0.983868,0.904867,0.136897},{0.993248,0.906157,0.143936},{0.993248,0.906157,0.143936},};

const std::vector<glm::vec3> CM_MAGMA =
{{0.001462, 0.000466, 0.013866}, {0.00186877, 0.000889637, 0.0161477}, {0.00228051, 0.00131726, 0.0184495}, {0.00280226, 0.0018334, 0.0211973}, {0.00333336, 0.00235724, 0.0239839}, {0.00396345, 0.00296281, 0.0271813}, {0.0046071, 0.00357948, 0.0304388}, {0.00534195, 0.00427089, 0.0341003}, {0.00609443, 0.00497641, 0.0378216}, {0.00693149, 0.00574959, 0.0418295}, {0.00779059, 0.00653963, 0.0458405}, {0.00872984, 0.00739099, 0.0498628}, {0.00969569, 0.00826087, 0.0538897}, {0.0107377, 0.00918378, 0.0579298}, {0.0118111, 0.0101278, 0.0619717}, {0.0129573, 0.0111207, 0.0660174}, {0.0141397, 0.0121359, 0.0700665}, {0.0153907, 0.0131932, 0.074122}, {0.0166835, 0.0142737, 0.0781864}, {0.0180423, 0.0153908, 0.0822649}, {0.0194492, 0.0165317, 0.0863533}, {0.0209194, 0.017704, 0.0904547}, {0.0224437, 0.0189008, 0.0945659}, {0.0240279, 0.0201246, 0.0986878}, {0.0256731, 0.0213726, 0.102822}, {0.0273754, 0.022643, 0.106966}, {0.0291469, 0.0239358, 0.111131}, {0.0309728, 0.025246, 0.11531}, {0.0328762, 0.0265773, 0.119509}, {0.0348303, 0.0279223, 0.123722}, {0.0368717, 0.0292875, 0.127955}, {0.0389608, 0.0306637, 0.1322}, {0.0410971, 0.0320564, 0.136467}, {0.0432547, 0.0334566, 0.140745}, {0.0454159, 0.0348707, 0.145049}, {0.0475786, 0.0362898, 0.149363}, {0.0497515, 0.037718, 0.153705}, {0.0519275, 0.0391489, 0.158056}, {0.0541189, 0.0405601, 0.162445}, {0.0563137, 0.0419669, 0.166842}, {0.0585258, 0.0433213, 0.171274}, {0.0607406, 0.0446673, 0.175711}, {0.0629771, 0.0459624, 0.180186}, {0.0652159, 0.0472522, 0.184664}, {0.0674804, 0.0484858, 0.189181}, {0.0697462, 0.0497163, 0.1937}, {0.072042, 0.0508876, 0.19826}, {0.0743392, 0.0520561, 0.202822}, {0.0766684, 0.0531635, 0.207425}, {0.0790006, 0.0542658, 0.212031}, {0.0813672, 0.0553088, 0.216675}, {0.0837385, 0.0563435, 0.221324}, {0.0861449, 0.0573186, 0.226009}, {0.0885585, 0.0582821, 0.2307}, {0.0910083, 0.0591871, 0.235423}, {0.0934677, 0.0600768, 0.240153}, {0.0959636, 0.0609082, 0.244912}, {0.098472, 0.0617207, 0.24968}, {0.101018, 0.0624765, 0.254472}, {0.103579, 0.0632091, 0.259272}, {0.106177, 0.0638863, 0.264094}, {0.108794, 0.0645344, 0.268925}, {0.111449, 0.0651257, 0.273778}, {0.114126, 0.0656839, 0.278638}, {0.11684, 0.0661882, 0.283512}, {0.11958, 0.0666556, 0.28839}, {0.122354, 0.0670726, 0.293274}, {0.125156, 0.0674476, 0.29816}, {0.12799, 0.0677747, 0.303047}, {0.130858, 0.0680538, 0.307933}, {0.133757, 0.0682868, 0.312817}, {0.136693, 0.0684653, 0.317695}, {0.13966, 0.0685997, 0.322569}, {0.142663, 0.0686796, 0.327424}, {0.145692, 0.0687225, 0.332264}, {0.148763, 0.068705, 0.337079}, {0.151856, 0.0686534, 0.341879}, {0.154994, 0.0685383, 0.346641}, {0.158151, 0.0683937, 0.351386}, {0.16135, 0.0681898, 0.356072}, {0.164564, 0.0679634, 0.360737}, {0.167824, 0.067673, 0.365333}, {0.171097, 0.0673633, 0.36991}, {0.174409, 0.0670026, 0.374391}, {0.17773, 0.0666301, 0.37885}, {0.181091, 0.0662073, 0.3832}, {0.184458, 0.065776, 0.387531}, {0.187857, 0.0653125, 0.391726}, {0.19126, 0.0648455, 0.395906}, {0.194691, 0.0643582, 0.399931}, {0.198123, 0.0638697, 0.403946}, {0.201576, 0.0633816, 0.407784}, {0.20503, 0.0628942, 0.411614}, {0.208496, 0.0624266, 0.415259}, {0.211963, 0.0619619, 0.41889}, {0.215435, 0.0615357, 0.422337}, {0.218907, 0.0611166, 0.425761}, {0.222376, 0.0607522, 0.429006}, {0.225845, 0.0604004, 0.432219}, {0.229307, 0.0601163, 0.435261}, {0.232767, 0.059851, 0.438263}, {0.236216, 0.0596609, 0.441105}, {0.239661, 0.0594965, 0.4439}, {0.243093, 0.0594122, 0.446547}, {0.24652, 0.0593612, 0.44914}, {0.249932, 0.0593934, 0.451599}, {0.253337, 0.059464, 0.453999}, {0.256729, 0.0596127, 0.456279}, {0.260112, 0.0598071, 0.458496}, {0.263479, 0.0600784, 0.460607}, {0.266838, 0.0603978, 0.462651}, {0.270183, 0.0607846, 0.464604}, {0.273518, 0.0612247, 0.466488}, {0.276839, 0.0617276, 0.468292}, {0.28015, 0.0622833, 0.470026}, {0.28345, 0.0628914, 0.471693}, {0.286738, 0.0635538, 0.47329}, {0.290016, 0.0642616, 0.474828}, {0.293282, 0.0650231, 0.476298}, {0.296539, 0.0658224, 0.477719}, {0.299786, 0.0666713, 0.479072}, {0.303026, 0.0675492, 0.480385}, {0.306253, 0.0684785, 0.48163}, {0.309473, 0.0694326, 0.482843}, {0.312682, 0.0704311, 0.483991}, {0.315887, 0.071447, 0.485112}, {0.319082, 0.0725033, 0.486168}, {0.322273, 0.0735724, 0.487205}, {0.325452, 0.0746817, 0.48818}, {0.328628, 0.0758008, 0.48914}, {0.331795, 0.076952, 0.490039}, {0.33496, 0.078109, 0.490927}, {0.338117, 0.079295, 0.491755}, {0.341272, 0.0804847, 0.492576}, {0.344417, 0.0817001, 0.493342}, {0.347562, 0.0829174, 0.494103}, {0.350698, 0.0841571, 0.49481}, {0.353834, 0.0853976, 0.495514}, {0.356964, 0.0866537, 0.496166}, {0.360094, 0.0879106, 0.496816}, {0.363218, 0.0891795, 0.49742}, {0.366342, 0.0904494, 0.498019}, {0.369462, 0.091728, 0.498578}, {0.37258, 0.0930076, 0.49913}, {0.375695, 0.0942933, 0.499648}, {0.378809, 0.0955797, 0.500159}, {0.38192, 0.0968691, 0.500637}, {0.385031, 0.0981585, 0.501106}, {0.38814, 0.0994483, 0.501546}, {0.39125, 0.100738, 0.501977}, {0.394358, 0.102027, 0.502383}, {0.397466, 0.103316, 0.502778}, {0.400574, 0.104603, 0.50315}, {0.403681, 0.105888, 0.50351}, {0.406789, 0.10717, 0.50385}, {0.409896, 0.108451, 0.504179}, {0.413003, 0.109729, 0.504491}, {0.416111, 0.111005, 0.504789}, {0.419219, 0.112277, 0.505072}, {0.422328, 0.113545, 0.505341}, {0.425438, 0.11481, 0.505596}, {0.428549, 0.11607, 0.505836}, {0.431661, 0.117328, 0.506064}, {0.434775, 0.11858, 0.506277}, {0.437889, 0.119828, 0.506479}, {0.441006, 0.121071, 0.506665}, {0.444124, 0.122311, 0.506842}, {0.447244, 0.123544, 0.507002}, {0.450365, 0.124775, 0.507154}, {0.453489, 0.125999, 0.507289}, {0.456614, 0.12722, 0.507416}, {0.459742, 0.128434, 0.507526}, {0.46287, 0.129646, 0.507631}, {0.466003, 0.13085, 0.507716}, {0.469136, 0.132052, 0.507796}, {0.472273, 0.133245, 0.507857}, {0.475411, 0.134437, 0.507914}, {0.478553, 0.13562, 0.507952}, {0.481695, 0.136803, 0.507986}, {0.484842, 0.137976, 0.507999}, {0.487989, 0.139149, 0.508011}, {0.491142, 0.140313, 0.508}, {0.494295, 0.141476, 0.507988}, {0.497453, 0.142629, 0.507953}, {0.500612, 0.143782, 0.507917}, {0.503775, 0.144926, 0.507859}, {0.50694, 0.146069, 0.507798}, {0.510109, 0.147204, 0.507717}, {0.513279, 0.148338, 0.507633}, {0.516455, 0.149464, 0.507528}, {0.519631, 0.150589, 0.507419}, {0.522812, 0.151706, 0.507291}, {0.525995, 0.152821, 0.507157}, {0.529182, 0.15393, 0.507006}, {0.532371, 0.155037, 0.506847}, {0.535564, 0.156138, 0.506672}, {0.538759, 0.157237, 0.506488}, {0.541958, 0.15833, 0.506288}, {0.545159, 0.159421, 0.506079}, {0.548364, 0.160506, 0.505854}, {0.551572, 0.16159, 0.505619}, {0.554783, 0.162668, 0.505369}, {0.557997, 0.163744, 0.505108}, {0.561213, 0.164817, 0.504833}, {0.564433, 0.165886, 0.504546}, {0.567656, 0.166952, 0.504246}, {0.570881, 0.168016, 0.503932}, {0.57411, 0.169077, 0.503606}, {0.577342, 0.170134, 0.503264}, {0.580576, 0.17119, 0.502912}, {0.583813, 0.172243, 0.502544}, {0.587053, 0.173294, 0.502164}, {0.590296, 0.174342, 0.501768}, {0.593541, 0.175389, 0.501362}, {0.596789, 0.176434, 0.500937}, {0.60004, 0.177477, 0.500504}, {0.603293, 0.178518, 0.500051}, {0.606549, 0.179559, 0.49959}, {0.609807, 0.180598, 0.499107}, {0.613067, 0.181636, 0.498618}, {0.616331, 0.182673, 0.498106}, {0.619595, 0.18371, 0.497589}, {0.622863, 0.184746, 0.497047}, {0.626132, 0.185782, 0.496501}, {0.629404, 0.186817, 0.495929}, {0.632677, 0.187852, 0.495355}, {0.635953, 0.188889, 0.494752}, {0.639229, 0.189925, 0.494148}, {0.642508, 0.190963, 0.493514}, {0.645787, 0.192001, 0.492879}, {0.64907, 0.19304, 0.492215}, {0.652352, 0.19408, 0.491548}, {0.655637, 0.195123, 0.490854}, {0.658921, 0.196167, 0.490156}, {0.662208, 0.197213, 0.489432}, {0.665495, 0.198261, 0.488703}, {0.668783, 0.199312, 0.487947}, {0.672071, 0.200365, 0.487185}, {0.675361, 0.201423, 0.486399}, {0.67865, 0.202483, 0.485604}, {0.68194, 0.203547, 0.484787}, {0.68523, 0.204614, 0.483959}, {0.68852, 0.205686, 0.483111}, {0.691809, 0.206762, 0.482251}, {0.695099, 0.207843, 0.48137}, {0.698387, 0.208929, 0.480477}, {0.701675, 0.210021, 0.479564}, {0.704962, 0.211118, 0.478638}, {0.708248, 0.212222, 0.477694}, {0.711533, 0.213331, 0.476734}, {0.714816, 0.214448, 0.475757}, {0.718098, 0.215572, 0.474764}, {0.721377, 0.216702, 0.473755}, {0.724655, 0.217841, 0.472728}, {0.727931, 0.218988, 0.471687}, {0.731204, 0.220144, 0.470627}, {0.734475, 0.221307, 0.469554}, {0.737741, 0.222482, 0.468461}, {0.741005, 0.223664, 0.467356}, {0.744265, 0.224859, 0.466229}, {0.747522, 0.226061, 0.465093}, {0.750774, 0.227279, 0.463933}, {0.754023, 0.228503, 0.462765}, {0.757265, 0.229746, 0.461573}, {0.760505, 0.230994, 0.460374}, {0.763737, 0.232262, 0.459149}, {0.766967, 0.233535, 0.457919}, {0.770187, 0.234832, 0.456663}, {0.773405, 0.236132, 0.455403}, {0.776612, 0.237459, 0.454116}, {0.779817, 0.238788, 0.452826}, {0.783009, 0.240148, 0.451508}, {0.786199, 0.241509, 0.450189}, {0.789376, 0.242903, 0.44884}, {0.792551, 0.244298, 0.447489}, {0.795709, 0.245728, 0.446112}, {0.798867, 0.247161, 0.444732}, {0.802006, 0.248628, 0.443329}, {0.805143, 0.2501, 0.441923}, {0.808262, 0.251608, 0.440493}, {0.811377, 0.253122, 0.43906}, {0.814473, 0.254673, 0.437607}, {0.817565, 0.256233, 0.436148}, {0.820637, 0.257828, 0.434673}, {0.823702, 0.259435, 0.433191}, {0.826748, 0.261076, 0.431695}, {0.829785, 0.262733, 0.430191}, {0.832803, 0.264425, 0.428672}, {0.83581, 0.266135, 0.427147}, {0.838797, 0.267879, 0.425611}, {0.841771, 0.269644, 0.42407}, {0.844725, 0.271443, 0.422519}, {0.847663, 0.273267, 0.420963}, {0.850581, 0.275123, 0.4194}, {0.853482, 0.277008, 0.417833}, {0.856362, 0.278925, 0.416261}, {0.859222, 0.280873, 0.414684}, {0.862062, 0.282852, 0.413103}, {0.864877, 0.284867, 0.411521}, {0.867674, 0.286911, 0.409937}, {0.870443, 0.288995, 0.408353}, {0.873194, 0.291106, 0.40677}, {0.875914, 0.293261, 0.40519}, {0.878616, 0.295441, 0.403613}, {0.881283, 0.29767, 0.402041}, {0.883934, 0.299921, 0.400472}, {0.886545, 0.302226, 0.398912}, {0.889141, 0.30455, 0.397356}, {0.891693, 0.306932, 0.395815}, {0.894232, 0.309331, 0.394278}, {0.896722, 0.311792, 0.392762}, {0.899202, 0.314267, 0.39125}, {0.901627, 0.316808, 0.389764}, {0.904044, 0.31936, 0.388282}, {0.906403, 0.321981, 0.386833}, {0.908755, 0.324611, 0.385387}, {0.911043, 0.327313, 0.383982}, {0.913327, 0.33002, 0.38258}, {0.915543, 0.332803, 0.381226}, {0.917756, 0.33559, 0.379874}, {0.9199, 0.33845, 0.378577}, {0.922038, 0.341317, 0.377284}, {0.924109, 0.344253, 0.376049}, {0.926172, 0.347198, 0.374822}, {0.928169, 0.350208, 0.373656}, {0.930154, 0.353229, 0.372501}, {0.932076, 0.356311, 0.37141}, {0.933983, 0.359406, 0.370335}, {0.93583, 0.362557, 0.369327}, {0.937659, 0.365724, 0.368339}, {0.93943, 0.368941, 0.367417}, {0.941179, 0.372176, 0.366521}, {0.942875, 0.375456, 0.36569}, {0.944547, 0.378756, 0.364892}, {0.946169, 0.382096, 0.364158}, {0.947763, 0.385456, 0.363464}, {0.949311, 0.38885, 0.362834}, {0.950829, 0.392266, 0.362247}, {0.952306, 0.395712, 0.361721}, {0.953749, 0.39918, 0.361244}, {0.955155, 0.402674, 0.360826}, {0.956525, 0.40619, 0.360463}, {0.957861, 0.409728, 0.360153}, {0.959161, 0.413288, 0.359906}, {0.960431, 0.416864, 0.35971}, {0.961663, 0.420462, 0.359581}, {0.96287, 0.424073, 0.359499}, {0.964037, 0.427704, 0.359488}, {0.965181, 0.431348, 0.359519}, {0.966285, 0.43501, 0.359626}, {0.967369, 0.438681, 0.35977}, {0.968412, 0.44237, 0.359995}, {0.969438, 0.446066, 0.360251}, {0.970423, 0.449779, 0.360592}, {0.971395, 0.453496, 0.360959}, {0.972325, 0.457228, 0.361416}, {0.973244, 0.460963, 0.361894}, {0.974121, 0.464712, 0.362463}, {0.97499, 0.468464, 0.363049}, {0.975817, 0.472226, 0.36373}, {0.976638, 0.47599, 0.364423}, {0.977418, 0.479764, 0.365213}, {0.978195, 0.483538, 0.366009}, {0.978929, 0.48732, 0.366906}, {0.979661, 0.491103, 0.367806}, {0.980354, 0.494892, 0.368803}, {0.981044, 0.498681, 0.369807}, {0.981697, 0.502474, 0.3709}, {0.982347, 0.506268, 0.372004}, {0.982963, 0.510064, 0.373192}, {0.983574, 0.513861, 0.374393}, {0.984155, 0.517659, 0.375671}, {0.984729, 0.521458, 0.376966}, {0.985277, 0.525257, 0.378332}, {0.985816, 0.529057, 0.379718}, {0.986331, 0.532856, 0.381169}, {0.986836, 0.536654, 0.382643}, {0.98732, 0.540453, 0.384176}, {0.987794, 0.544251, 0.385735}, {0.988247, 0.548048, 0.387347}, {0.988689, 0.551845, 0.388988}, {0.989114, 0.55564, 0.390677}, {0.989526, 0.559435, 0.392397}, {0.989922, 0.563228, 0.394161}, {0.990308, 0.567019, 0.395957}, {0.990683, 0.570805, 0.397793}, {0.991046, 0.57459, 0.399663}, {0.991397, 0.578372, 0.401567}, {0.991734, 0.582152, 0.403508}, {0.99206, 0.585931, 0.405479}, {0.992372, 0.589708, 0.407489}, {0.992673, 0.593484, 0.409525}, {0.992958, 0.597259, 0.4116}, {0.993235, 0.601032, 0.413699}, {0.9935, 0.6048, 0.415837}, {0.99376, 0.608566, 0.417995}, {0.994007, 0.612327, 0.420195}, {0.99425, 0.616085, 0.422411}, {0.994475, 0.619842, 0.42467}, {0.994694, 0.623599, 0.426942}, {0.994895, 0.627353, 0.429259}, {0.995091, 0.631107, 0.431586}, {0.995276, 0.634855, 0.433957}, {0.995459, 0.638601, 0.436336}, {0.995629, 0.642341, 0.43876}, {0.995798, 0.64608, 0.44119}, {0.995946, 0.649818, 0.443665}, {0.996092, 0.653556, 0.446145}, {0.996218, 0.657292, 0.448672}, {0.996343, 0.661027, 0.4512}, {0.996465, 0.664751, 0.453772}, {0.996586, 0.668475, 0.456346}, {0.996686, 0.672198, 0.458963}, {0.996783, 0.675921, 0.461586}, {0.996859, 0.679645, 0.464249}, {0.996936, 0.683366, 0.466918}, {0.997014, 0.687076, 0.469619}, {0.997087, 0.690786, 0.472328}, {0.997143, 0.694497, 0.475072}, {0.997194, 0.698208, 0.477827}, {0.997229, 0.701919, 0.480613}, {0.997264, 0.705626, 0.483409}, {0.9973, 0.709324, 0.48623}, {0.997329, 0.713023, 0.489063}, {0.997343, 0.716724, 0.491925}, {0.997351, 0.720423, 0.494801}, {0.997351, 0.72412, 0.4977}, {0.997349, 0.727815, 0.500613}, {0.997344, 0.731505, 0.503545}, {0.997328, 0.735196, 0.506496}, {0.9973, 0.738889, 0.509468}, {0.997271, 0.742578, 0.512455}, {0.997242, 0.746262, 0.515458}, {0.997203, 0.749946, 0.51848}, {0.997157, 0.75363, 0.521519}, {0.997103, 0.757313, 0.524578}, {0.997042, 0.760997, 0.527652}, {0.996981, 0.764676, 0.530743}, {0.996919, 0.768351, 0.533846}, {0.99684, 0.772031, 0.536974}, {0.996753, 0.775712, 0.540115}, {0.996671, 0.779386, 0.543273}, {0.996591, 0.783056, 0.546439}, {0.996494, 0.786729, 0.549631}, {0.99639, 0.790403, 0.552833}, {0.996285, 0.794074, 0.556058}, {0.996179, 0.797744, 0.559288}, {0.996064, 0.801412, 0.562544}, {0.995946, 0.805081, 0.565806}, {0.995819, 0.80875, 0.569094}, {0.99569, 0.812418, 0.572387}, {0.995559, 0.816082, 0.575704}, {0.995429, 0.819746, 0.579023}, {0.99528, 0.823413, 0.582374}, {0.99513, 0.827081, 0.585727}, {0.994987, 0.83074, 0.589103}, {0.994842, 0.8344, 0.592481}, {0.994675, 0.838066, 0.595893}, {0.994509, 0.841731, 0.599306}, {0.994355, 0.845386, 0.602736}, {0.994197, 0.849043, 0.606172}, {0.994015, 0.852708, 0.60964}, {0.993836, 0.85637, 0.61311}, {0.993672, 0.860023, 0.616594}, {0.993502, 0.863677, 0.620086}, {0.993311, 0.867339, 0.623607}, {0.993124, 0.870998, 0.627132}, {0.992951, 0.874648, 0.630668}, {0.992769, 0.878302, 0.634216}, {0.992569, 0.881962, 0.637788}, {0.992377, 0.885618, 0.641365}, {0.992197, 0.889266, 0.64495}, {0.992008, 0.892919, 0.64855}, {0.991803, 0.896576, 0.652171}, {0.991608, 0.900229, 0.655797}, {0.991426, 0.903875, 0.659429}, {0.991233, 0.907526, 0.663077}, {0.991027, 0.911181, 0.666742}, {0.990833, 0.914831, 0.670413}, {0.990649, 0.918476, 0.674089}, {0.990455, 0.922126, 0.677782}, {0.990253, 0.925778, 0.681487}, {0.990062, 0.929426, 0.685199}, {0.989878, 0.933071, 0.688916}, {0.989687, 0.936719, 0.692648}, {0.989493, 0.940368, 0.696389}, {0.989307, 0.944015, 0.700139}, {0.989124, 0.94766, 0.703892}, {0.988941, 0.951308, 0.707658}, {0.988757, 0.954955, 0.711429}, {0.988577, 0.958602, 0.715211}, {0.988398, 0.962249, 0.718996}, {0.988226, 0.965895, 0.72279}, {0.988055, 0.96954, 0.726586}, {0.987881, 0.973189, 0.730395}, {0.987706, 0.976839, 0.734207}, {0.987549, 0.980485, 0.738022}, {0.987394, 0.984131, 0.741837}, {0.987224, 0.987784, 0.74567}, {0.987053, 0.991438, 0.749504}};

const std::vector<glm::vec3> CM_INFERNO =
{{0.001462, 0.000466, 0.013866}, {0.00187337, 0.000876862, 0.0162698}, {0.00228975, 0.00129158, 0.018695}, {0.00281712, 0.00179187, 0.021592}, {0.00335402, 0.00229939, 0.0245331}, {0.00399178, 0.00288349, 0.0279416}, {0.00464349, 0.00347797, 0.0314148}, {0.00538907, 0.0041423, 0.0353237}, {0.00615325, 0.00481933, 0.0392879}, {0.00700666, 0.00555724, 0.0435182}, {0.00788377, 0.00630982, 0.0477516}, {0.00884704, 0.0071157, 0.0519967}, {0.00983902, 0.00793838, 0.056243}, {0.0109132, 0.00880916, 0.0604932}, {0.0120228, 0.00969599, 0.0647565}, {0.0132146, 0.0106199, 0.0690501}, {0.0144475, 0.011562, 0.0733469}, {0.0157588, 0.0125386, 0.0776497}, {0.0171189, 0.0135322, 0.0819654}, {0.0185559, 0.0145527, 0.0863014}, {0.0200506, 0.0155884, 0.090654}, {0.0216215, 0.0166442, 0.0950283}, {0.0232584, 0.017716, 0.0994131}, {0.0249682, 0.0188055, 0.103809}, {0.0267556, 0.0199056, 0.108229}, {0.0286152, 0.0210155, 0.11267}, {0.0305648, 0.0221332, 0.117136}, {0.0325849, 0.0232569, 0.121621}, {0.0347068, 0.0243868, 0.126124}, {0.0368955, 0.0255208, 0.130639}, {0.0391841, 0.0266544, 0.135178}, {0.0415271, 0.0277879, 0.139731}, {0.0438973, 0.0289097, 0.144323}, {0.0462797, 0.0300262, 0.148934}, {0.0486872, 0.0311297, 0.15357}, {0.0511038, 0.0322284, 0.158216}, {0.0535506, 0.0333053, 0.162889}, {0.0560061, 0.0343759, 0.16757}, {0.0584975, 0.0354155, 0.172279}, {0.0609969, 0.0364482, 0.176995}, {0.0635404, 0.0374338, 0.181751}, {0.066091, 0.0384119, 0.186514}, {0.068691, 0.0393326, 0.19131}, {0.0712962, 0.0402474, 0.196109}, {0.0739547, 0.0410753, 0.200935}, {0.0766161, 0.0418985, 0.205761}, {0.0793368, 0.0426265, 0.210611}, {0.0820603, 0.0433501, 0.215461}, {0.0848448, 0.0439777, 0.220328}, {0.0876346, 0.0445972, 0.225196}, {0.0904856, 0.045122, 0.230073}, {0.0933449, 0.0456339, 0.234951}, {0.0962638, 0.0460524, 0.239829}, {0.0991943, 0.046453, 0.244706}, {0.102183, 0.0467627, 0.249574}, {0.105187, 0.0470495, 0.254437}, {0.108245, 0.0472493, 0.259283}, {0.111321, 0.0474214, 0.264119}, {0.114448, 0.0475109, 0.268927}, {0.117596, 0.0475683, 0.273719}, {0.120791, 0.0475489, 0.278473}, {0.124007, 0.0474941, 0.283203}, {0.127266, 0.0473699, 0.287886}, {0.130547, 0.0472081, 0.292535}, {0.133865, 0.0469847, 0.297128}, {0.137206, 0.0467231, 0.301676}, {0.140579, 0.0464093, 0.306161}, {0.143975, 0.0460574, 0.310588}, {0.147396, 0.0456619, 0.314948}, {0.150839, 0.0452312, 0.319235}, {0.154302, 0.0447667, 0.323453}, {0.157782, 0.044275, 0.327581}, {0.161277, 0.0437614, 0.331638}, {0.164787, 0.0432296, 0.335591}, {0.168305, 0.0426854, 0.339473}, {0.171835, 0.0421339, 0.343238}, {0.17537, 0.0415784, 0.346936}, {0.178912, 0.0410278, 0.350503}, {0.182456, 0.0404795, 0.354009}, {0.186001, 0.0399508, 0.357372}, {0.189547, 0.0394296, 0.360681}, {0.193089, 0.038952, 0.363838}, {0.19663, 0.0384874, 0.366949}, {0.200164, 0.0380814, 0.369901}, {0.203696, 0.0376889, 0.372815}, {0.207217, 0.037369, 0.375564}, {0.210736, 0.0370614, 0.378286}, {0.21424, 0.0368395, 0.380839}, {0.217743, 0.0366275, 0.383373}, {0.221226, 0.036514, 0.385738}, {0.224708, 0.0364067, 0.388092}, {0.228171, 0.036405, 0.390277}, {0.231632, 0.036408, 0.392455}, {0.235074, 0.0365184, 0.394476}, {0.238514, 0.0366367, 0.396485}, {0.241935, 0.0368584, 0.398352}, {0.245354, 0.0370928, 0.400203}, {0.248753, 0.0374249, 0.401926}, {0.25215, 0.0377744, 0.403627}, {0.25553, 0.038217, 0.405215}, {0.258906, 0.038681, 0.406777}, {0.262267, 0.0392308, 0.408239}, {0.265622, 0.0398054, 0.409672}, {0.268963, 0.040457, 0.411016}, {0.272298, 0.0411313, 0.412329}, {0.275622, 0.0418626, 0.413564}, {0.278939, 0.042619, 0.414765}, {0.282246, 0.0434264, 0.415897}, {0.285547, 0.0442587, 0.416994}, {0.288839, 0.0451331, 0.418031}, {0.292126, 0.0460319, 0.419031}, {0.295404, 0.046965, 0.419978}, {0.298676, 0.0479216, 0.420887}, {0.301942, 0.0489058, 0.421752}, {0.305201, 0.0499118, 0.422577}, {0.308455, 0.0509395, 0.423363}, {0.311703, 0.0519872, 0.42411}, {0.314946, 0.0530517, 0.424822}, {0.318184, 0.0541345, 0.425495}, {0.321418, 0.0552301, 0.426139}, {0.324646, 0.0563416, 0.426743}, {0.327871, 0.0574622, 0.427322}, {0.331091, 0.0585967, 0.42786}, {0.334309, 0.0597378, 0.428378}, {0.337521, 0.0608907, 0.428854}, {0.340732, 0.0620481, 0.429315}, {0.343938, 0.0632157, 0.429733}, {0.347143, 0.0643864, 0.430138}, {0.350343, 0.0655646, 0.4305}, {0.353543, 0.0667445, 0.430852}, {0.356738, 0.0679301, 0.431162}, {0.359933, 0.0691167, 0.431464}, {0.363125, 0.0703078, 0.431723}, {0.366316, 0.0714996, 0.431977}, {0.369505, 0.0726932, 0.432188}, {0.372693, 0.0738869, 0.432395}, {0.375878, 0.0750817, 0.432559}, {0.379063, 0.0762764, 0.432721}, {0.382246, 0.0774712, 0.432842}, {0.385428, 0.0786659, 0.43296}, {0.388609, 0.0798596, 0.433039}, {0.39179, 0.0810531, 0.433113}, {0.394969, 0.0822438, 0.433151}, {0.398148, 0.0834339, 0.433183}, {0.401326, 0.084621, 0.433181}, {0.404505, 0.0858074, 0.433171}, {0.407683, 0.0869909, 0.43313}, {0.410861, 0.0881734, 0.433079}, {0.414038, 0.0893523, 0.433}, {0.417216, 0.09053, 0.43291}, {0.420393, 0.0917043, 0.432793}, {0.423571, 0.0928771, 0.432664}, {0.426749, 0.0940469, 0.43251}, {0.429927, 0.0952147, 0.432342}, {0.433105, 0.0963794, 0.432152}, {0.436283, 0.0975419, 0.431946}, {0.439462, 0.0987014, 0.431719}, {0.442641, 0.0998586, 0.431476}, {0.44582, 0.101013, 0.431213}, {0.448999, 0.102165, 0.430933}, {0.452179, 0.103316, 0.430636}, {0.45536, 0.104463, 0.430319}, {0.45854, 0.105608, 0.429986}, {0.461721, 0.106751, 0.429632}, {0.464902, 0.107892, 0.429264}, {0.468084, 0.109031, 0.428873}, {0.471267, 0.110168, 0.428469}, {0.47445, 0.111302, 0.428041}, {0.477634, 0.112435, 0.427602}, {0.480818, 0.113566, 0.427139}, {0.484002, 0.114695, 0.426665}, {0.487187, 0.115822, 0.426165}, {0.490372, 0.116949, 0.425656}, {0.493558, 0.118074, 0.425119}, {0.496745, 0.119198, 0.424575}, {0.499931, 0.120321, 0.424003}, {0.503118, 0.121443, 0.423424}, {0.506305, 0.122564, 0.422815}, {0.509493, 0.123685, 0.422202}, {0.51268, 0.124805, 0.421556}, {0.515867, 0.125925, 0.420907}, {0.519055, 0.127044, 0.420225}, {0.522244, 0.128163, 0.419541}, {0.525431, 0.129283, 0.418822}, {0.528619, 0.130403, 0.418101}, {0.531807, 0.131523, 0.417347}, {0.534995, 0.132644, 0.41659}, {0.538183, 0.133766, 0.415801}, {0.54137, 0.134888, 0.415007}, {0.544557, 0.136012, 0.414183}, {0.547744, 0.137137, 0.413353}, {0.55093, 0.138263, 0.412493}, {0.554116, 0.139391, 0.411625}, {0.557301, 0.140521, 0.410731}, {0.560485, 0.141653, 0.409826}, {0.563669, 0.142788, 0.408896}, {0.566852, 0.143925, 0.407955}, {0.570034, 0.145064, 0.40699}, {0.573216, 0.146206, 0.406012}, {0.576396, 0.147352, 0.405011}, {0.579575, 0.1485, 0.403997}, {0.582752, 0.149652, 0.402962}, {0.585928, 0.150808, 0.401911}, {0.589103, 0.151967, 0.40084}, {0.592276, 0.153131, 0.399752}, {0.595448, 0.154299, 0.398646}, {0.598617, 0.155471, 0.397521}, {0.601785, 0.156648, 0.396379}, {0.60495, 0.157831, 0.395217}, {0.608114, 0.159018, 0.394041}, {0.611275, 0.160211, 0.392843}, {0.614435, 0.161409, 0.391632}, {0.617591, 0.162614, 0.390398}, {0.620745, 0.163823, 0.389152}, {0.623895, 0.165042, 0.387882}, {0.627044, 0.166264, 0.386602}, {0.630189, 0.167495, 0.385297}, {0.633332, 0.16873, 0.383982}, {0.63647, 0.169977, 0.382641}, {0.639606, 0.171227, 0.381292}, {0.642737, 0.17249, 0.379915}, {0.645867, 0.173755, 0.378533}, {0.64899, 0.175034, 0.37712}, {0.652112, 0.176315, 0.375703}, {0.655227, 0.177613, 0.374255}, {0.658341, 0.178911, 0.372805}, {0.661447, 0.180227, 0.371323}, {0.664552, 0.181544, 0.36984}, {0.667648, 0.18288, 0.368324}, {0.670744, 0.184217, 0.366806}, {0.67383, 0.185573, 0.365258}, {0.676915, 0.186931, 0.363706}, {0.679991, 0.188308, 0.362126}, {0.683065, 0.189688, 0.360542}, {0.686129, 0.191087, 0.35893}, {0.689192, 0.19249, 0.357313}, {0.692245, 0.193912, 0.35567}, {0.695295, 0.195339, 0.35402}, {0.698335, 0.196785, 0.352347}, {0.701371, 0.198237, 0.350665}, {0.704399, 0.199708, 0.34896}, {0.707422, 0.201186, 0.347246}, {0.710435, 0.202682, 0.345512}, {0.713443, 0.204188, 0.343767}, {0.716441, 0.20571, 0.342003}, {0.719434, 0.207244, 0.340228}, {0.722418, 0.208794, 0.338436}, {0.725395, 0.210357, 0.336631}, {0.728362, 0.211936, 0.334811}, {0.731321, 0.213528, 0.332977}, {0.734271, 0.215137, 0.331129}, {0.737213, 0.216761, 0.329267}, {0.740146, 0.218399, 0.327392}, {0.743069, 0.220054, 0.325502}, {0.745984, 0.221723, 0.323601}, {0.748887, 0.223412, 0.321685}, {0.751783, 0.225112, 0.319758}, {0.754666, 0.226834, 0.317815}, {0.757542, 0.228567, 0.315863}, {0.760404, 0.230323, 0.313894}, {0.76326, 0.23209, 0.311918}, {0.766099, 0.233881, 0.309924}, {0.768934, 0.235682, 0.307924}, {0.771751, 0.237509, 0.305906}, {0.774563, 0.239343, 0.303883}, {0.777356, 0.241207, 0.301841}, {0.780145, 0.243078, 0.299796}, {0.782914, 0.244979, 0.297731}, {0.78568, 0.246884, 0.295663}, {0.788423, 0.248823, 0.293577}, {0.791164, 0.250765, 0.291488}, {0.793881, 0.252742, 0.289381}, {0.796596, 0.25472, 0.287272}, {0.799286, 0.256737, 0.285144}, {0.801975, 0.258755, 0.283015}, {0.804638, 0.260808, 0.280868}, {0.807299, 0.262864, 0.27872}, {0.809934, 0.264956, 0.276554}, {0.812566, 0.267053, 0.274387}, {0.815173, 0.269183, 0.272205}, {0.817776, 0.27132, 0.270019}, {0.820354, 0.273488, 0.267819}, {0.822926, 0.275664, 0.265616}, {0.825474, 0.277872, 0.263401}, {0.828014, 0.28009, 0.261181}, {0.830532, 0.282336, 0.25895}, {0.83304, 0.284594, 0.256714}, {0.835527, 0.286879, 0.254468}, {0.838003, 0.289178, 0.252217}, {0.840458, 0.291502, 0.249956}, {0.8429, 0.293842, 0.24769}, {0.845322, 0.296206, 0.245415}, {0.84773, 0.298586, 0.243135}, {0.850119, 0.300988, 0.240847}, {0.852492, 0.303409, 0.238553}, {0.854847, 0.305851, 0.236252}, {0.857184, 0.308312, 0.233945}, {0.859505, 0.310793, 0.231631}, {0.861806, 0.313295, 0.229311}, {0.864092, 0.315814, 0.226985}, {0.866356, 0.318357, 0.224653}, {0.868606, 0.320914, 0.222316}, {0.870833, 0.323496, 0.219971}, {0.873046, 0.326092, 0.217623}, {0.875235, 0.328714, 0.215266}, {0.877412, 0.331347, 0.212907}, {0.879562, 0.334008, 0.210538}, {0.881701, 0.336679, 0.208167}, {0.883812, 0.339379, 0.205788}, {0.885915, 0.342087, 0.203407}, {0.887986, 0.344825, 0.201016}, {0.89005, 0.34757, 0.198624}, {0.892081, 0.350345, 0.196222}, {0.894106, 0.353126, 0.19382}, {0.896097, 0.35594, 0.191407}, {0.898083, 0.358756, 0.188993}, {0.900033, 0.361606, 0.186569}, {0.90198, 0.364458, 0.184145}, {0.903888, 0.367344, 0.181709}, {0.905794, 0.370232, 0.179273}, {0.907661, 0.373153, 0.176827}, {0.909526, 0.376076, 0.17438}, {0.911354, 0.37903, 0.171923}, {0.913176, 0.381987, 0.169465}, {0.914963, 0.384974, 0.166996}, {0.916743, 0.387966, 0.164525}, {0.918489, 0.390986, 0.162045}, {0.920227, 0.394011, 0.159562}, {0.921931, 0.397062, 0.15707}, {0.923626, 0.40012, 0.154574}, {0.925289, 0.403202, 0.15207}, {0.926941, 0.406292, 0.149562}, {0.928563, 0.409404, 0.147045}, {0.930171, 0.412526, 0.144524}, {0.931751, 0.415668, 0.141994}, {0.933316, 0.418821, 0.13946}, {0.934854, 0.421991, 0.136916}, {0.936375, 0.425173, 0.134367}, {0.937871, 0.428372, 0.131811}, {0.939348, 0.431584, 0.129249}, {0.940802, 0.434811, 0.126679}, {0.942236, 0.438051, 0.124102}, {0.943648, 0.441305, 0.121519}, {0.945037, 0.444573, 0.118928}, {0.946406, 0.447853, 0.116331}, {0.947751, 0.451149, 0.113726}, {0.949078, 0.454454, 0.111116}, {0.950378, 0.457776, 0.108498}, {0.951662, 0.461107, 0.105874}, {0.952918, 0.464454, 0.103243}, {0.95416, 0.46781, 0.100608}, {0.955371, 0.471182, 0.0979643}, {0.95657, 0.474561, 0.0953177}, {0.957736, 0.477958, 0.0926645}, {0.958892, 0.48136, 0.0900092}, {0.960014, 0.484781, 0.0873482}, {0.961127, 0.488206, 0.0846858}, {0.962204, 0.49165, 0.0820207}, {0.963274, 0.495097, 0.0793552}, {0.964305, 0.498563, 0.0766907}, {0.965333, 0.502031, 0.0740262}, {0.966319, 0.50552, 0.0713684}, {0.967303, 0.50901, 0.0687111}, {0.968244, 0.512519, 0.0660683}, {0.969184, 0.516029, 0.0634264}, {0.970081, 0.519559, 0.0608095}, {0.970976, 0.52309, 0.0581952}, {0.97183, 0.526639, 0.0556181}, {0.972679, 0.53019, 0.0530473}, {0.973489, 0.533758, 0.0505269}, {0.974293, 0.537329, 0.0480189}, {0.97506, 0.540915, 0.0455793}, {0.975819, 0.544505, 0.0431603}, {0.976542, 0.548108, 0.0408259}, {0.977255, 0.551716, 0.0385465}, {0.977934, 0.555338, 0.0364416}, {0.978601, 0.558964, 0.0344228}, {0.979237, 0.562603, 0.032623}, {0.979859, 0.566247, 0.0309265}, {0.98045, 0.569901, 0.029444}, {0.981026, 0.573562, 0.0280826}, {0.981573, 0.577233, 0.0269288}, {0.982102, 0.58091, 0.0259156}, {0.982606, 0.584596, 0.0251036}, {0.98309, 0.58829, 0.0244539}, {0.983549, 0.591991, 0.0239986}, {0.983986, 0.5957, 0.0237283}, {0.984401, 0.599417, 0.0236445}, {0.984791, 0.603141, 0.0237708}, {0.985161, 0.606871, 0.0240754}, {0.985505, 0.61061, 0.0246171}, {0.985831, 0.614355, 0.0253274}, {0.986128, 0.618108, 0.0263045}, {0.986409, 0.621867, 0.02744}, {0.98666, 0.625634, 0.0288743}, {0.986896, 0.629405, 0.0304554}, {0.9871, 0.633186, 0.0323698}, {0.987291, 0.63697, 0.034418}, {0.987447, 0.640764, 0.0368383}, {0.987593, 0.644561, 0.039378}, {0.987703, 0.648367, 0.0422142}, {0.987803, 0.652176, 0.0451245}, {0.987865, 0.655995, 0.048239}, {0.98792, 0.659815, 0.0513915}, {0.987935, 0.663646, 0.0547297}, {0.987944, 0.667478, 0.0580917}, {0.987911, 0.671319, 0.0616195}, {0.987875, 0.675162, 0.0651598}, {0.987794, 0.679014, 0.0688513}, {0.987712, 0.682868, 0.0725491}, {0.987584, 0.68673, 0.0763823}, {0.987454, 0.690594, 0.0802227}, {0.98728, 0.694466, 0.0841785}, {0.987102, 0.69834, 0.0881459}, {0.986882, 0.702222, 0.0922152}, {0.986656, 0.706104, 0.0962997}, {0.98639, 0.709995, 0.100474}, {0.986116, 0.713887, 0.104668}, {0.985805, 0.717785, 0.108943}, {0.985483, 0.721686, 0.113241}, {0.985125, 0.725593, 0.117613}, {0.984754, 0.729502, 0.122011}, {0.98435, 0.733416, 0.126479}, {0.983932, 0.737332, 0.130976}, {0.983483, 0.741252, 0.135537}, {0.983018, 0.745175, 0.140133}, {0.982523, 0.749101, 0.144789}, {0.98201, 0.75303, 0.149484}, {0.981471, 0.756961, 0.154236}, {0.980912, 0.760895, 0.159031}, {0.980329, 0.76483, 0.163881}, {0.979725, 0.768768, 0.168779}, {0.979098, 0.772707, 0.173728}, {0.978449, 0.776647, 0.178731}, {0.97778, 0.780589, 0.183783}, {0.977088, 0.784531, 0.188897}, {0.976378, 0.788474, 0.194056}, {0.975643, 0.792418, 0.199284}, {0.974891, 0.796362, 0.204554}, {0.974113, 0.800306, 0.209903}, {0.973321, 0.804249, 0.215292}, {0.972504, 0.808191, 0.220766}, {0.971676, 0.812133, 0.226275}, {0.970823, 0.81607, 0.231879}, {0.969962, 0.820007, 0.237515}, {0.969078, 0.823938, 0.243255}, {0.968188, 0.827868, 0.249022}, {0.967274, 0.831792, 0.254907}, {0.966355, 0.835714, 0.260816}, {0.965413, 0.839628, 0.266857}, {0.964468, 0.843541, 0.272916}, {0.96351, 0.84744, 0.279115}, {0.962551, 0.851338, 0.285327}, {0.961585, 0.855219, 0.291691}, {0.960618, 0.859099, 0.298061}, {0.959644, 0.86296, 0.304608}, {0.958671, 0.86682, 0.311163}, {0.957707, 0.870655, 0.317885}, {0.956746, 0.874487, 0.324623}, {0.955807, 0.878289, 0.331523}, {0.954872, 0.882086, 0.33845}, {0.953961, 0.885854, 0.34555}, {0.953061, 0.889613, 0.352683}, {0.952208, 0.893336, 0.35997}, {0.951371, 0.897047, 0.3673}, {0.950591, 0.900717, 0.374783}, {0.949836, 0.904372, 0.382318}, {0.949154, 0.907981, 0.389992}, {0.948511, 0.91157, 0.397723}, {0.947954, 0.915109, 0.405581}, {0.947452, 0.91862, 0.413498}, {0.947051, 0.922079, 0.421525}, {0.946725, 0.925502, 0.42961}, {0.946512, 0.928872, 0.437784}, {0.946394, 0.932197, 0.446009}, {0.9464, 0.935467, 0.4543}, {0.946526, 0.938685, 0.462629}, {0.946782, 0.941847, 0.470999}, {0.947181, 0.944951, 0.479389}, {0.947709, 0.948002, 0.487798}, {0.948404, 0.950987, 0.496201}, {0.949226, 0.953923, 0.5046}, {0.950231, 0.95679, 0.512969}, {0.951353, 0.959613, 0.521321}, {0.952673, 0.962364, 0.529611}, {0.954099, 0.965077, 0.537868}, {0.95573, 0.967718, 0.546038}, {0.957451, 0.970328, 0.55417}, {0.959379, 0.972867, 0.562203}, {0.96138, 0.975382, 0.5702}, {0.96359, 0.977829, 0.57805}, {0.965858, 0.980259, 0.585859}, {0.968326, 0.982625, 0.593527}, {0.970837, 0.984978, 0.601165}, {0.973542, 0.987271, 0.608652}, {0.976275, 0.989556, 0.616116}, {0.979194, 0.991787, 0.623417}, {0.98213, 0.994013, 0.630703}, {0.985242, 0.99619, 0.637817}, {0.988362, 0.998364, 0.644924}};

const std::vector<glm::vec3> CM_PLASMA =
{{0.050383, 0.029803, 0.527975}, {0.0571045, 0.0290993, 0.530606}, {0.0637965, 0.0283991, 0.533232}, {0.0698352, 0.0277757, 0.535727}, {0.0758322, 0.0271583, 0.538212}, {0.0813865, 0.0266059, 0.540589}, {0.0868937, 0.0260615, 0.542952}, {0.0920842, 0.0255709, 0.545223}, {0.0972256, 0.0250895, 0.547479}, {0.102132, 0.0246521, 0.549659}, {0.106988, 0.024226, 0.55182}, {0.111661, 0.0238412, 0.553915}, {0.116285, 0.0234663, 0.555991}, {0.120771, 0.0231199, 0.558012}, {0.125211, 0.0227823, 0.560014}, {0.129544, 0.0224655, 0.561969}, {0.133831, 0.0221573, 0.563904}, {0.138033, 0.0218655, 0.565799}, {0.142191, 0.0215813, 0.567674}, {0.146281, 0.0213089, 0.569515}, {0.15033, 0.0210431, 0.571334}, {0.154323, 0.0207861, 0.573124}, {0.158276, 0.0205346, 0.574893}, {0.162185, 0.0202893, 0.576637}, {0.166055, 0.020048, 0.578358}, {0.16989, 0.0198104, 0.580059}, {0.173688, 0.0195759, 0.581737}, {0.177457, 0.0193439, 0.583397}, {0.181191, 0.0191134, 0.585034}, {0.184903, 0.018884, 0.586656}, {0.188581, 0.0186545, 0.588254}, {0.19224, 0.0184251, 0.589839}, {0.195868, 0.0181946, 0.5914}, {0.199481, 0.0179636, 0.592951}, {0.203064, 0.0177296, 0.594477}, {0.206637, 0.0174945, 0.595994}, {0.210179, 0.0172559, 0.597486}, {0.213713, 0.0170162, 0.598971}, {0.217218, 0.0167736, 0.60043}, {0.220717, 0.0165304, 0.601884}, {0.224189, 0.016281, 0.60331}, {0.227657, 0.0160306, 0.604733}, {0.231099, 0.0157732, 0.606128}, {0.23454, 0.0155152, 0.607521}, {0.237955, 0.0152484, 0.608885}, {0.241369, 0.0149811, 0.610248}, {0.244761, 0.0147052, 0.611582}, {0.248151, 0.014429, 0.612914}, {0.251521, 0.0141443, 0.614218}, {0.25489, 0.013859, 0.615519}, {0.25824, 0.0135657, 0.616792}, {0.261588, 0.0132712, 0.618062}, {0.26492, 0.0129687, 0.619307}, {0.268249, 0.0126649, 0.620546}, {0.271565, 0.0123547, 0.62176}, {0.274877, 0.012043, 0.622968}, {0.278176, 0.0117257, 0.624152}, {0.281472, 0.0114068, 0.625327}, {0.284757, 0.0110833, 0.626481}, {0.288038, 0.0107585, 0.627625}, {0.29131, 0.0104304, 0.628747}, {0.294577, 0.0101006, 0.629858}, {0.297836, 0.00976744, 0.630948}, {0.30109, 0.0094329, 0.632026}, {0.304338, 0.00909613, 0.633084}, {0.307581, 0.00875851, 0.634128}, {0.310817, 0.0084197, 0.635153}, {0.314049, 0.00808089, 0.636163}, {0.317275, 0.00774208, 0.637154}, {0.320496, 0.0074038, 0.638128}, {0.323712, 0.00706601, 0.639084}, {0.326923, 0.0067302, 0.640022}, {0.330131, 0.00639599, 0.640944}, {0.333332, 0.00606514, 0.641844}, {0.336529, 0.00573655, 0.642729}, {0.339722, 0.00541319, 0.643592}, {0.342912, 0.00509278, 0.64444}, {0.346096, 0.00477864, 0.645264}, {0.349277, 0.00446743, 0.646075}, {0.352452, 0.00416549, 0.646859}, {0.355625, 0.00386705, 0.647631}, {0.358792, 0.00358, 0.648374}, {0.361957, 0.00329639, 0.649107}, {0.365117, 0.0030277, 0.649808}, {0.368275, 0.00276248, 0.6505}, {0.371426, 0.00251474, 0.651158}, {0.374576, 0.00226996, 0.65181}, {0.377719, 0.00204721, 0.652423}, {0.380862, 0.00182696, 0.653032}, {0.383998, 0.00163123, 0.653601}, {0.387134, 0.00143705, 0.654168}, {0.390262, 0.00127304, 0.654691}, {0.39339, 0.00111042, 0.655212}, {0.39651, 0.000980112, 0.655689}, {0.399631, 0.000852471, 0.656163}, {0.402744, 0.000759976, 0.656595}, {0.405856, 0.00067213, 0.657021}, {0.408962, 0.000620517, 0.657406}, {0.412066, 0.000575958, 0.657783}, {0.415164, 0.000569315, 0.65812}, {0.41826, 0.000572381, 0.658448}, {0.42135, 0.000614285, 0.658737}, {0.424438, 0.000668986, 0.659014}, {0.42752, 0.000763525, 0.659254}, {0.430599, 0.000874303, 0.659479}, {0.433673, 0.00102557, 0.65967}, {0.436744, 0.00119652, 0.659843}, {0.439809, 0.00140758, 0.659982}, {0.442871, 0.00164281, 0.660102}, {0.445928, 0.00191876, 0.660189}, {0.448981, 0.00222339, 0.660255}, {0.452028, 0.00256833, 0.660291}, {0.455071, 0.00294703, 0.660302}, {0.45811, 0.00336556, 0.660285}, {0.461143, 0.00382307, 0.660242}, {0.464172, 0.00431928, 0.660171}, {0.467195, 0.00486061, 0.660072}, {0.470214, 0.00543959, 0.659948}, {0.473227, 0.00606938, 0.659792}, {0.476235, 0.00673473, 0.659615}, {0.479237, 0.00745752, 0.659403}, {0.482234, 0.00821383, 0.659171}, {0.485225, 0.0090346, 0.658902}, {0.488212, 0.00988647, 0.658615}, {0.49119, 0.0108102, 0.658289}, {0.494165, 0.0117623, 0.657947}, {0.497132, 0.0127928, 0.657563}, {0.500097, 0.0138481, 0.657166}, {0.503051, 0.0149909, 0.656724}, {0.506003, 0.016155, 0.656271}, {0.508944, 0.0174152, 0.655772}, {0.511883, 0.0186927, 0.655265}, {0.514811, 0.0200749, 0.654709}, {0.517738, 0.02147, 0.654146}, {0.520652, 0.0229791, 0.653533}, {0.523564, 0.0244963, 0.652916}, {0.526464, 0.0261363, 0.652245}, {0.529363, 0.0277818, 0.651572}, {0.532248, 0.029555, 0.650846}, {0.535132, 0.0313367, 0.650116}, {0.538003, 0.0332443, 0.649337}, {0.540872, 0.0351666, 0.648552}, {0.543727, 0.0372128, 0.647719}, {0.54658, 0.0392725, 0.646878}, {0.549421, 0.0414096, 0.645992}, {0.552258, 0.0435479, 0.645097}, {0.555083, 0.0456917, 0.64416}, {0.557904, 0.0478377, 0.64321}, {0.560712, 0.0499911, 0.642222}, {0.563515, 0.0521473, 0.64122}, {0.566307, 0.0543104, 0.640181}, {0.569093, 0.0564764, 0.639127}, {0.571869, 0.0586482, 0.63804}, {0.574638, 0.0608234, 0.636935}, {0.577396, 0.0630045, 0.635799}, {0.580147, 0.0651886, 0.634644}, {0.582887, 0.0673774, 0.633461}, {0.585619, 0.0695698, 0.632258}, {0.588342, 0.0717666, 0.631029}, {0.591055, 0.0739668, 0.629779}, {0.59376, 0.0761703, 0.628506}, {0.596454, 0.0783777, 0.62721}, {0.599139, 0.0805884, 0.625894}, {0.601814, 0.0828026, 0.624554}, {0.604481, 0.0850194, 0.623197}, {0.607136, 0.0872401, 0.621815}, {0.609784, 0.089463, 0.620417}, {0.61242, 0.0916894, 0.618993}, {0.615049, 0.0939174, 0.617557}, {0.617664, 0.0961491, 0.616094}, {0.620274, 0.0983823, 0.614621}, {0.622869, 0.100619, 0.613121}, {0.625459, 0.102856, 0.611613}, {0.628033, 0.105097, 0.610077}, {0.630604, 0.107339, 0.608535}, {0.633157, 0.109583, 0.606965}, {0.635708, 0.111828, 0.60539}, {0.63824, 0.114076, 0.603789}, {0.64077, 0.116324, 0.602185}, {0.643283, 0.118576, 0.600553}, {0.645793, 0.120827, 0.598918}, {0.648285, 0.123081, 0.597258}, {0.650775, 0.125336, 0.595597}, {0.653245, 0.127592, 0.593911}, {0.655715, 0.129849, 0.592223}, {0.658164, 0.132107, 0.590513}, {0.660612, 0.134366, 0.588801}, {0.663042, 0.136625, 0.587069}, {0.665469, 0.138885, 0.585335}, {0.667879, 0.141147, 0.583583}, {0.670286, 0.143409, 0.581828}, {0.672676, 0.145671, 0.580056}, {0.675061, 0.147934, 0.578281}, {0.677431, 0.150197, 0.576493}, {0.679796, 0.152461, 0.574701}, {0.682145, 0.154724, 0.572898}, {0.684489, 0.156988, 0.57109}, {0.686819, 0.159253, 0.569272}, {0.689143, 0.161517, 0.56745}, {0.691453, 0.163782, 0.56562}, {0.693757, 0.166047, 0.563786}, {0.696048, 0.168312, 0.561944}, {0.698331, 0.170577, 0.560099}, {0.700602, 0.172841, 0.558247}, {0.702865, 0.175106, 0.556392}, {0.705118, 0.177371, 0.554532}, {0.707361, 0.179636, 0.552669}, {0.709594, 0.1819, 0.550802}, {0.711817, 0.184164, 0.548931}, {0.714032, 0.186429, 0.547058}, {0.716235, 0.188693, 0.545182}, {0.718432, 0.190957, 0.543304}, {0.720616, 0.19322, 0.541423}, {0.722795, 0.195483, 0.539542}, {0.72496, 0.197746, 0.537658}, {0.72712, 0.200009, 0.535773}, {0.729266, 0.202272, 0.533887}, {0.731408, 0.204534, 0.532001}, {0.733536, 0.206796, 0.530113}, {0.735661, 0.209058, 0.528226}, {0.737771, 0.211319, 0.526339}, {0.739879, 0.21358, 0.524453}, {0.74197, 0.215841, 0.522566}, {0.74406, 0.218102, 0.520679}, {0.746134, 0.220362, 0.518794}, {0.748208, 0.222622, 0.516908}, {0.750264, 0.224882, 0.515025}, {0.75232, 0.227142, 0.513142}, {0.75436, 0.229402, 0.511261}, {0.756399, 0.231661, 0.50938}, {0.758423, 0.233921, 0.507502}, {0.760445, 0.23618, 0.505625}, {0.762453, 0.238438, 0.503751}, {0.764459, 0.240697, 0.501877}, {0.76645, 0.242956, 0.500006}, {0.768439, 0.245216, 0.498136}, {0.770415, 0.247474, 0.496269}, {0.772389, 0.249733, 0.494404}, {0.77435, 0.251992, 0.492543}, {0.776307, 0.254251, 0.490683}, {0.778253, 0.25651, 0.488827}, {0.780195, 0.258769, 0.486973}, {0.782126, 0.261029, 0.485123}, {0.784052, 0.263289, 0.483274}, {0.785968, 0.265548, 0.481429}, {0.787879, 0.267808, 0.479585}, {0.789781, 0.270069, 0.477745}, {0.791677, 0.272329, 0.475908}, {0.793564, 0.274591, 0.474074}, {0.795446, 0.276852, 0.472242}, {0.797319, 0.279115, 0.470413}, {0.799186, 0.281377, 0.468587}, {0.801046, 0.283641, 0.466764}, {0.802897, 0.285905, 0.464945}, {0.804743, 0.288169, 0.463128}, {0.806581, 0.290434, 0.461314}, {0.808413, 0.2927, 0.459502}, {0.810236, 0.294967, 0.457695}, {0.812056, 0.297234, 0.45589}, {0.813865, 0.299503, 0.454089}, {0.81567, 0.301772, 0.452289}, {0.817465, 0.304042, 0.450494}, {0.819257, 0.306313, 0.4487}, {0.821039, 0.308586, 0.44691}, {0.822818, 0.31086, 0.445122}, {0.824586, 0.313135, 0.443337}, {0.826353, 0.315411, 0.441554}, {0.828107, 0.317688, 0.439775}, {0.82986, 0.319967, 0.437996}, {0.831601, 0.322247, 0.436223}, {0.83334, 0.324528, 0.434449}, {0.835067, 0.326812, 0.432681}, {0.836794, 0.329096, 0.430912}, {0.838508, 0.331383, 0.429149}, {0.840222, 0.33367, 0.427386}, {0.841923, 0.33596, 0.425627}, {0.843623, 0.338251, 0.423868}, {0.845311, 0.340545, 0.422114}, {0.846998, 0.342839, 0.420359}, {0.848673, 0.345137, 0.418609}, {0.850346, 0.347436, 0.416858}, {0.852009, 0.349738, 0.415111}, {0.853668, 0.352041, 0.413365}, {0.855318, 0.354348, 0.411621}, {0.856964, 0.356655, 0.409878}, {0.858601, 0.358966, 0.408138}, {0.860234, 0.361278, 0.406399}, {0.861857, 0.363594, 0.404663}, {0.863476, 0.365911, 0.402927}, {0.865087, 0.368231, 0.401194}, {0.866692, 0.370554, 0.399461}, {0.868289, 0.37288, 0.397729}, {0.86988, 0.375209, 0.395999}, {0.871464, 0.37754, 0.39427}, {0.873042, 0.379874, 0.392542}, {0.874612, 0.382211, 0.390816}, {0.876176, 0.384551, 0.38909}, {0.877733, 0.386894, 0.387365}, {0.879282, 0.38924, 0.385641}, {0.880826, 0.391589, 0.383918}, {0.882361, 0.393942, 0.382196}, {0.88389, 0.396298, 0.380474}, {0.885411, 0.398658, 0.378754}, {0.886926, 0.401021, 0.377034}, {0.888432, 0.403388, 0.375314}, {0.889934, 0.405757, 0.373595}, {0.891425, 0.408131, 0.371877}, {0.892912, 0.410508, 0.370159}, {0.894387, 0.412889, 0.368441}, {0.89586, 0.415273, 0.366724}, {0.89732, 0.417663, 0.365006}, {0.898778, 0.420054, 0.363289}, {0.900223, 0.422453, 0.361573}, {0.901666, 0.424852, 0.359856}, {0.903095, 0.427258, 0.35814}, {0.904523, 0.429665, 0.356423}, {0.905936, 0.43208, 0.354707}, {0.907348, 0.434496, 0.35299}, {0.908745, 0.43692, 0.351273}, {0.910141, 0.439344, 0.349556}, {0.911522, 0.441777, 0.34784}, {0.912902, 0.444211, 0.346123}, {0.914267, 0.446653, 0.344405}, {0.91563, 0.449095, 0.342688}, {0.916978, 0.451546, 0.34097}, {0.918323, 0.453999, 0.339253}, {0.919654, 0.456459, 0.337534}, {0.920982, 0.458921, 0.335815}, {0.922297, 0.461391, 0.334096}, {0.923607, 0.463864, 0.332376}, {0.924904, 0.466343, 0.330656}, {0.926196, 0.468826, 0.328936}, {0.927476, 0.471315, 0.327214}, {0.92875, 0.473808, 0.325493}, {0.930011, 0.476308, 0.323771}, {0.931266, 0.478812, 0.322048}, {0.93251, 0.481323, 0.320325}, {0.933746, 0.483838, 0.318602}, {0.934972, 0.486358, 0.316878}, {0.936188, 0.488884, 0.315153}, {0.937394, 0.491416, 0.313428}, {0.938591, 0.493953, 0.311702}, {0.939778, 0.496495, 0.309975}, {0.940954, 0.499044, 0.308248}, {0.942122, 0.501598, 0.306521}, {0.943278, 0.504158, 0.304792}, {0.944425, 0.506723, 0.303063}, {0.945561, 0.509295, 0.301334}, {0.946688, 0.511871, 0.299605}, {0.947802, 0.514455, 0.297875}, {0.948909, 0.517042, 0.296144}, {0.950001, 0.519639, 0.294413}, {0.951088, 0.522238, 0.292682}, {0.952158, 0.524847, 0.290949}, {0.953223, 0.527458, 0.289216}, {0.954271, 0.530079, 0.287482}, {0.955314, 0.532702, 0.285748}, {0.956339, 0.535336, 0.284014}, {0.957361, 0.537971, 0.28228}, {0.958362, 0.540617, 0.280545}, {0.959361, 0.543265, 0.27881}, {0.96034, 0.545924, 0.277074}, {0.961317, 0.548584, 0.275339}, {0.962271, 0.551256, 0.273604}, {0.963225, 0.553928, 0.271868}, {0.964155, 0.556613, 0.270133}, {0.965084, 0.559298, 0.268397}, {0.965991, 0.561995, 0.266662}, {0.966895, 0.564694, 0.264927}, {0.967778, 0.567404, 0.263191}, {0.968657, 0.570116, 0.261456}, {0.969515, 0.572839, 0.25972}, {0.970368, 0.575564, 0.257985}, {0.971201, 0.5783, 0.256251}, {0.972028, 0.58104, 0.254516}, {0.972836, 0.583788, 0.252784}, {0.973637, 0.586541, 0.251051}, {0.974419, 0.589302, 0.249319}, {0.975193, 0.592068, 0.247588}, {0.97595, 0.594843, 0.245859}, {0.976697, 0.597623, 0.24413}, {0.977427, 0.600411, 0.242403}, {0.978146, 0.603204, 0.240677}, {0.978849, 0.606005, 0.238953}, {0.979541, 0.608812, 0.23723}, {0.980217, 0.611626, 0.23551}, {0.980879, 0.614447, 0.233791}, {0.981528, 0.617275, 0.232075}, {0.982162, 0.620109, 0.230361}, {0.982783, 0.62295, 0.228649}, {0.983387, 0.625799, 0.226939}, {0.983979, 0.628654, 0.225231}, {0.984552, 0.631517, 0.223527}, {0.985115, 0.634385, 0.221826}, {0.985659, 0.637262, 0.220128}, {0.986192, 0.640144, 0.218433}, {0.986705, 0.643035, 0.216744}, {0.987209, 0.645929, 0.215058}, {0.987691, 0.648835, 0.213378}, {0.988165, 0.651743, 0.2117}, {0.988615, 0.654663, 0.21003}, {0.989058, 0.657585, 0.208362}, {0.989476, 0.660518, 0.206704}, {0.989888, 0.663453, 0.205047}, {0.990273, 0.666401, 0.203402}, {0.990654, 0.66935, 0.201758}, {0.991006, 0.672311, 0.200127}, {0.991355, 0.675274, 0.198498}, {0.991673, 0.678249, 0.196883}, {0.991989, 0.681226, 0.19527}, {0.992274, 0.684216, 0.193673}, {0.992556, 0.687207, 0.192077}, {0.992807, 0.69021, 0.1905}, {0.993054, 0.693215, 0.188925}, {0.993271, 0.696231, 0.18737}, {0.993483, 0.69925, 0.185819}, {0.993665, 0.702281, 0.184287}, {0.993842, 0.705314, 0.18276}, {0.993989, 0.708358, 0.181254}, {0.994129, 0.711405, 0.179755}, {0.994242, 0.714463, 0.178279}, {0.994345, 0.717524, 0.176811}, {0.994422, 0.720595, 0.175367}, {0.994487, 0.723671, 0.173933}, {0.994527, 0.726756, 0.172523}, {0.994554, 0.729846, 0.171127}, {0.994559, 0.732944, 0.169756}, {0.994547, 0.736048, 0.168401}, {0.994514, 0.739159, 0.167071}, {0.994463, 0.742277, 0.165761}, {0.994391, 0.745402, 0.164476}, {0.994301, 0.748534, 0.163216}, {0.994192, 0.751672, 0.16198}, {0.994062, 0.754818, 0.160774}, {0.993914, 0.757969, 0.159592}, {0.993742, 0.761129, 0.158444}, {0.993554, 0.764295, 0.157319}, {0.99334, 0.767469, 0.156231}, {0.99311, 0.770648, 0.155167}, {0.992854, 0.773836, 0.154147}, {0.992584, 0.777028, 0.153149}, {0.992286, 0.780229, 0.152201}, {0.991975, 0.783435, 0.151275}, {0.991634, 0.78665, 0.150405}, {0.991282, 0.789868, 0.149554}, {0.990897, 0.793096, 0.148767}, {0.990504, 0.796327, 0.147997}, {0.990075, 0.799568, 0.147298}, {0.98964, 0.802811, 0.146612}, {0.989165, 0.806066, 0.146003}, {0.988686, 0.809324, 0.145404}, {0.988164, 0.812593, 0.144889}, {0.98764, 0.815863, 0.144381}, {0.987073, 0.819144, 0.143966}, {0.986504, 0.822427, 0.143555}, {0.985894, 0.82572, 0.143242}, {0.985281, 0.829015, 0.142934}, {0.984625, 0.83232, 0.142721}, {0.983965, 0.835627, 0.142517}, {0.983261, 0.838948, 0.142402}, {0.98255, 0.842269, 0.142301}, {0.981803, 0.845599, 0.142289}, {0.981047, 0.848932, 0.142295}, {0.980257, 0.852272, 0.142384}, {0.979456, 0.855616, 0.142494}, {0.978613, 0.858971, 0.142675}, {0.977759, 0.862329, 0.142882}, {0.976875, 0.865694, 0.143159}, {0.975977, 0.869062, 0.143463}, {0.975045, 0.872438, 0.143826}, {0.974098, 0.875817, 0.144216}, {0.97312, 0.879204, 0.144657}, {0.972126, 0.882595, 0.145125}, {0.971105, 0.885991, 0.145634}, {0.970064, 0.889393, 0.146165}, {0.968996, 0.8928, 0.146724}, {0.967908, 0.896212, 0.147301}, {0.966798, 0.899628, 0.147897}, {0.965667, 0.903048, 0.1485}, {0.964517, 0.906473, 0.149108}, {0.963341, 0.909903, 0.149704}, {0.962145, 0.913338, 0.150292}, {0.960929, 0.916778, 0.150847}, {0.9597, 0.920219, 0.151382}, {0.95845, 0.923664, 0.151848}, {0.957189, 0.927111, 0.152279}, {0.955909, 0.930562, 0.152592}, {0.95462, 0.934014, 0.152853}, {0.953317, 0.93747, 0.152923}, {0.952008, 0.940926, 0.152925}, {0.950694, 0.944382, 0.152626}, {0.949378, 0.947839, 0.152244}, {0.948073, 0.951291, 0.151396}, {0.946771, 0.954743, 0.15045}, {0.945512, 0.958182, 0.148786}, {0.94426, 0.961619, 0.147014}, {0.943099, 0.965032, 0.144104}, {0.941946, 0.968443, 0.141086}, {0.940976, 0.971802, 0.136247}, {0.940015, 0.975158, 0.131326}};

const std::vector<glm::vec3> CM_COOLWARM =
{{0.2298057,0.298717966,0.753683153},{0.2298057,0.298717966,0.753683153},{0.234377079451,0.305541730329,0.759679527588},{0.234377079451,0.305541730329,0.759679527588},{0.238948458902,0.312365494659,0.765675902176},{0.238948458902,0.312365494659,0.765675902176},{0.243519838353,0.319189258988,0.771672276765},{0.243519838353,0.319189258988,0.771672276765},{0.248091217804,0.326013023318,0.777668651353},{0.248091217804,0.326013023318,0.777668651353},{0.252662597255,0.332836787647,0.783665025941},{0.252662597255,0.332836787647,0.783665025941},{0.257233976706,0.339660551976,0.789661400529},{0.257233976706,0.339660551976,0.789661400529},{0.261805356157,0.346484316306,0.795657775118},{0.261805356157,0.346484316306,0.795657775118},{0.266381468353,0.353304408424,0.801637319498},{0.266381468353,0.353304408424,0.801637319498},{0.271104295647,0.360010661976,0.807095127435},{0.271104295647,0.360010661976,0.807095127435},{0.275827122941,0.366716915529,0.812552935373},{0.275827122941,0.366716915529,0.812552935373},{0.280549950235,0.373423169082,0.81801074331},{0.280549950235,0.373423169082,0.81801074331},{0.285272777529,0.380129422635,0.823468551247},{0.285272777529,0.380129422635,0.823468551247},{0.289995604824,0.386835676188,0.828926359184},{0.289995604824,0.386835676188,0.828926359184},{0.294718432118,0.393541929741,0.834384167122},{0.294718432118,0.393541929741,0.834384167122},{0.299441259412,0.400248183294,0.839841975059},{0.299441259412,0.400248183294,0.839841975059},{0.304174287004,0.406944882839,0.845262726698},{0.304174287004,0.406944882839,0.845262726698},{0.309060319067,0.413498272267,0.850127633867},{0.309060319067,0.413498272267,0.850127633867},{0.313946351129,0.420051661694,0.854992541035},{0.313946351129,0.420051661694,0.854992541035},{0.318832383192,0.426605051122,0.859857448204},{0.323718415255,0.433158440549,0.864722355373},{0.323718415255,0.433158440549,0.864722355373},{0.328604447318,0.439711829976,0.869587262541},{0.328604447318,0.439711829976,0.869587262541},{0.33349047938,0.446265219404,0.87445216971},{0.33349047938,0.446265219404,0.87445216971},{0.338376511443,0.452818608831,0.879317076878},{0.338376511443,0.452818608831,0.879317076878},{0.343277523435,0.459353634729,0.884121921624},{0.343277523435,0.459353634729,0.884121921624},{0.348323341412,0.46571114651,0.888346162941},{0.348323341412,0.46571114651,0.888346162941},{0.353369159388,0.47206865829,0.892570404259},{0.353369159388,0.47206865829,0.892570404259},{0.358414977365,0.478426170071,0.896794645576},{0.358414977365,0.478426170071,0.896794645576},{0.363460795341,0.484783681851,0.901018886894},{0.363460795341,0.484783681851,0.901018886894},{0.368506613318,0.491141193631,0.905243128212},{0.368506613318,0.491141193631,0.905243128212},{0.373552431294,0.497498705412,0.909467369529},{0.373552431294,0.497498705412,0.909467369529},{0.378598249271,0.503856217192,0.913691610847},{0.378598249271,0.503856217192,0.913691610847},{0.383662065773,0.510183417286,0.917830673231},{0.383662065773,0.510183417286,0.917830673231},{0.388851871953,0.516298435576,0.921373483082},{0.388851871953,0.516298435576,0.921373483082},{0.394041678133,0.522413453867,0.924916292933},{0.394041678133,0.522413453867,0.924916292933},{0.399231484314,0.528528472157,0.928459102784},{0.399231484314,0.528528472157,0.928459102784},{0.404421290494,0.534643490447,0.932001912635},{0.404421290494,0.534643490447,0.932001912635},{0.409611096675,0.540758508737,0.935544722486},{0.409611096675,0.540758508737,0.935544722486},{0.414800902855,0.546873527027,0.939087532337},{0.414800902855,0.546873527027,0.939087532337},{0.419990709035,0.552988545318,0.942630342188},{0.425198970196,0.559058179765,0.946061457078},{0.425198970196,0.559058179765,0.946061457078},{0.430506888251,0.564882741459,0.94888941918},{0.430506888251,0.564882741459,0.94888941918},{0.435814806306,0.570707303153,0.951717381282},{0.435814806306,0.570707303153,0.951717381282},{0.441122724361,0.576531864847,0.954545343384},{0.441122724361,0.576531864847,0.954545343384},{0.446430642416,0.582356426541,0.957373305486},{0.446430642416,0.582356426541,0.957373305486},{0.451738560471,0.588180988235,0.960201267588},{0.451738560471,0.588180988235,0.960201267588},{0.457046478525,0.594005549929,0.96302922969},{0.457046478525,0.594005549929,0.96302922969},{0.46235439658,0.599830111624,0.965857191792},{0.46235439658,0.599830111624,0.965857191792},{0.467678094682,0.605591231624,0.968546281094},{0.467678094682,0.605591231624,0.968546281094},{0.473070172988,0.611077437616,0.970633588263},{0.473070172988,0.611077437616,0.970633588263},{0.478462251294,0.616563643608,0.972720895431},{0.478462251294,0.616563643608,0.972720895431},{0.4838543296,0.6220498496,0.9748082026},{0.4838543296,0.6220498496,0.9748082026},{0.489246407906,0.627536055592,0.976895509769},{0.489246407906,0.627536055592,0.976895509769},{0.494638486212,0.633022261584,0.978982816937},{0.494638486212,0.633022261584,0.978982816937},{0.500030564518,0.638508467576,0.981070124106},{0.500030564518,0.638508467576,0.981070124106},{0.505422642824,0.643994673569,0.983157431275},{0.505422642824,0.643994673569,0.983157431275},{0.510824324251,0.649396614824,0.985078776376},{0.510824324251,0.649396614824,0.985078776376},{0.516260302541,0.654497610588,0.986407399812},{0.516260302541,0.654497610588,0.986407399812},{0.521696280831,0.659598606353,0.987736023247},{0.521696280831,0.659598606353,0.987736023247},{0.527132259122,0.664699602118,0.989064646682},{0.532568237412,0.669800597882,0.990393270118},{0.532568237412,0.669800597882,0.990393270118},{0.538004215702,0.674901593647,0.991721893553},{0.538004215702,0.674901593647,0.991721893553},{0.543440193992,0.680002589412,0.993050516988},{0.543440193992,0.680002589412,0.993050516988},{0.548876172282,0.685103585176,0.994379140424},{0.548876172282,0.685103585176,0.994379140424},{0.554311869914,0.690097011216,0.995515548235},{0.554311869914,0.690097011216,0.995515548235},{0.559746725569,0.694767728078,0.996075309176},{0.559746725569,0.694767728078,0.996075309176},{0.565181581224,0.699438444941,0.996635070118},{0.565181581224,0.699438444941,0.996635070118},{0.570616436878,0.704109161804,0.997194831059},{0.570616436878,0.704109161804,0.997194831059},{0.576051292533,0.708779878667,0.997754592},{0.576051292533,0.708779878667,0.997754592},{0.581486148188,0.713450595529,0.998314352941},{0.581486148188,0.713450595529,0.998314352941},{0.586921003843,0.718121312392,0.998874113882},{0.586921003843,0.718121312392,0.998874113882},{0.592355859498,0.722792029255,0.999433874824},{0.592355859498,0.722792029255,0.999433874824},{0.597776775494,0.727329724882,0.999776731776},{0.597776775494,0.727329724882,0.999776731776},{0.603162067918,0.731527477353,0.999565278537},{0.603162067918,0.731527477353,0.999565278537},{0.608547360341,0.735725229824,0.999353825298},{0.608547360341,0.735725229824,0.999353825298},{0.613932652765,0.739922982294,0.999142372059},{0.613932652765,0.739922982294,0.999142372059},{0.619317945188,0.744120734765,0.99893091882},{0.619317945188,0.744120734765,0.99893091882},{0.624703237612,0.748318487235,0.99871946558},{0.624703237612,0.748318487235,0.99871946558},{0.630088530035,0.752516239706,0.998508012341},{0.635473822459,0.756713992176,0.998296559102},{0.635473822459,0.756713992176,0.998296559102},{0.640827782373,0.760751506412,0.997845774882},{0.640827782373,0.760751506412,0.997845774882},{0.646112810765,0.764436496529,0.996868462506},{0.646112810765,0.764436496529,0.996868462506},{0.651397839157,0.768121486647,0.995891150129},{0.651397839157,0.768121486647,0.995891150129},{0.656682867549,0.771806476765,0.994913837753},{0.656682867549,0.771806476765,0.994913837753},{0.661967895941,0.775491466882,0.993936525376},{0.661967895941,0.775491466882,0.993936525376},{0.667252924333,0.779176457,0.992959213},{0.667252924333,0.779176457,0.992959213},{0.672537952725,0.782861447118,0.991981900624},{0.672537952725,0.782861447118,0.991981900624},{0.677822981118,0.786546437235,0.991004588247},{0.677822981118,0.786546437235,0.991004588247},{0.683055681561,0.79004262689,0.989768428184},{0.683055681561,0.79004262689,0.989768428184},{0.688188483192,0.793178379298,0.988038104357},{0.688188483192,0.793178379298,0.988038104357},{0.693321284824,0.796314131706,0.986307780529},{0.693321284824,0.796314131706,0.986307780529},{0.698454086455,0.799449884114,0.984577456702},{0.698454086455,0.799449884114,0.984577456702},{0.703586888086,0.802585636522,0.982847132875},{0.703586888086,0.802585636522,0.982847132875},{0.708719689718,0.805721388929,0.981116809047},{0.708719689718,0.805721388929,0.981116809047},{0.713852491349,0.808857141337,0.97938648522},{0.713852491349,0.808857141337,0.97938648522},{0.71898529298,0.811992893745,0.977656161392},{0.71898529298,0.811992893745,0.977656161392},{0.724041371882,0.814910392647,0.975650970647},{0.724041371882,0.814910392647,0.975650970647},{0.728969579569,0.817464135706,0.973187668373},{0.728969579569,0.817464135706,0.973187668373},{0.733897787255,0.820017878765,0.970724366098},{0.738825994941,0.822571621824,0.968261063824},{0.738825994941,0.822571621824,0.968261063824},{0.743754202627,0.825125364882,0.965797761549},{0.743754202627,0.825125364882,0.965797761549},{0.748682410314,0.827679107941,0.963334459275},{0.748682410314,0.827679107941,0.963334459275},{0.753610618,0.830232851,0.960871157},{0.753610618,0.830232851,0.960871157},{0.758538825686,0.832786594059,0.958407854725},{0.758538825686,0.832786594059,0.958407854725},{0.763362780102,0.83509222182,0.955657676557},{0.763362780102,0.83509222182,0.955657676557},{0.768034364353,0.837035219529,0.952488218235},{0.768034364353,0.837035219529,0.952488218235},{0.772705948604,0.838978217239,0.949318759914},{0.772705948604,0.838978217239,0.949318759914},{0.777377532855,0.840921214949,0.946149301592},{0.777377532855,0.840921214949,0.946149301592},{0.782049117106,0.842864212659,0.942979843271},{0.782049117106,0.842864212659,0.942979843271},{0.786720701357,0.844807210369,0.939810384949},{0.786720701357,0.844807210369,0.939810384949},{0.791392285608,0.846750208078,0.936640926627},{0.791392285608,0.846750208078,0.936640926627},{0.796063869859,0.848693205788,0.933471468306},{0.796063869859,0.848693205788,0.933471468306},{0.800600847294,0.850358321561,0.930007560392},{0.800600847294,0.850358321561,0.930007560392},{0.804964758824,0.851666160557,0.926165074431},{0.804964758824,0.851666160557,0.926165074431},{0.809328670353,0.852973999553,0.922322588471},{0.809328670353,0.852973999553,0.922322588471},{0.813692581882,0.854281838549,0.91848010251},{0.813692581882,0.854281838549,0.91848010251},{0.818056493412,0.855589677545,0.914637616549},{0.818056493412,0.855589677545,0.914637616549},{0.822420404941,0.856897516541,0.910795130588},{0.822420404941,0.856897516541,0.910795130588},{0.826784316471,0.858205355537,0.906952644627},{0.831148228,0.859513194533,0.903110158667},{0.831148228,0.859513194533,0.903110158667},{0.835344711353,0.860513997294,0.898970409941},{0.835344711353,0.860513997294,0.898970409941},{0.839351442773,0.861166825655,0.894493763416},{0.839351442773,0.861166825655,0.894493763416},{0.843358174192,0.861819654016,0.89001711689},{0.843358174192,0.861819654016,0.89001711689},{0.847364905612,0.862472482376,0.885540470365},{0.847364905612,0.862472482376,0.885540470365},{0.851371637031,0.863125310737,0.881063823839},{0.851371637031,0.863125310737,0.881063823839},{0.855378368451,0.863778139098,0.876587177314},{0.855378368451,0.863778139098,0.876587177314},{0.859385099871,0.864430967459,0.872110530788},{0.859385099871,0.864430967459,0.872110530788},{0.86339183129,0.86508379582,0.867633884263},{0.86339183129,0.86508379582,0.867633884263},{0.867427635086,0.864376599773,0.86260246202},{0.867427635086,0.864376599773,0.86260246202},{0.871492511259,0.862309379318,0.857016264059},{0.871492511259,0.862309379318,0.857016264059},{0.875557387431,0.860242158863,0.851430066098},{0.875557387431,0.860242158863,0.851430066098},{0.879622263604,0.858174938408,0.845843868137},{0.879622263604,0.858174938408,0.845843868137},{0.883687139776,0.856107717953,0.840257670176},{0.883687139776,0.856107717953,0.840257670176},{0.887752015949,0.854040497498,0.834671472216},{0.887752015949,0.854040497498,0.834671472216},{0.891816892122,0.851973277043,0.829085274255},{0.891816892122,0.851973277043,0.829085274255},{0.895881768294,0.849906056588,0.823499076294},{0.895881768294,0.849906056588,0.823499076294},{0.8995432066,0.847500236,0.8177890744},{0.8995432066,0.847500236,0.8177890744},{0.902848670318,0.844795650588,0.811969833741},{0.906154134035,0.842091065176,0.806150593082},{0.906154134035,0.842091065176,0.806150593082},{0.909459597753,0.839386479765,0.800331352424},{0.909459597753,0.839386479765,0.800331352424},{0.912765061471,0.836681894353,0.794512111765},{0.912765061471,0.836681894353,0.794512111765},{0.916070525188,0.833977308941,0.788692871106},{0.916070525188,0.833977308941,0.788692871106},{0.919375988906,0.831272723529,0.782873630447},{0.919375988906,0.831272723529,0.782873630447},{0.922681452624,0.828568138118,0.777054389788},{0.922681452624,0.828568138118,0.777054389788},{0.925563423,0.825517298071,0.771136307812},{0.925563423,0.825517298071,0.771136307812},{0.928116009667,0.822197148863,0.765141349255},{0.928116009667,0.822197148863,0.765141349255},{0.930668596333,0.818876999655,0.759146390698},{0.930668596333,0.818876999655,0.759146390698},{0.933221183,0.815556850447,0.753151432141},{0.933221183,0.815556850447,0.753151432141},{0.935773769667,0.812236701239,0.747156473584},{0.935773769667,0.812236701239,0.747156473584},{0.938326356333,0.808916552031,0.741161515027},{0.938326356333,0.808916552031,0.741161515027},{0.940878943,0.805596402824,0.735166556471},{0.940878943,0.805596402824,0.735166556471},{0.943431529667,0.802276253616,0.729171597914},{0.943431529667,0.802276253616,0.729171597914},{0.94554029891,0.798605740533,0.723105417298},{0.94554029891,0.798605740533,0.723105417298},{0.9473454036,0.7946955048,0.7169905058},{0.9473454036,0.7946955048,0.7169905058},{0.94915050829,0.790785269067,0.710875594302},{0.94915050829,0.790785269067,0.710875594302},{0.95095561298,0.786875033333,0.704760682804},{0.95095561298,0.786875033333,0.704760682804},{0.952760717671,0.7829647976,0.698645771306},{0.952760717671,0.7829647976,0.698645771306},{0.954565822361,0.779054561867,0.692530859808},{0.956370927051,0.775144326133,0.68641594831},{0.956370927051,0.775144326133,0.68641594831},{0.958176031741,0.7712340904,0.680301036812},{0.958176031741,0.7712340904,0.680301036812},{0.959517658471,0.76697285451,0.674144715039},{0.959517658471,0.76697285451,0.674144715039},{0.960581198424,0.762501018525,0.667963547102},{0.960581198424,0.762501018525,0.667963547102},{0.961644738376,0.758029182541,0.661782379165},{0.961644738376,0.758029182541,0.661782379165},{0.962708278329,0.753557346557,0.655601211227},{0.962708278329,0.753557346557,0.655601211227},{0.963771818282,0.749085510573,0.64942004329},{0.963771818282,0.749085510573,0.64942004329},{0.964835358235,0.744613674588,0.643238875353},{0.964835358235,0.744613674588,0.643238875353},{0.965898898188,0.740141838604,0.637057707416},{0.965898898188,0.740141838604,0.637057707416},{0.966962438141,0.73567000262,0.630876539478},{0.966962438141,0.73567000262,0.630876539478},{0.967544297635,0.730849716188,0.624685478235},{0.967544297635,0.730849716188,0.624685478235},{0.967873848318,0.725846908094,0.618489234784},{0.967873848318,0.725846908094,0.618489234784},{0.968203399,0.7208441,0.612292991333},{0.968203399,0.7208441,0.612292991333},{0.968532949682,0.715841291906,0.606096747882},{0.968532949682,0.715841291906,0.606096747882},{0.968862500365,0.710838483812,0.599900504431},{0.968862500365,0.710838483812,0.599900504431},{0.969192051047,0.705835675718,0.59370426098},{0.969192051047,0.705835675718,0.59370426098},{0.969521601729,0.700832867624,0.587508017529},{0.969521601729,0.700832867624,0.587508017529},{0.969851152412,0.695830059529,0.581311774078},{0.969851152412,0.695830059529,0.581311774078},{0.969682979667,0.690483930737,0.575138361365},{0.969682979667,0.690483930737,0.575138361365},{0.969288569,0.684981747082,0.568975326259},{0.968894158333,0.679479563427,0.562812291153},{0.968894158333,0.679479563427,0.562812291153},{0.968499747667,0.673977379773,0.556649256047},{0.968499747667,0.673977379773,0.556649256047},{0.968105337,0.668475196118,0.550486220941},{0.968105337,0.668475196118,0.550486220941},{0.967710926333,0.662973012463,0.544323185835},{0.967710926333,0.662973012463,0.544323185835},{0.967316515667,0.657470828808,0.538160150729},{0.967316515667,0.657470828808,0.538160150729},{0.966922105,0.651968645153,0.531997115624},{0.966922105,0.651968645153,0.531997115624},{0.966016719839,0.646129741588,0.525890348259},{0.966016719839,0.646129741588,0.525890348259},{0.964911388137,0.640159078059,0.519805598706},{0.964911388137,0.640159078059,0.519805598706},{0.963806056435,0.634188414529,0.513720849153},{0.963806056435,0.634188414529,0.513720849153},{0.962700724733,0.628217751,0.5076360996},{0.962700724733,0.628217751,0.5076360996},{0.961595393031,0.622247087471,0.501551350047},{0.961595393031,0.622247087471,0.501551350047},{0.960490061329,0.616276423941,0.495466600494},{0.960490061329,0.616276423941,0.495466600494},{0.959384729627,0.610305760412,0.489381850941},{0.959384729627,0.610305760412,0.489381850941},{0.958279397925,0.604335096882,0.483297101388},{0.958279397925,0.604335096882,0.483297101388},{0.956653210976,0.598033822718,0.477302292353},{0.956653210976,0.598033822718,0.477302292353},{0.954853405612,0.591622345008,0.47133746349},{0.954853405612,0.591622345008,0.47133746349},{0.953053600247,0.585210867298,0.465372634627},{0.953053600247,0.585210867298,0.465372634627},{0.951253794882,0.578799389588,0.459407805765},{0.951253794882,0.578799389588,0.459407805765},{0.949453989518,0.572387911878,0.453442976902},{0.947654184153,0.565976434169,0.447478148039},{0.947654184153,0.565976434169,0.447478148039},{0.945854378788,0.559564956459,0.441513319176},{0.945854378788,0.559564956459,0.441513319176},{0.944054573424,0.553153478749,0.435548490314},{0.944054573424,0.553153478749,0.435548490314},{0.941727929824,0.54641347702,0.429707070373},{0.941727929824,0.54641347702,0.429707070373},{0.939253771518,0.539581488565,0.423900204929},{0.939253771518,0.539581488565,0.423900204929},{0.936779613212,0.53274950011,0.418093339486},{0.936779613212,0.53274950011,0.418093339486},{0.934305454906,0.525917511655,0.412286474043},{0.934305454906,0.525917511655,0.412286474043},{0.9318312966,0.5190855232,0.4064796086},{0.9318312966,0.5190855232,0.4064796086},{0.929357138294,0.512253534745,0.400672743157},{0.929357138294,0.512253534745,0.400672743157},{0.926882979988,0.50542154629,0.394865877714},{0.926882979988,0.50542154629,0.394865877714},{0.924408821682,0.498589557835,0.389059012271},{0.924408821682,0.498589557835,0.389059012271},{0.921406221227,0.491420417184,0.383408435376},{0.921406221227,0.491420417184,0.383408435376},{0.918281672584,0.48417347218,0.377793925071},{0.918281672584,0.48417347218,0.377793925071},{0.915157123941,0.476926527176,0.372179414765},{0.915157123941,0.476926527176,0.372179414765},{0.912032575298,0.469679582173,0.366564904459},{0.912032575298,0.469679582173,0.366564904459},{0.908908026655,0.462432637169,0.360950394153},{0.908908026655,0.462432637169,0.360950394153},{0.905783478012,0.455185692165,0.355335883847},{0.905783478012,0.455185692165,0.355335883847},{0.902658929369,0.447938747161,0.349721373541},{0.902658929369,0.447938747161,0.349721373541},{0.899534380725,0.440691802157,0.344106863235},{0.899534380725,0.440691802157,0.344106863235},{0.895884594835,0.433074556706,0.338680634518},{0.892137542788,0.425388737098,0.333289272761},{0.892137542788,0.425388737098,0.333289272761},{0.888390490741,0.41770291749,0.327897911004},{0.888390490741,0.41770291749,0.327897911004},{0.884643438694,0.410017097882,0.322506549247},{0.884643438694,0.410017097882,0.322506549247},{0.880896386647,0.402331278275,0.31711518749},{0.880896386647,0.402331278275,0.31711518749},{0.8771493346,0.394645458667,0.311723825733},{0.8771493346,0.394645458667,0.311723825733},{0.873402282553,0.386959639059,0.306332463976},{0.873402282553,0.386959639059,0.306332463976},{0.869655230506,0.379273819451,0.30094110222},{0.869655230506,0.379273819451,0.30094110222},{0.865391332937,0.371127672047,0.295768956416},{0.865391332937,0.371127672047,0.295768956416},{0.861053600294,0.362915763529,0.290628127176},{0.861053600294,0.362915763529,0.290628127176},{0.856715867651,0.354703855012,0.285487297937},{0.856715867651,0.354703855012,0.285487297937},{0.852378135008,0.346491946494,0.280346468698},{0.852378135008,0.346491946494,0.280346468698},{0.848040402365,0.338280037976,0.275205639459},{0.848040402365,0.338280037976,0.275205639459},{0.843702669722,0.330068129459,0.27006481022},{0.843702669722,0.330068129459,0.27006481022},{0.839364937078,0.321856220941,0.26492398098},{0.839364937078,0.321856220941,0.26492398098},{0.835027204435,0.313644312424,0.259783151741},{0.835027204435,0.313644312424,0.259783151741},{0.830186521949,0.304732763553,0.254891428067},{0.830186521949,0.304732763553,0.254891428067},{0.825293810169,0.295748838094,0.250025473933},{0.825293810169,0.295748838094,0.250025473933},{0.820401098388,0.286764912635,0.2451595198},{0.820401098388,0.286764912635,0.2451595198},{0.815508386608,0.277780987176,0.240293565667},{0.815508386608,0.277780987176,0.240293565667},{0.810615674827,0.268797061718,0.235427611533},{0.805722963047,0.259813136259,0.2305616574},{0.805722963047,0.259813136259,0.2305616574},{0.800830251267,0.2508292108,0.225695703267},{0.800830251267,0.2508292108,0.225695703267},{0.795937539486,0.241845285341,0.220829749133},{0.795937539486,0.241845285341,0.220829749133},{0.790561531941,0.231396999059,0.216242038294},{0.790561531941,0.231396999059,0.216242038294},{0.785153304678,0.220851088722,0.211672877008},{0.785153304678,0.220851088722,0.211672877008},{0.779745077416,0.210305178384,0.207103715722},{0.779745077416,0.210305178384,0.207103715722},{0.774336850153,0.199759268047,0.202534554435},{0.774336850153,0.199759268047,0.202534554435},{0.76892862289,0.18921335771,0.197965393149},{0.76892862289,0.18921335771,0.197965393149},{0.763520395627,0.178667447373,0.193396231863},{0.763520395627,0.178667447373,0.193396231863},{0.758112168365,0.168121537035,0.188827070576},{0.758112168365,0.168121537035,0.188827070576},{0.752703941102,0.157575626698,0.18425790929},{0.752703941102,0.157575626698,0.18425790929},{0.746838012212,0.140021019482,0.179996096957},{0.746838012212,0.140021019482,0.179996096957},{0.740957318753,0.122240325271,0.175744199106},{0.740957318753,0.122240325271,0.175744199106},{0.735076625294,0.104459631059,0.171492301255},{0.735076625294,0.104459631059,0.171492301255},{0.729195931835,0.0866789368471,0.167240403404},{0.729195931835,0.0866789368471,0.167240403404},{0.723315238376,0.0688982426353,0.162988505553},{0.723315238376,0.0688982426353,0.162988505553},{0.717434544918,0.0511175484235,0.158736607702},{0.717434544918,0.0511175484235,0.158736607702},{0.711553851459,0.0333368542118,0.154484709851},{0.711553851459,0.0333368542118,0.154484709851},{0.705673158,0.01555616,0.150232812},{0.705673158,0.01555616,0.150232812},};

const std::vector<glm::vec3> CM_BLUES =
{{0.96862745285,0.984313726425,1.0},{0.96862745285,0.984313726425,1.0},{0.965551712934,0.982345252879,0.999015763227},{0.965551712934,0.982345252879,0.999015763227},{0.962475973017,0.980376779332,0.998031526453},{0.962475973017,0.980376779332,0.998031526453},{0.9594002331,0.978408305785,0.99704728968},{0.9594002331,0.978408305785,0.99704728968},{0.956324493184,0.976439832239,0.996063052907},{0.956324493184,0.976439832239,0.996063052907},{0.953248753267,0.974471358692,0.995078816133},{0.953248753267,0.974471358692,0.995078816133},{0.950173013351,0.972502885145,0.99409457936},{0.950173013351,0.972502885145,0.99409457936},{0.947097273434,0.970534411599,0.993110342587},{0.947097273434,0.970534411599,0.993110342587},{0.944021533517,0.968565938052,0.992126105813},{0.944021533517,0.968565938052,0.992126105813},{0.940945793601,0.966597464505,0.99114186904},{0.940945793601,0.966597464505,0.99114186904},{0.937870053684,0.964628990959,0.990157632267},{0.937870053684,0.964628990959,0.990157632267},{0.934794313767,0.962660517412,0.989173395493},{0.934794313767,0.962660517412,0.989173395493},{0.931718573851,0.960692043865,0.98818915872},{0.931718573851,0.960692043865,0.98818915872},{0.928642833934,0.958723570319,0.987204921947},{0.928642833934,0.958723570319,0.987204921947},{0.925567094017,0.956755096772,0.986220685173},{0.925567094017,0.956755096772,0.986220685173},{0.922491354101,0.954786623225,0.9852364484},{0.922491354101,0.954786623225,0.9852364484},{0.919415614184,0.952818149679,0.984252211627},{0.919415614184,0.952818149679,0.984252211627},{0.916339874268,0.950849676132,0.983267974854},{0.916339874268,0.950849676132,0.983267974854},{0.913264134351,0.948881202586,0.98228373808},{0.913264134351,0.948881202586,0.98228373808},{0.910188394434,0.946912729039,0.981299501307},{0.907112654518,0.944944255492,0.980315264534},{0.907112654518,0.944944255492,0.980315264534},{0.904036914601,0.942975781946,0.97933102776},{0.904036914601,0.942975781946,0.97933102776},{0.900961174684,0.941007308399,0.978346790987},{0.900961174684,0.941007308399,0.978346790987},{0.897885434768,0.939038834852,0.977362554214},{0.897885434768,0.939038834852,0.977362554214},{0.894809694851,0.937070361306,0.97637831744},{0.894809694851,0.937070361306,0.97637831744},{0.891733954935,0.935101887759,0.975394080667},{0.891733954935,0.935101887759,0.975394080667},{0.888658215018,0.933133414212,0.974409843894},{0.888658215018,0.933133414212,0.974409843894},{0.885582475101,0.931164940666,0.97342560712},{0.885582475101,0.931164940666,0.97342560712},{0.882506735185,0.929196467119,0.972441370347},{0.882506735185,0.929196467119,0.972441370347},{0.879430995268,0.927227993572,0.971457133574},{0.879430995268,0.927227993572,0.971457133574},{0.876355255351,0.925259520026,0.9704728968},{0.876355255351,0.925259520026,0.9704728968},{0.873279515435,0.923291046479,0.969488660027},{0.873279515435,0.923291046479,0.969488660027},{0.870219154218,0.921322572933,0.968504423254},{0.870219154218,0.921322572933,0.968504423254},{0.867266443898,0.919354099386,0.96752018648},{0.867266443898,0.919354099386,0.96752018648},{0.864313733578,0.917385625839,0.966535949707},{0.864313733578,0.917385625839,0.966535949707},{0.861361023258,0.915417152293,0.965551712934},{0.861361023258,0.915417152293,0.965551712934},{0.858408312938,0.913448678746,0.96456747616},{0.858408312938,0.913448678746,0.96456747616},{0.855455602618,0.911480205199,0.963583239387},{0.855455602618,0.911480205199,0.963583239387},{0.852502892298,0.909511731653,0.962599002614},{0.852502892298,0.909511731653,0.962599002614},{0.849550181978,0.907543258106,0.96161476584},{0.846597471658,0.905574784559,0.960630529067},{0.846597471658,0.905574784559,0.960630529067},{0.843644761338,0.903606311013,0.959646292294},{0.843644761338,0.903606311013,0.959646292294},{0.840692051018,0.901637837466,0.95866205552},{0.840692051018,0.901637837466,0.95866205552},{0.837739340698,0.899669363919,0.957677818747},{0.837739340698,0.899669363919,0.957677818747},{0.834786630378,0.897700890373,0.956693581974},{0.834786630378,0.897700890373,0.956693581974},{0.831833920058,0.895732416826,0.9557093452},{0.831833920058,0.895732416826,0.9557093452},{0.828881209738,0.893763943279,0.954725108427},{0.828881209738,0.893763943279,0.954725108427},{0.825928499418,0.891795469733,0.953740871654},{0.825928499418,0.891795469733,0.953740871654},{0.822975789098,0.889826996186,0.952756634881},{0.822975789098,0.889826996186,0.952756634881},{0.820023078778,0.88785852264,0.951772398107},{0.820023078778,0.88785852264,0.951772398107},{0.817070368458,0.885890049093,0.950788161334},{0.817070368458,0.885890049093,0.950788161334},{0.814117658138,0.883921575546,0.949803924561},{0.814117658138,0.883921575546,0.949803924561},{0.811164947818,0.881953102,0.948819687787},{0.811164947818,0.881953102,0.948819687787},{0.808212237498,0.879984628453,0.947835451014},{0.808212237498,0.879984628453,0.947835451014},{0.805259527178,0.878016154906,0.946851214241},{0.805259527178,0.878016154906,0.946851214241},{0.802306816858,0.87604768136,0.945866977467},{0.802306816858,0.87604768136,0.945866977467},{0.799354106538,0.874079207813,0.944882740694},{0.799354106538,0.874079207813,0.944882740694},{0.796401396218,0.872110734266,0.943898503921},{0.796401396218,0.872110734266,0.943898503921},{0.793448685899,0.87014226072,0.942914267147},{0.793448685899,0.87014226072,0.942914267147},{0.790495975579,0.868173787173,0.941930030374},{0.787543265259,0.866205313626,0.940945793601},{0.787543265259,0.866205313626,0.940945793601},{0.784590554939,0.86423684008,0.939961556827},{0.784590554939,0.86423684008,0.939961556827},{0.781637844619,0.862268366533,0.938977320054},{0.781637844619,0.862268366533,0.938977320054},{0.778685134299,0.860299892987,0.937993083281},{0.778685134299,0.860299892987,0.937993083281},{0.775240305592,0.858300662041,0.936824302112},{0.775240305592,0.858300662041,0.936824302112},{0.770319121725,0.856209158897,0.935101887759},{0.770319121725,0.856209158897,0.935101887759},{0.765397937859,0.854117655754,0.933379473406},{0.765397937859,0.854117655754,0.933379473406},{0.760476753992,0.852026152611,0.931657059052},{0.760476753992,0.852026152611,0.931657059052},{0.755555570126,0.849934649467,0.929934644699},{0.755555570126,0.849934649467,0.929934644699},{0.750634386259,0.847843146324,0.928212230346},{0.750634386259,0.847843146324,0.928212230346},{0.745713202392,0.845751643181,0.926489815992},{0.745713202392,0.845751643181,0.926489815992},{0.740792018526,0.843660140038,0.924767401639},{0.740792018526,0.843660140038,0.924767401639},{0.735870834659,0.841568636894,0.923044987286},{0.735870834659,0.841568636894,0.923044987286},{0.730949650793,0.839477133751,0.921322572933},{0.730949650793,0.839477133751,0.921322572933},{0.726028466926,0.837385630608,0.919600158579},{0.726028466926,0.837385630608,0.919600158579},{0.721107283059,0.835294127464,0.917877744226},{0.721107283059,0.835294127464,0.917877744226},{0.716186099193,0.833202624321,0.916155329873},{0.716186099193,0.833202624321,0.916155329873},{0.711264915326,0.831111121178,0.914432915519},{0.711264915326,0.831111121178,0.914432915519},{0.706343731459,0.829019618034,0.912710501166},{0.701422547593,0.826928114891,0.910988086813},{0.701422547593,0.826928114891,0.910988086813},{0.696501363726,0.824836611748,0.909265672459},{0.696501363726,0.824836611748,0.909265672459},{0.69158017986,0.822745108604,0.907543258106},{0.69158017986,0.822745108604,0.907543258106},{0.686658995993,0.820653605461,0.905820843753},{0.686658995993,0.820653605461,0.905820843753},{0.681737812126,0.818562102318,0.904098429399},{0.681737812126,0.818562102318,0.904098429399},{0.67681662826,0.816470599174,0.902376015046},{0.67681662826,0.816470599174,0.902376015046},{0.671895444393,0.814379096031,0.900653600693},{0.671895444393,0.814379096031,0.900653600693},{0.666974260527,0.812287592888,0.898931186339},{0.666974260527,0.812287592888,0.898931186339},{0.66205307666,0.810196089745,0.897208771986},{0.66205307666,0.810196089745,0.897208771986},{0.657131892793,0.808104586601,0.895486357633},{0.657131892793,0.808104586601,0.895486357633},{0.652210708927,0.806013083458,0.893763943279},{0.652210708927,0.806013083458,0.893763943279},{0.64728952506,0.803921580315,0.892041528926},{0.64728952506,0.803921580315,0.892041528926},{0.642368341193,0.801830077171,0.890319114573},{0.642368341193,0.801830077171,0.890319114573},{0.637447157327,0.799738574028,0.88859670022},{0.637447157327,0.799738574028,0.88859670022},{0.63252597346,0.797647070885,0.886874285866},{0.63252597346,0.797647070885,0.886874285866},{0.627604789594,0.795555567741,0.885151871513},{0.627604789594,0.795555567741,0.885151871513},{0.622683605727,0.793464064598,0.88342945716},{0.622683605727,0.793464064598,0.88342945716},{0.617254924424,0.790865064369,0.881845451103},{0.617254924424,0.790865064369,0.881845451103},{0.610980414059,0.787420235662,0.880492125539},{0.610980414059,0.787420235662,0.880492125539},{0.604705903694,0.783975406955,0.879138799976},{0.598431393329,0.780530578249,0.877785474413},{0.598431393329,0.780530578249,0.877785474413},{0.592156882964,0.777085749542,0.876432148849},{0.592156882964,0.777085749542,0.876432148849},{0.585882372599,0.773640920835,0.875078823286},{0.585882372599,0.773640920835,0.875078823286},{0.579607862234,0.770196092129,0.873725497723},{0.579607862234,0.770196092129,0.873725497723},{0.573333351869,0.766751263422,0.872372172159},{0.573333351869,0.766751263422,0.872372172159},{0.567058841504,0.763306434715,0.871018846596},{0.567058841504,0.763306434715,0.871018846596},{0.560784331139,0.759861606009,0.869665521033},{0.560784331139,0.759861606009,0.869665521033},{0.554509820774,0.756416777302,0.868312195469},{0.554509820774,0.756416777302,0.868312195469},{0.54823531041,0.752971948596,0.866958869906},{0.54823531041,0.752971948596,0.866958869906},{0.541960800045,0.749527119889,0.865605544343},{0.541960800045,0.749527119889,0.865605544343},{0.53568628968,0.746082291182,0.864252218779},{0.53568628968,0.746082291182,0.864252218779},{0.529411779315,0.742637462476,0.862898893216},{0.529411779315,0.742637462476,0.862898893216},{0.52313726895,0.739192633769,0.861545567653},{0.52313726895,0.739192633769,0.861545567653},{0.516862758585,0.735747805062,0.860192242089},{0.516862758585,0.735747805062,0.860192242089},{0.51058824822,0.732302976356,0.858838916526},{0.51058824822,0.732302976356,0.858838916526},{0.504313737855,0.728858147649,0.857485590963},{0.504313737855,0.728858147649,0.857485590963},{0.49803922749,0.725413318943,0.856132265399},{0.49803922749,0.725413318943,0.856132265399},{0.491764717125,0.721968490236,0.854778939836},{0.491764717125,0.721968490236,0.854778939836},{0.485490206761,0.718523661529,0.853425614273},{0.485490206761,0.718523661529,0.853425614273},{0.479215696396,0.715078832823,0.85207228871},{0.472941186031,0.711634004116,0.850718963146},{0.472941186031,0.711634004116,0.850718963146},{0.466666675666,0.708189175409,0.849365637583},{0.466666675666,0.708189175409,0.849365637583},{0.460392165301,0.704744346703,0.84801231202},{0.460392165301,0.704744346703,0.84801231202},{0.454117654936,0.701299517996,0.846658986456},{0.454117654936,0.701299517996,0.846658986456},{0.447843144571,0.69785468929,0.845305660893},{0.447843144571,0.69785468929,0.845305660893},{0.441568634206,0.694409860583,0.84395233533},{0.441568634206,0.694409860583,0.84395233533},{0.435294123841,0.690965031876,0.842599009766},{0.435294123841,0.690965031876,0.842599009766},{0.429019613476,0.68752020317,0.841245684203},{0.429019613476,0.68752020317,0.841245684203},{0.422745103111,0.684075374463,0.83989235864},{0.422745103111,0.684075374463,0.83989235864},{0.417085741197,0.680630545756,0.838231459085},{0.417085741197,0.680630545756,0.838231459085},{0.412041527734,0.67718571705,0.836262985538},{0.412041527734,0.67718571705,0.836262985538},{0.406997314271,0.673740888343,0.834294511991},{0.406997314271,0.673740888343,0.834294511991},{0.401953100808,0.670296059637,0.832326038445},{0.401953100808,0.670296059637,0.832326038445},{0.396908887344,0.66685123093,0.830357564898},{0.396908887344,0.66685123093,0.830357564898},{0.391864673881,0.663406402223,0.828389091351},{0.391864673881,0.663406402223,0.828389091351},{0.386820460418,0.659961573517,0.826420617805},{0.386820460418,0.659961573517,0.826420617805},{0.381776246954,0.65651674481,0.824452144258},{0.381776246954,0.65651674481,0.824452144258},{0.376732033491,0.653071916103,0.822483670712},{0.376732033491,0.653071916103,0.822483670712},{0.371687820028,0.649627087397,0.820515197165},{0.366643606565,0.64618225869,0.818546723618},{0.366643606565,0.64618225869,0.818546723618},{0.361599393101,0.642737429983,0.816578250072},{0.361599393101,0.642737429983,0.816578250072},{0.356555179638,0.639292601277,0.814609776525},{0.356555179638,0.639292601277,0.814609776525},{0.351510966175,0.63584777257,0.812641302978},{0.351510966175,0.63584777257,0.812641302978},{0.346466752711,0.632402943864,0.810672829432},{0.346466752711,0.632402943864,0.810672829432},{0.341422539248,0.628958115157,0.808704355885},{0.341422539248,0.628958115157,0.808704355885},{0.336378325785,0.62551328645,0.806735882338},{0.336378325785,0.62551328645,0.806735882338},{0.331334112322,0.622068457744,0.804767408792},{0.331334112322,0.622068457744,0.804767408792},{0.326289898858,0.618623629037,0.802798935245},{0.326289898858,0.618623629037,0.802798935245},{0.321245685395,0.61517880033,0.800830461698},{0.321245685395,0.61517880033,0.800830461698},{0.316201471932,0.611733971624,0.798861988152},{0.316201471932,0.611733971624,0.798861988152},{0.311157258469,0.608289142917,0.796893514605},{0.311157258469,0.608289142917,0.796893514605},{0.306113045005,0.604844314211,0.794925041058},{0.306113045005,0.604844314211,0.794925041058},{0.301068831542,0.601399485504,0.792956567512},{0.301068831542,0.601399485504,0.792956567512},{0.296024618079,0.597954656797,0.790988093965},{0.296024618079,0.597954656797,0.790988093965},{0.290980404615,0.594509828091,0.789019620419},{0.290980404615,0.594509828091,0.789019620419},{0.285936191152,0.591064999384,0.787051146872},{0.285936191152,0.591064999384,0.787051146872},{0.280891977689,0.587620170677,0.785082673325},{0.280891977689,0.587620170677,0.785082673325},{0.275847764226,0.584175341971,0.783114199779},{0.275847764226,0.584175341971,0.783114199779},{0.270803550762,0.580730513264,0.781145726232},{0.265759337299,0.577285684558,0.779177252685},{0.265759337299,0.577285684558,0.779177252685},{0.260715123836,0.573840855851,0.777208779139},{0.260715123836,0.573840855851,0.777208779139},{0.256286058064,0.57001155907,0.775163412094},{0.256286058064,0.57001155907,0.775163412094},{0.252226080906,0.565951581445,0.773071908951},{0.252226080906,0.565951581445,0.773071908951},{0.248166103749,0.56189160382,0.770980405807},{0.248166103749,0.56189160382,0.770980405807},{0.244106126591,0.557831626196,0.768888902664},{0.244106126591,0.557831626196,0.768888902664},{0.240046149434,0.553771648571,0.766797399521},{0.240046149434,0.553771648571,0.766797399521},{0.235986172276,0.549711670946,0.764705896378},{0.235986172276,0.549711670946,0.764705896378},{0.231926195119,0.545651693321,0.762614393234},{0.231926195119,0.545651693321,0.762614393234},{0.227866217961,0.541591715696,0.760522890091},{0.227866217961,0.541591715696,0.760522890091},{0.223806240804,0.537531738071,0.758431386948},{0.223806240804,0.537531738071,0.758431386948},{0.219746263647,0.533471760446,0.756339883804},{0.219746263647,0.533471760446,0.756339883804},{0.215686286489,0.529411782821,0.754248380661},{0.215686286489,0.529411782821,0.754248380661},{0.211626309332,0.525351805196,0.752156877518},{0.211626309332,0.525351805196,0.752156877518},{0.207566332174,0.521291827571,0.750065374374},{0.207566332174,0.521291827571,0.750065374374},{0.203506355017,0.517231849946,0.747973871231},{0.203506355017,0.517231849946,0.747973871231},{0.199446377859,0.513171872321,0.745882368088},{0.199446377859,0.513171872321,0.745882368088},{0.195386400702,0.509111894696,0.743790864944},{0.195386400702,0.509111894696,0.743790864944},{0.191326423545,0.505051917071,0.741699361801},{0.191326423545,0.505051917071,0.741699361801},{0.187266446387,0.500991939447,0.739607858658},{0.18320646923,0.496931961822,0.737516355515},{0.18320646923,0.496931961822,0.737516355515},{0.179146492072,0.492871984197,0.735424852371},{0.179146492072,0.492871984197,0.735424852371},{0.175086514915,0.488812006572,0.733333349228},{0.175086514915,0.488812006572,0.733333349228},{0.171026537757,0.484752028947,0.731241846085},{0.171026537757,0.484752028947,0.731241846085},{0.1669665606,0.480692051322,0.729150342941},{0.1669665606,0.480692051322,0.729150342941},{0.162906583442,0.476632073697,0.727058839798},{0.162906583442,0.476632073697,0.727058839798},{0.158846606285,0.472572096072,0.724967336655},{0.158846606285,0.472572096072,0.724967336655},{0.154786629128,0.468512118447,0.722875833511},{0.154786629128,0.468512118447,0.722875833511},{0.15072665197,0.464452140822,0.720784330368},{0.15072665197,0.464452140822,0.720784330368},{0.146666674813,0.460392163197,0.718692827225},{0.146666674813,0.460392163197,0.718692827225},{0.142606697655,0.456332185572,0.716601324081},{0.142606697655,0.456332185572,0.716601324081},{0.138546720498,0.452272207947,0.714509820938},{0.138546720498,0.452272207947,0.714509820938},{0.13448674334,0.448212230322,0.712418317795},{0.13448674334,0.448212230322,0.712418317795},{0.130426766183,0.444152252697,0.710326814651},{0.130426766183,0.444152252697,0.710326814651},{0.127104966693,0.440184547971,0.707497133928},{0.127104966693,0.440184547971,0.707497133928},{0.124029226426,0.436247600878,0.704421394012},{0.124029226426,0.436247600878,0.704421394012},{0.120953486159,0.432310653785,0.701345654095},{0.120953486159,0.432310653785,0.701345654095},{0.117877745891,0.428373706691,0.698269914178},{0.117877745891,0.428373706691,0.698269914178},{0.114802005624,0.424436759598,0.695194174262},{0.111726265357,0.420499812505,0.692118434345},{0.111726265357,0.420499812505,0.692118434345},{0.10865052509,0.416562865412,0.689042694428},{0.10865052509,0.416562865412,0.689042694428},{0.105574784822,0.412625918318,0.685966954512},{0.105574784822,0.412625918318,0.685966954512},{0.102499044555,0.408688971225,0.682891214595},{0.102499044555,0.408688971225,0.682891214595},{0.0994233042878,0.404752024132,0.679815474678},{0.0994233042878,0.404752024132,0.679815474678},{0.0963475640206,0.400815077038,0.676739734762},{0.0963475640206,0.400815077038,0.676739734762},{0.0932718237533,0.396878129945,0.673663994845},{0.0932718237533,0.396878129945,0.673663994845},{0.0901960834861,0.392941182852,0.670588254929},{0.0901960834861,0.392941182852,0.670588254929},{0.0871203432188,0.389004235759,0.667512515012},{0.0871203432188,0.389004235759,0.667512515012},{0.0840446029516,0.385067288665,0.664436775095},{0.0840446029516,0.385067288665,0.664436775095},{0.0809688626843,0.381130341572,0.661361035179},{0.0809688626843,0.381130341572,0.661361035179},{0.0778931224171,0.377193394479,0.658285295262},{0.0778931224171,0.377193394479,0.658285295262},{0.0748173821498,0.373256447385,0.655209555345},{0.0748173821498,0.373256447385,0.655209555345},{0.0717416418826,0.369319500292,0.652133815429},{0.0717416418826,0.369319500292,0.652133815429},{0.0686659016153,0.365382553199,0.649058075512},{0.0686659016153,0.365382553199,0.649058075512},{0.0655901613481,0.361445606105,0.645982335596},{0.0655901613481,0.361445606105,0.645982335596},{0.0625144210808,0.357508659012,0.642906595679},{0.0625144210808,0.357508659012,0.642906595679},{0.0594386808136,0.353571711919,0.639830855762},{0.0594386808136,0.353571711919,0.639830855762},{0.0563629405463,0.349634764826,0.636755115846},{0.0563629405463,0.349634764826,0.636755115846},{0.0532872002791,0.345697817732,0.633679375929},{0.0502114600118,0.341760870639,0.630603636012},{0.0502114600118,0.341760870639,0.630603636012},{0.0471357197446,0.337823923546,0.627527896096},{0.0471357197446,0.337823923546,0.627527896096},{0.0440599794773,0.333886976452,0.624452156179},{0.0440599794773,0.333886976452,0.624452156179},{0.0409842392101,0.329950029359,0.621376416262},{0.0409842392101,0.329950029359,0.621376416262},{0.0379084989429,0.326013082266,0.618300676346},{0.0379084989429,0.326013082266,0.618300676346},{0.0348327586756,0.322076135173,0.615224936429},{0.0348327586756,0.322076135173,0.615224936429},{0.0317570184084,0.318139188079,0.612149196513},{0.0317570184084,0.318139188079,0.612149196513},{0.0313725508749,0.31409458968,0.606489834248},{0.0313725508749,0.31409458968,0.606489834248},{0.0313725508749,0.310034612522,0.600461383076},{0.0313725508749,0.310034612522,0.600461383076},{0.0313725508749,0.305974635365,0.594432931905},{0.0313725508749,0.305974635365,0.594432931905},{0.0313725508749,0.301914658208,0.588404480733},{0.0313725508749,0.301914658208,0.588404480733},{0.0313725508749,0.29785468105,0.582376029562},{0.0313725508749,0.29785468105,0.582376029562},{0.0313725508749,0.293794703893,0.57634757839},{0.0313725508749,0.293794703893,0.57634757839},{0.0313725508749,0.289734726735,0.570319127218},{0.0313725508749,0.289734726735,0.570319127218},{0.0313725508749,0.285674749578,0.564290676047},{0.0313725508749,0.285674749578,0.564290676047},{0.0313725508749,0.28161477242,0.558262224875},{0.0313725508749,0.28161477242,0.558262224875},{0.0313725508749,0.277554795263,0.552233773704},{0.0313725508749,0.277554795263,0.552233773704},{0.0313725508749,0.273494818105,0.546205322532},{0.0313725508749,0.273494818105,0.546205322532},{0.0313725508749,0.269434840948,0.540176871361},{0.0313725508749,0.269434840948,0.540176871361},{0.0313725508749,0.265374863791,0.534148420189},{0.0313725508749,0.261314886633,0.528119969017},{0.0313725508749,0.261314886633,0.528119969017},{0.0313725508749,0.257254909476,0.522091517846},{0.0313725508749,0.257254909476,0.522091517846},{0.0313725508749,0.253194932318,0.516063066674},{0.0313725508749,0.253194932318,0.516063066674},{0.0313725508749,0.249134955161,0.510034615503},{0.0313725508749,0.249134955161,0.510034615503},{0.0313725508749,0.245074978003,0.504006164331},{0.0313725508749,0.245074978003,0.504006164331},{0.0313725508749,0.241015000846,0.497977713159},{0.0313725508749,0.241015000846,0.497977713159},{0.0313725508749,0.236955023688,0.491949261988},{0.0313725508749,0.236955023688,0.491949261988},{0.0313725508749,0.232895046531,0.485920810816},{0.0313725508749,0.232895046531,0.485920810816},{0.0313725508749,0.228835069374,0.479892359645},{0.0313725508749,0.228835069374,0.479892359645},{0.0313725508749,0.224775092216,0.473863908473},{0.0313725508749,0.224775092216,0.473863908473},{0.0313725508749,0.220715115059,0.467835457302},{0.0313725508749,0.220715115059,0.467835457302},{0.0313725508749,0.216655137901,0.46180700613},{0.0313725508749,0.216655137901,0.46180700613},{0.0313725508749,0.212595160744,0.455778554958},{0.0313725508749,0.212595160744,0.455778554958},{0.0313725508749,0.208535183586,0.449750103787},{0.0313725508749,0.208535183586,0.449750103787},{0.0313725508749,0.204475206429,0.443721652615},{0.0313725508749,0.204475206429,0.443721652615},{0.0313725508749,0.200415229271,0.437693201444},{0.0313725508749,0.200415229271,0.437693201444},{0.0313725508749,0.196355252114,0.431664750272},{0.0313725508749,0.196355252114,0.431664750272},{0.0313725508749,0.192295274957,0.425636299101},{0.0313725508749,0.192295274957,0.425636299101},{0.0313725508749,0.188235297799,0.419607847929},{0.0313725508749,0.188235297799,0.419607847929},};

const std::vector<glm::vec3> CM_PIYG = 
{{0.556862771511,0.00392156885937,0.321568638086},{0.556862771511,0.00392156885937,0.321568638086},{0.565321056282,0.00792003102491,0.328181478907},{0.565321056282,0.00792003102491,0.328181478907},{0.573779341053,0.0119184931905,0.334794319728},{0.573779341053,0.0119184931905,0.334794319728},{0.582237625823,0.015916955356,0.341407160549},{0.582237625823,0.015916955356,0.341407160549},{0.590695910594,0.0199154175215,0.348020001369},{0.590695910594,0.0199154175215,0.348020001369},{0.599154195365,0.0239138796871,0.35463284219},{0.599154195365,0.0239138796871,0.35463284219},{0.607612480136,0.0279123418526,0.361245683011},{0.607612480136,0.0279123418526,0.361245683011},{0.616070764906,0.0319108040182,0.367858523832},{0.616070764906,0.0319108040182,0.367858523832},{0.624529049677,0.0359092661837,0.374471364652},{0.624529049677,0.0359092661837,0.374471364652},{0.632987334448,0.0399077283492,0.381084205473},{0.632987334448,0.0399077283492,0.381084205473},{0.641445619218,0.0439061905148,0.387697046294},{0.641445619218,0.0439061905148,0.387697046294},{0.649903903989,0.0479046526803,0.394309887115},{0.649903903989,0.0479046526803,0.394309887115},{0.65836218876,0.0519031148459,0.400922727935},{0.65836218876,0.0519031148459,0.400922727935},{0.666820473531,0.0559015770114,0.407535568756},{0.666820473531,0.0559015770114,0.407535568756},{0.675278758301,0.0599000391769,0.414148409577},{0.675278758301,0.0599000391769,0.414148409577},{0.683737043072,0.0638985013425,0.420761250398},{0.683737043072,0.0638985013425,0.420761250398},{0.692195327843,0.067896963508,0.427374091218},{0.692195327843,0.067896963508,0.427374091218},{0.700653612614,0.0718954256736,0.433986932039},{0.700653612614,0.0718954256736,0.433986932039},{0.709111897384,0.0758938878391,0.44059977286},{0.709111897384,0.0758938878391,0.44059977286},{0.717570182155,0.0798923500047,0.447212613681},{0.726028466926,0.0838908121702,0.453825454502},{0.726028466926,0.0838908121702,0.453825454502},{0.734486751697,0.0878892743357,0.460438295322},{0.734486751697,0.0878892743357,0.460438295322},{0.742945036467,0.0918877365013,0.467051136143},{0.742945036467,0.0918877365013,0.467051136143},{0.751403321238,0.0958861986668,0.473663976964},{0.751403321238,0.0958861986668,0.473663976964},{0.759861606009,0.0998846608324,0.480276817785},{0.759861606009,0.0998846608324,0.480276817785},{0.76831989078,0.103883122998,0.486889658605},{0.76831989078,0.103883122998,0.486889658605},{0.774471370613,0.112956556327,0.493963860998},{0.774471370613,0.112956556327,0.493963860998},{0.778316045509,0.12710496082,0.501499424962},{0.778316045509,0.12710496082,0.501499424962},{0.782160720404,0.141253365313,0.509034988927},{0.782160720404,0.141253365313,0.509034988927},{0.7860053953,0.155401769806,0.516570552891},{0.7860053953,0.155401769806,0.516570552891},{0.789850070196,0.169550174299,0.524106116856},{0.789850070196,0.169550174299,0.524106116856},{0.793694745092,0.183698578792,0.53164168082},{0.793694745092,0.183698578792,0.53164168082},{0.797539419988,0.197846983286,0.539177244785},{0.797539419988,0.197846983286,0.539177244785},{0.801384094883,0.211995387779,0.546712808749},{0.801384094883,0.211995387779,0.546712808749},{0.805228769779,0.226143792272,0.554248372714},{0.805228769779,0.226143792272,0.554248372714},{0.809073444675,0.240292196765,0.561783936678},{0.809073444675,0.240292196765,0.561783936678},{0.812918119571,0.254440601258,0.569319500643},{0.812918119571,0.254440601258,0.569319500643},{0.816762794467,0.268589005751,0.576855064607},{0.816762794467,0.268589005751,0.576855064607},{0.820607469362,0.282737410244,0.584390628572},{0.820607469362,0.282737410244,0.584390628572},{0.824452144258,0.296885814737,0.591926192536},{0.828296819154,0.31103421923,0.599461756501},{0.828296819154,0.31103421923,0.599461756501},{0.83214149405,0.325182623723,0.606997320465},{0.83214149405,0.325182623723,0.606997320465},{0.835986168946,0.339331028216,0.614532884429},{0.835986168946,0.339331028216,0.614532884429},{0.839830843841,0.353479432709,0.622068448394},{0.839830843841,0.353479432709,0.622068448394},{0.843675518737,0.367627837202,0.629604012358},{0.843675518737,0.367627837202,0.629604012358},{0.847520193633,0.381776241695,0.637139576323},{0.847520193633,0.381776241695,0.637139576323},{0.851364868529,0.395924646188,0.644675140287},{0.851364868529,0.395924646188,0.644675140287},{0.855209543424,0.410073050681,0.652210704252},{0.855209543424,0.410073050681,0.652210704252},{0.85905421832,0.424221455174,0.659746268216},{0.85905421832,0.424221455174,0.659746268216},{0.862898893216,0.438369859667,0.667281832181},{0.862898893216,0.438369859667,0.667281832181},{0.866743568112,0.45251826416,0.674817396145},{0.866743568112,0.45251826416,0.674817396145},{0.870588243008,0.466666668653,0.68235296011},{0.870588243008,0.466666668653,0.68235296011},{0.873510195928,0.47635525056,0.689119587926},{0.873510195928,0.47635525056,0.689119587926},{0.876432148849,0.486043832466,0.695886215743},{0.876432148849,0.486043832466,0.695886215743},{0.87935410177,0.495732414372,0.702652843559},{0.87935410177,0.495732414372,0.702652843559},{0.882276054691,0.505420996278,0.709419471376},{0.882276054691,0.505420996278,0.709419471376},{0.885198007612,0.515109578184,0.716186099193},{0.885198007612,0.515109578184,0.716186099193},{0.888119960532,0.52479816009,0.722952727009},{0.888119960532,0.52479816009,0.722952727009},{0.891041913453,0.534486741996,0.729719354826},{0.891041913453,0.534486741996,0.729719354826},{0.893963866374,0.544175323902,0.736485982642},{0.896885819295,0.553863905809,0.743252610459},{0.896885819295,0.553863905809,0.743252610459},{0.899807772216,0.563552487715,0.750019238276},{0.899807772216,0.563552487715,0.750019238276},{0.902729725136,0.573241069621,0.756785866092},{0.902729725136,0.573241069621,0.756785866092},{0.905651678057,0.582929651527,0.763552493909},{0.905651678057,0.582929651527,0.763552493909},{0.908573630978,0.592618233433,0.770319121725},{0.908573630978,0.592618233433,0.770319121725},{0.911495583899,0.602306815339,0.777085749542},{0.911495583899,0.602306815339,0.777085749542},{0.91441753682,0.611995397245,0.783852377359},{0.91441753682,0.611995397245,0.783852377359},{0.91733948974,0.621683979151,0.790619005175},{0.91733948974,0.621683979151,0.790619005175},{0.920261442661,0.631372561057,0.797385632992},{0.920261442661,0.631372561057,0.797385632992},{0.923183395582,0.641061142964,0.804152260808},{0.923183395582,0.641061142964,0.804152260808},{0.926105348503,0.65074972487,0.810918888625},{0.926105348503,0.65074972487,0.810918888625},{0.929027301424,0.660438306776,0.817685516442},{0.929027301424,0.660438306776,0.817685516442},{0.931949254344,0.670126888682,0.824452144258},{0.931949254344,0.670126888682,0.824452144258},{0.934871207265,0.679815470588,0.831218772075},{0.934871207265,0.679815470588,0.831218772075},{0.937793160186,0.689504052494,0.837985399891},{0.937793160186,0.689504052494,0.837985399891},{0.940715113107,0.6991926344,0.844752027708},{0.940715113107,0.6991926344,0.844752027708},{0.943637066028,0.708881216306,0.851518655525},{0.943637066028,0.708881216306,0.851518655525},{0.946020764463,0.716955034172,0.856516732889},{0.946020764463,0.716955034172,0.856516732889},{0.947866208413,0.723414087997,0.859746259802},{0.949711652363,0.729873141822,0.862975786714},{0.949711652363,0.729873141822,0.862975786714},{0.951557096313,0.736332195647,0.866205313626},{0.951557096313,0.736332195647,0.866205313626},{0.953402540263,0.742791249472,0.869434840539},{0.953402540263,0.742791249472,0.869434840539},{0.955247984213,0.749250303296,0.872664367451},{0.955247984213,0.749250303296,0.872664367451},{0.957093428163,0.755709357121,0.875893894364},{0.957093428163,0.755709357121,0.875893894364},{0.958938872113,0.762168410946,0.879123421276},{0.958938872113,0.762168410946,0.879123421276},{0.960784316063,0.768627464771,0.882352948189},{0.960784316063,0.768627464771,0.882352948189},{0.962629760013,0.775086518596,0.885582475101},{0.962629760013,0.775086518596,0.885582475101},{0.964475203963,0.781545572421,0.888812002014},{0.964475203963,0.781545572421,0.888812002014},{0.966320647913,0.788004626246,0.892041528926},{0.966320647913,0.788004626246,0.892041528926},{0.968166091863,0.794463680071,0.895271055839},{0.968166091863,0.794463680071,0.895271055839},{0.970011535813,0.800922733896,0.898500582751},{0.970011535813,0.800922733896,0.898500582751},{0.971856979763,0.807381787721,0.901730109664},{0.971856979763,0.807381787721,0.901730109664},{0.973702423713,0.813840841546,0.904959636576},{0.973702423713,0.813840841546,0.904959636576},{0.975547867663,0.820299895371,0.908189163489},{0.975547867663,0.820299895371,0.908189163489},{0.977393311613,0.826758949196,0.911418690401},{0.977393311613,0.826758949196,0.911418690401},{0.979238755563,0.833218003021,0.914648217313},{0.979238755563,0.833218003021,0.914648217313},{0.981084199513,0.839677056845,0.917877744226},{0.981084199513,0.839677056845,0.917877744226},{0.982929643463,0.84613611067,0.921107271138},{0.982929643463,0.84613611067,0.921107271138},{0.984775087413,0.852595164495,0.924336798051},{0.986620531363,0.85905421832,0.927566324963},{0.986620531363,0.85905421832,0.927566324963},{0.988465975313,0.865513272145,0.930795851876},{0.988465975313,0.865513272145,0.930795851876},{0.990311419263,0.87197232597,0.934025378788},{0.990311419263,0.87197232597,0.934025378788},{0.992156863213,0.878431379795,0.937254905701},{0.992156863213,0.878431379795,0.937254905701},{0.991234141238,0.881968480699,0.938485201667},{0.991234141238,0.881968480699,0.938485201667},{0.990311419263,0.885505581603,0.939715497634},{0.990311419263,0.885505581603,0.939715497634},{0.989388697288,0.889042682507,0.940945793601},{0.989388697288,0.889042682507,0.940945793601},{0.988465975313,0.892579783412,0.942176089567},{0.988465975313,0.892579783412,0.942176089567},{0.987543253338,0.896116884316,0.943406385534},{0.987543253338,0.896116884316,0.943406385534},{0.986620531363,0.89965398522,0.944636681501},{0.986620531363,0.89965398522,0.944636681501},{0.985697809388,0.903191086124,0.945866977467},{0.985697809388,0.903191086124,0.945866977467},{0.984775087413,0.906728187028,0.947097273434},{0.984775087413,0.906728187028,0.947097273434},{0.983852365438,0.910265287932,0.948327569401},{0.983852365438,0.910265287932,0.948327569401},{0.982929643463,0.913802388836,0.949557865367},{0.982929643463,0.913802388836,0.949557865367},{0.982006921488,0.91733948974,0.950788161334},{0.982006921488,0.91733948974,0.950788161334},{0.981084199513,0.920876590645,0.952018457301},{0.981084199513,0.920876590645,0.952018457301},{0.980161477538,0.924413691549,0.953248753267},{0.980161477538,0.924413691549,0.953248753267},{0.979238755563,0.927950792453,0.954479049234},{0.979238755563,0.927950792453,0.954479049234},{0.978316033588,0.931487893357,0.9557093452},{0.978316033588,0.931487893357,0.9557093452},{0.977393311613,0.935024994261,0.956939641167},{0.976470589638,0.938562095165,0.958169937134},{0.976470589638,0.938562095165,0.958169937134},{0.975547867663,0.942099196069,0.9594002331},{0.975547867663,0.942099196069,0.9594002331},{0.974625145688,0.945636296974,0.960630529067},{0.974625145688,0.945636296974,0.960630529067},{0.973702423713,0.949173397878,0.961860825034},{0.973702423713,0.949173397878,0.961860825034},{0.972779701738,0.952710498782,0.963091121},{0.972779701738,0.952710498782,0.963091121},{0.971856979763,0.956247599686,0.964321416967},{0.971856979763,0.956247599686,0.964321416967},{0.970934257788,0.95978470059,0.965551712934},{0.970934257788,0.95978470059,0.965551712934},{0.970011535813,0.963321801494,0.9667820089},{0.970011535813,0.963321801494,0.9667820089},{0.969088813838,0.966858902398,0.968012304867},{0.969088813838,0.966858902398,0.968012304867},{0.967320263386,0.968473665855,0.965628606432},{0.967320263386,0.968473665855,0.965628606432},{0.964705884457,0.968166091863,0.959630913594},{0.964705884457,0.968166091863,0.959630913594},{0.962091505527,0.967858517871,0.953633220757},{0.962091505527,0.967858517871,0.953633220757},{0.959477126598,0.96755094388,0.947635527919},{0.959477126598,0.96755094388,0.947635527919},{0.956862747669,0.967243369888,0.941637835082},{0.956862747669,0.967243369888,0.941637835082},{0.95424836874,0.966935795896,0.935640142244},{0.95424836874,0.966935795896,0.935640142244},{0.951633989811,0.966628221905,0.929642449407},{0.951633989811,0.966628221905,0.929642449407},{0.949019610882,0.966320647913,0.92364475657},{0.949019610882,0.966320647913,0.92364475657},{0.946405231953,0.966013073921,0.917647063732},{0.946405231953,0.966013073921,0.917647063732},{0.943790853024,0.96570549993,0.911649370895},{0.941176474094,0.965397925938,0.905651678057},{0.941176474094,0.965397925938,0.905651678057},{0.938562095165,0.965090351946,0.89965398522},{0.938562095165,0.965090351946,0.89965398522},{0.935947716236,0.964782777955,0.893656292382},{0.935947716236,0.964782777955,0.893656292382},{0.933333337307,0.964475203963,0.887658599545},{0.933333337307,0.964475203963,0.887658599545},{0.930718958378,0.964167629971,0.881660906708},{0.930718958378,0.964167629971,0.881660906708},{0.928104579449,0.96386005598,0.87566321387},{0.928104579449,0.96386005598,0.87566321387},{0.92549020052,0.963552481988,0.869665521033},{0.92549020052,0.963552481988,0.869665521033},{0.92287582159,0.963244907996,0.863667828195},{0.92287582159,0.963244907996,0.863667828195},{0.920261442661,0.962937334005,0.857670135358},{0.920261442661,0.962937334005,0.857670135358},{0.917647063732,0.962629760013,0.85167244252},{0.917647063732,0.962629760013,0.85167244252},{0.915032684803,0.962322186021,0.845674749683},{0.915032684803,0.962322186021,0.845674749683},{0.912418305874,0.96201461203,0.839677056845},{0.912418305874,0.96201461203,0.839677056845},{0.909803926945,0.961707038038,0.833679364008},{0.909803926945,0.961707038038,0.833679364008},{0.907189548016,0.961399464046,0.827681671171},{0.907189548016,0.961399464046,0.827681671171},{0.904575169086,0.961091890055,0.821683978333},{0.904575169086,0.961091890055,0.821683978333},{0.901960790157,0.960784316063,0.815686285496},{0.901960790157,0.960784316063,0.815686285496},{0.894886588349,0.957708576146,0.804306047804},{0.894886588349,0.957708576146,0.804306047804},{0.887812386541,0.95463283623,0.792925810113},{0.887812386541,0.95463283623,0.792925810113},{0.880738184733,0.951557096313,0.781545572421},{0.880738184733,0.951557096313,0.781545572421},{0.873663982924,0.948481356396,0.77016533473},{0.866589781116,0.94540561648,0.758785097038},{0.866589781116,0.94540561648,0.758785097038},{0.859515579308,0.942329876563,0.747404859347},{0.859515579308,0.942329876563,0.747404859347},{0.8524413775,0.939254136646,0.736024621655},{0.8524413775,0.939254136646,0.736024621655},{0.845367175691,0.93617839673,0.724644383963},{0.845367175691,0.93617839673,0.724644383963},{0.838292973883,0.933102656813,0.713264146272},{0.838292973883,0.933102656813,0.713264146272},{0.831218772075,0.930026916897,0.70188390858},{0.831218772075,0.930026916897,0.70188390858},{0.824144570266,0.92695117698,0.690503670889},{0.824144570266,0.92695117698,0.690503670889},{0.817070368458,0.923875437063,0.679123433197},{0.817070368458,0.923875437063,0.679123433197},{0.80999616665,0.920799697147,0.667743195506},{0.80999616665,0.920799697147,0.667743195506},{0.802921964842,0.91772395723,0.656362957814},{0.802921964842,0.91772395723,0.656362957814},{0.795847763033,0.914648217313,0.644982720123},{0.795847763033,0.914648217313,0.644982720123},{0.788773561225,0.911572477397,0.633602482431},{0.788773561225,0.911572477397,0.633602482431},{0.781699359417,0.90849673748,0.62222224474},{0.781699359417,0.90849673748,0.62222224474},{0.774625157609,0.905420997564,0.610842007048},{0.774625157609,0.905420997564,0.610842007048},{0.7675509558,0.902345257647,0.599461769356},{0.7675509558,0.902345257647,0.599461769356},{0.760476753992,0.89926951773,0.588081531665},{0.760476753992,0.89926951773,0.588081531665},{0.753402552184,0.896193777814,0.576701293973},{0.753402552184,0.896193777814,0.576701293973},{0.746328350376,0.893118037897,0.565321056282},{0.746328350376,0.893118037897,0.565321056282},{0.739254148567,0.89004229798,0.55394081859},{0.739254148567,0.89004229798,0.55394081859},{0.732179946759,0.886966558064,0.542560580899},{0.725105744951,0.883890818147,0.531180343207},{0.725105744951,0.883890818147,0.531180343207},{0.717185714081,0.879507888766,0.520184572421},{0.717185714081,0.879507888766,0.520184572421},{0.70841985415,0.87381776992,0.50957326854},{0.70841985415,0.87381776992,0.50957326854},{0.699653994219,0.868127651074,0.498961964659},{0.699653994219,0.868127651074,0.498961964659},{0.690888134288,0.862437532229,0.488350660778},{0.690888134288,0.862437532229,0.488350660778},{0.682122274357,0.856747413383,0.477739356896},{0.682122274357,0.856747413383,0.477739356896},{0.673356414426,0.851057294537,0.467128053015},{0.673356414426,0.851057294537,0.467128053015},{0.664590554494,0.845367175691,0.456516749134},{0.664590554494,0.845367175691,0.456516749134},{0.655824694563,0.839677056845,0.445905445253},{0.655824694563,0.839677056845,0.445905445253},{0.647058834632,0.833986938,0.435294141372},{0.647058834632,0.833986938,0.435294141372},{0.638292974701,0.828296819154,0.424682837491},{0.638292974701,0.828296819154,0.424682837491},{0.62952711477,0.822606700308,0.41407153361},{0.62952711477,0.822606700308,0.41407153361},{0.620761254839,0.816916581462,0.403460229729},{0.620761254839,0.816916581462,0.403460229729},{0.611995394908,0.811226462617,0.392848925848},{0.611995394908,0.811226462617,0.392848925848},{0.603229534977,0.805536343771,0.382237621967},{0.603229534977,0.805536343771,0.382237621967},{0.594463675045,0.799846224925,0.371626318085},{0.594463675045,0.799846224925,0.371626318085},{0.585697815114,0.794156106079,0.361015014204},{0.585697815114,0.794156106079,0.361015014204},{0.576931955183,0.788465987234,0.350403710323},{0.576931955183,0.788465987234,0.350403710323},{0.568166095252,0.782775868388,0.339792406442},{0.559400235321,0.777085749542,0.329181102561},{0.559400235321,0.777085749542,0.329181102561},{0.55063437539,0.771395630696,0.31856979868},{0.55063437539,0.771395630696,0.31856979868},{0.541868515459,0.76570551185,0.307958494799},{0.541868515459,0.76570551185,0.307958494799},{0.533102655528,0.760015393005,0.297347190918},{0.533102655528,0.760015393005,0.297347190918},{0.524336795597,0.754325274159,0.286735887037},{0.524336795597,0.754325274159,0.286735887037},{0.515570935665,0.748635155313,0.276124583156},{0.515570935665,0.748635155313,0.276124583156},{0.506805075734,0.742945036467,0.265513279274},{0.506805075734,0.742945036467,0.265513279274},{0.498039215803,0.737254917622,0.254901975393},{0.498039215803,0.737254917622,0.254901975393},{0.490349866012,0.730795863797,0.249980790942},{0.490349866012,0.730795863797,0.249980790942},{0.48266051622,0.724336809972,0.245059606491},{0.48266051622,0.724336809972,0.245059606491},{0.474971166428,0.717877756147,0.24013842204},{0.474971166428,0.717877756147,0.24013842204},{0.467281816637,0.711418702322,0.235217237589},{0.467281816637,0.711418702322,0.235217237589},{0.459592466845,0.704959648497,0.230296053138},{0.459592466845,0.704959648497,0.230296053138},{0.451903117054,0.698500594672,0.225374868687},{0.451903117054,0.698500594672,0.225374868687},{0.444213767262,0.692041540847,0.220453684236},{0.444213767262,0.692041540847,0.220453684236},{0.43652441747,0.685582487022,0.215532499786},{0.43652441747,0.685582487022,0.215532499786},{0.428835067679,0.679123433197,0.210611315335},{0.428835067679,0.679123433197,0.210611315335},{0.421145717887,0.672664379372,0.205690130884},{0.421145717887,0.672664379372,0.205690130884},{0.413456368096,0.666205325547,0.200768946433},{0.413456368096,0.666205325547,0.200768946433},{0.405767018304,0.659746271722,0.195847761982},{0.398077668513,0.653287217898,0.190926577531},{0.398077668513,0.653287217898,0.190926577531},{0.390388318721,0.646828164073,0.18600539308},{0.390388318721,0.646828164073,0.18600539308},{0.382698968929,0.640369110248,0.181084208629},{0.382698968929,0.640369110248,0.181084208629},{0.375009619138,0.633910056423,0.176163024178},{0.375009619138,0.633910056423,0.176163024178},{0.367320269346,0.627451002598,0.171241839727},{0.367320269346,0.627451002598,0.171241839727},{0.359630919555,0.620991948773,0.166320655276},{0.359630919555,0.620991948773,0.166320655276},{0.351941569763,0.614532894948,0.161399470825},{0.351941569763,0.614532894948,0.161399470825},{0.344252219971,0.608073841123,0.156478286374},{0.344252219971,0.608073841123,0.156478286374},{0.33656287018,0.601614787298,0.151557101923},{0.33656287018,0.601614787298,0.151557101923},{0.328873520388,0.595155733473,0.146635917472},{0.328873520388,0.595155733473,0.146635917472},{0.321184170597,0.588696679648,0.141714733021},{0.321184170597,0.588696679648,0.141714733021},{0.313494820805,0.582237625823,0.13679354857},{0.313494820805,0.582237625823,0.13679354857},{0.305805471014,0.575778571998,0.131872364119},{0.305805471014,0.575778571998,0.131872364119},{0.299038842905,0.569011943597,0.128796623764},{0.299038842905,0.569011943597,0.128796623764},{0.293194936479,0.56193774062,0.127566327505},{0.293194936479,0.56193774062,0.127566327505},{0.287351030053,0.554863537643,0.126336031246},{0.287351030053,0.554863537643,0.126336031246},{0.281507123627,0.547789334666,0.125105734988},{0.281507123627,0.547789334666,0.125105734988},{0.275663217201,0.54071513169,0.123875438729},{0.275663217201,0.54071513169,0.123875438729},{0.269819310775,0.533640928713,0.12264514247},{0.269819310775,0.533640928713,0.12264514247},{0.263975404349,0.526566725736,0.121414846211},{0.258131497923,0.519492522759,0.120184549952},{0.258131497923,0.519492522759,0.120184549952},{0.252287591497,0.512418319782,0.118954253693},{0.252287591497,0.512418319782,0.118954253693},{0.246443685071,0.505344116805,0.117723957435},{0.246443685071,0.505344116805,0.117723957435},{0.240599778645,0.498269913828,0.116493661176},{0.240599778645,0.498269913828,0.116493661176},{0.234755872219,0.491195710851,0.115263364917},{0.234755872219,0.491195710851,0.115263364917},{0.228911965793,0.484121507874,0.114033068658},{0.228911965793,0.484121507874,0.114033068658},{0.223068059367,0.477047304897,0.112802772399},{0.223068059367,0.477047304897,0.112802772399},{0.217224152941,0.46997310192,0.11157247614},{0.217224152941,0.46997310192,0.11157247614},{0.211380246515,0.462898898943,0.110342179882},{0.211380246515,0.462898898943,0.110342179882},{0.205536340089,0.455824695966,0.109111883623},{0.205536340089,0.455824695966,0.109111883623},{0.199692433663,0.448750492989,0.107881587364},{0.199692433663,0.448750492989,0.107881587364},{0.193848527237,0.441676290012,0.106651291105},{0.193848527237,0.441676290012,0.106651291105},{0.188004620812,0.434602087035,0.105420994846},{0.188004620812,0.434602087035,0.105420994846},{0.182160714386,0.427527884058,0.104190698587},{0.182160714386,0.427527884058,0.104190698587},{0.17631680796,0.420453681081,0.102960402329},{0.17631680796,0.420453681081,0.102960402329},{0.170472901534,0.413379478104,0.10173010607},{0.170472901534,0.413379478104,0.10173010607},{0.164628995108,0.406305275127,0.100499809811},{0.164628995108,0.406305275127,0.100499809811},{0.158785088682,0.39923107215,0.0992695135521},{0.158785088682,0.39923107215,0.0992695135521},{0.152941182256,0.392156869173,0.0980392172933},{0.152941182256,0.392156869173,0.0980392172933},};

const std::vector<glm::vec3> CM_SPECTRAL =
{{0.61960786581,0.00392156885937,0.258823543787},{0.61960786581,0.00392156885937,0.258823543787},{0.628066150581,0.0133025761712,0.260822774733},{0.628066150581,0.0133025761712,0.260822774733},{0.636524435352,0.022683583483,0.262822005679},{0.636524435352,0.022683583483,0.262822005679},{0.644982720123,0.0320645907949,0.264821236624},{0.644982720123,0.0320645907949,0.264821236624},{0.653441004893,0.0414455981067,0.26682046757},{0.653441004893,0.0414455981067,0.26682046757},{0.661899289664,0.0508266054185,0.268819698516},{0.661899289664,0.0508266054185,0.268819698516},{0.670357574435,0.0602076127303,0.270818929462},{0.670357574435,0.0602076127303,0.270818929462},{0.678815859206,0.0695886200422,0.272818160408},{0.678815859206,0.0695886200422,0.272818160408},{0.687274143976,0.078969627354,0.274817391353},{0.687274143976,0.078969627354,0.274817391353},{0.695732428747,0.0883506346658,0.276816622299},{0.695732428747,0.0883506346658,0.276816622299},{0.704190713518,0.0977316419777,0.278815853245},{0.704190713518,0.0977316419777,0.278815853245},{0.712648998289,0.107112649289,0.280815084191},{0.712648998289,0.107112649289,0.280815084191},{0.721107283059,0.116493656601,0.282814315137},{0.721107283059,0.116493656601,0.282814315137},{0.72956556783,0.125874663913,0.284813546083},{0.72956556783,0.125874663913,0.284813546083},{0.738023852601,0.135255671225,0.286812777028},{0.738023852601,0.135255671225,0.286812777028},{0.746482137372,0.144636678537,0.288812007974},{0.746482137372,0.144636678537,0.288812007974},{0.754940422142,0.154017685849,0.29081123892},{0.754940422142,0.154017685849,0.29081123892},{0.763398706913,0.16339869316,0.292810469866},{0.763398706913,0.16339869316,0.292810469866},{0.771856991684,0.172779700472,0.294809700812},{0.771856991684,0.172779700472,0.294809700812},{0.780315276454,0.182160707784,0.296808931757},{0.788773561225,0.191541715096,0.298808162703},{0.788773561225,0.191541715096,0.298808162703},{0.797231845996,0.200922722408,0.300807393649},{0.797231845996,0.200922722408,0.300807393649},{0.805690130767,0.21030372972,0.302806624595},{0.805690130767,0.21030372972,0.302806624595},{0.814148415537,0.219684737031,0.304805855541},{0.814148415537,0.219684737031,0.304805855541},{0.822606700308,0.229065744343,0.306805086486},{0.822606700308,0.229065744343,0.306805086486},{0.831064985079,0.238446751655,0.308804317432},{0.831064985079,0.238446751655,0.308804317432},{0.8376778259,0.246751250005,0.30888121093},{0.8376778259,0.246751250005,0.30888121093},{0.84244522277,0.253979239394,0.30703576698},{0.84244522277,0.253979239394,0.30703576698},{0.847212619641,0.261207228782,0.30519032303},{0.847212619641,0.261207228782,0.30519032303},{0.851980016512,0.268435218171,0.30334487908},{0.851980016512,0.268435218171,0.30334487908},{0.856747413383,0.275663207559,0.30149943513},{0.856747413383,0.275663207559,0.30149943513},{0.861514810254,0.282891196947,0.29965399118},{0.861514810254,0.282891196947,0.29965399118},{0.866282207124,0.290119186336,0.29780854723},{0.866282207124,0.290119186336,0.29780854723},{0.871049603995,0.297347175724,0.29596310328},{0.871049603995,0.297347175724,0.29596310328},{0.875817000866,0.304575165113,0.29411765933},{0.875817000866,0.304575165113,0.29411765933},{0.880584397737,0.311803154501,0.29227221538},{0.880584397737,0.311803154501,0.29227221538},{0.885351794607,0.31903114389,0.29042677143},{0.885351794607,0.31903114389,0.29042677143},{0.890119191478,0.326259133278,0.28858132748},{0.890119191478,0.326259133278,0.28858132748},{0.894886588349,0.333487122667,0.28673588353},{0.894886588349,0.333487122667,0.28673588353},{0.89965398522,0.340715112055,0.28489043958},{0.904421382091,0.347943101443,0.28304499563},{0.904421382091,0.347943101443,0.28304499563},{0.909188778961,0.355171090832,0.281199551681},{0.909188778961,0.355171090832,0.281199551681},{0.913956175832,0.36239908022,0.279354107731},{0.913956175832,0.36239908022,0.279354107731},{0.918723572703,0.369627069609,0.277508663781},{0.918723572703,0.369627069609,0.277508663781},{0.923490969574,0.376855058997,0.275663219831},{0.923490969574,0.376855058997,0.275663219831},{0.928258366445,0.384083048386,0.273817775881},{0.928258366445,0.384083048386,0.273817775881},{0.933025763315,0.391311037774,0.271972331931},{0.933025763315,0.391311037774,0.271972331931},{0.937793160186,0.398539027163,0.270126887981},{0.937793160186,0.398539027163,0.270126887981},{0.942560557057,0.405767016551,0.268281444031},{0.942560557057,0.405767016551,0.268281444031},{0.947327953928,0.41299500594,0.266436000081},{0.947327953928,0.41299500594,0.266436000081},{0.952095350798,0.420222995328,0.264590556131},{0.952095350798,0.420222995328,0.264590556131},{0.956862747669,0.427450984716,0.262745112181},{0.956862747669,0.427450984716,0.262745112181},{0.958246830632,0.437447140614,0.267358722056},{0.958246830632,0.437447140614,0.267358722056},{0.959630913594,0.447443296512,0.271972331931},{0.959630913594,0.447443296512,0.271972331931},{0.961014996557,0.45743945241,0.276585941806},{0.961014996557,0.45743945241,0.276585941806},{0.962399079519,0.467435608308,0.281199551681},{0.962399079519,0.467435608308,0.281199551681},{0.963783162482,0.477431764205,0.285813161555},{0.963783162482,0.477431764205,0.285813161555},{0.965167245444,0.487427920103,0.29042677143},{0.965167245444,0.487427920103,0.29042677143},{0.966551328407,0.497424076001,0.295040381305},{0.966551328407,0.497424076001,0.295040381305},{0.967935411369,0.507420231899,0.29965399118},{0.969319494332,0.517416387796,0.304267601055},{0.969319494332,0.517416387796,0.304267601055},{0.970703577294,0.527412543694,0.30888121093},{0.970703577294,0.527412543694,0.30888121093},{0.972087660257,0.537408699592,0.313494820805},{0.972087660257,0.537408699592,0.313494820805},{0.973471743219,0.54740485549,0.31810843068},{0.973471743219,0.54740485549,0.31810843068},{0.974855826182,0.557401011388,0.322722040555},{0.974855826182,0.557401011388,0.322722040555},{0.976239909144,0.567397167285,0.32733565043},{0.976239909144,0.567397167285,0.32733565043},{0.977623992106,0.577393323183,0.331949260305},{0.977623992106,0.577393323183,0.331949260305},{0.979008075069,0.587389479081,0.33656287018},{0.979008075069,0.587389479081,0.33656287018},{0.980392158031,0.597385634979,0.341176480055},{0.980392158031,0.597385634979,0.341176480055},{0.981776240994,0.607381790876,0.34579008993},{0.981776240994,0.607381790876,0.34579008993},{0.983160323956,0.617377946774,0.350403699805},{0.983160323956,0.617377946774,0.350403699805},{0.984544406919,0.627374102672,0.35501730968},{0.984544406919,0.627374102672,0.35501730968},{0.985928489881,0.63737025857,0.359630919555},{0.985928489881,0.63737025857,0.359630919555},{0.987312572844,0.647366414467,0.36424452943},{0.987312572844,0.647366414467,0.36424452943},{0.988696655806,0.657362570365,0.368858139305},{0.988696655806,0.657362570365,0.368858139305},{0.990080738769,0.667358726263,0.37347174918},{0.990080738769,0.667358726263,0.37347174918},{0.991464821731,0.677354882161,0.378085359054},{0.991464821731,0.677354882161,0.378085359054},{0.992233756711,0.686197635006,0.383621691489},{0.992233756711,0.686197635006,0.383621691489},{0.992387543706,0.693886984797,0.390080746482},{0.992541330702,0.701576334589,0.396539801476},{0.992541330702,0.701576334589,0.396539801476},{0.992695117698,0.70926568438,0.40299885647},{0.992695117698,0.70926568438,0.40299885647},{0.992848904694,0.716955034172,0.409457911463},{0.992848904694,0.716955034172,0.409457911463},{0.99300269169,0.724644383963,0.415916966457},{0.99300269169,0.724644383963,0.415916966457},{0.993156478685,0.732333733755,0.422376021451},{0.993156478685,0.732333733755,0.422376021451},{0.993310265681,0.740023083547,0.428835076444},{0.993310265681,0.740023083547,0.428835076444},{0.993464052677,0.747712433338,0.435294131438},{0.993464052677,0.747712433338,0.435294131438},{0.993617839673,0.75540178313,0.441753186432},{0.993617839673,0.75540178313,0.441753186432},{0.993771626669,0.763091132921,0.448212241425},{0.993771626669,0.763091132921,0.448212241425},{0.993925413665,0.770780482713,0.454671296419},{0.993925413665,0.770780482713,0.454671296419},{0.99407920066,0.778469832504,0.461130351413},{0.99407920066,0.778469832504,0.461130351413},{0.994232987656,0.786159182296,0.467589406406},{0.994232987656,0.786159182296,0.467589406406},{0.994386774652,0.793848532088,0.4740484614},{0.994386774652,0.793848532088,0.4740484614},{0.994540561648,0.801537881879,0.480507516393},{0.994540561648,0.801537881879,0.480507516393},{0.994694348644,0.809227231671,0.486966571387},{0.994694348644,0.809227231671,0.486966571387},{0.99484813564,0.816916581462,0.493425626381},{0.99484813564,0.816916581462,0.493425626381},{0.995001922635,0.824605931254,0.499884681374},{0.995001922635,0.824605931254,0.499884681374},{0.995155709631,0.832295281046,0.506343736368},{0.995155709631,0.832295281046,0.506343736368},{0.995309496627,0.839984630837,0.512802791362},{0.995309496627,0.839984630837,0.512802791362},{0.995463283623,0.847673980629,0.519261846355},{0.995617070619,0.85536333042,0.525720901349},{0.995617070619,0.85536333042,0.525720901349},{0.995770857615,0.863052680212,0.532179956343},{0.995770857615,0.863052680212,0.532179956343},{0.99592464461,0.870742030003,0.538639011336},{0.99592464461,0.870742030003,0.538639011336},{0.996078431606,0.878431379795,0.54509806633},{0.996078431606,0.878431379795,0.54509806633},{0.996232218602,0.883198776666,0.553094990113},{0.996232218602,0.883198776666,0.553094990113},{0.996386005598,0.887966173537,0.561091913896},{0.996386005598,0.887966173537,0.561091913896},{0.996539792594,0.892733570407,0.56908883768},{0.996539792594,0.892733570407,0.56908883768},{0.99669357959,0.897500967278,0.577085761463},{0.99669357959,0.897500967278,0.577085761463},{0.996847366585,0.902268364149,0.585082685246},{0.996847366585,0.902268364149,0.585082685246},{0.997001153581,0.90703576102,0.593079609029},{0.997001153581,0.90703576102,0.593079609029},{0.997154940577,0.911803157891,0.601076532813},{0.997154940577,0.911803157891,0.601076532813},{0.997308727573,0.916570554761,0.609073456596},{0.997308727573,0.916570554761,0.609073456596},{0.997462514569,0.921337951632,0.617070380379},{0.997462514569,0.921337951632,0.617070380379},{0.997616301565,0.926105348503,0.625067304162},{0.997616301565,0.926105348503,0.625067304162},{0.99777008856,0.930872745374,0.633064227946},{0.99777008856,0.930872745374,0.633064227946},{0.997923875556,0.935640142244,0.641061151729},{0.997923875556,0.935640142244,0.641061151729},{0.998077662552,0.940407539115,0.649058075512},{0.998077662552,0.940407539115,0.649058075512},{0.998231449548,0.945174935986,0.657054999295},{0.998231449548,0.945174935986,0.657054999295},{0.998385236544,0.949942332857,0.665051923079},{0.998385236544,0.949942332857,0.665051923079},{0.99853902354,0.954709729728,0.673048846862},{0.998692810535,0.959477126598,0.681045770645},{0.998692810535,0.959477126598,0.681045770645},{0.998846597531,0.964244523469,0.689042694428},{0.998846597531,0.964244523469,0.689042694428},{0.999000384527,0.96901192034,0.697039618212},{0.999000384527,0.96901192034,0.697039618212},{0.999154171523,0.973779317211,0.705036541995},{0.999154171523,0.973779317211,0.705036541995},{0.999307958519,0.978546714081,0.713033465778},{0.999307958519,0.978546714081,0.713033465778},{0.999461745515,0.983314110952,0.721030389561},{0.999461745515,0.983314110952,0.721030389561},{0.99961553251,0.988081507823,0.729027313345},{0.99961553251,0.988081507823,0.729027313345},{0.999769319506,0.992848904694,0.737024237128},{0.999769319506,0.992848904694,0.737024237128},{0.999923106502,0.997616301565,0.745021160911},{0.999923106502,0.997616301565,0.745021160911},{0.998077662552,0.999231065021,0.746020776384},{0.998077662552,0.999231065021,0.746020776384},{0.994232987656,0.997693195063,0.740023083547},{0.994232987656,0.997693195063,0.740023083547},{0.990388312761,0.996155325104,0.734025390709},{0.990388312761,0.996155325104,0.734025390709},{0.986543637865,0.994617455146,0.728027697872},{0.986543637865,0.994617455146,0.728027697872},{0.982698962969,0.993079585188,0.722030005034},{0.982698962969,0.993079585188,0.722030005034},{0.978854288073,0.991541715229,0.716032312197},{0.978854288073,0.991541715229,0.716032312197},{0.975009613177,0.990003845271,0.710034619359},{0.975009613177,0.990003845271,0.710034619359},{0.971164938282,0.988465975313,0.704036926522},{0.971164938282,0.988465975313,0.704036926522},{0.967320263386,0.986928105354,0.698039233685},{0.967320263386,0.986928105354,0.698039233685},{0.96347558849,0.985390235396,0.692041540847},{0.959630913594,0.983852365438,0.68604384801},{0.959630913594,0.983852365438,0.68604384801},{0.955786238698,0.982314495479,0.680046155172},{0.955786238698,0.982314495479,0.680046155172},{0.951941563803,0.980776625521,0.674048462335},{0.951941563803,0.980776625521,0.674048462335},{0.948096888907,0.979238755563,0.668050769497},{0.948096888907,0.979238755563,0.668050769497},{0.944252214011,0.977700885604,0.66205307666},{0.944252214011,0.977700885604,0.66205307666},{0.940407539115,0.976163015646,0.656055383822},{0.940407539115,0.976163015646,0.656055383822},{0.936562864219,0.974625145688,0.650057690985},{0.936562864219,0.974625145688,0.650057690985},{0.932718189324,0.973087275729,0.644059998148},{0.932718189324,0.973087275729,0.644059998148},{0.928873514428,0.971549405771,0.63806230531},{0.928873514428,0.971549405771,0.63806230531},{0.925028839532,0.970011535813,0.632064612473},{0.925028839532,0.970011535813,0.632064612473},{0.921184164636,0.968473665855,0.626066919635},{0.921184164636,0.968473665855,0.626066919635},{0.91733948974,0.966935795896,0.620069226798},{0.91733948974,0.966935795896,0.620069226798},{0.913494814845,0.965397925938,0.61407153396},{0.913494814845,0.965397925938,0.61407153396},{0.909650139949,0.96386005598,0.608073841123},{0.909650139949,0.96386005598,0.608073841123},{0.905805465053,0.962322186021,0.602076148286},{0.905805465053,0.962322186021,0.602076148286},{0.901960790157,0.960784316063,0.596078455448},{0.901960790157,0.960784316063,0.596078455448},{0.892887357403,0.957093428163,0.597923899398},{0.892887357403,0.957093428163,0.597923899398},{0.883813924649,0.953402540263,0.599769343348},{0.883813924649,0.953402540263,0.599769343348},{0.874740491895,0.949711652363,0.601614787298},{0.874740491895,0.949711652363,0.601614787298},{0.865667059141,0.946020764463,0.603460231248},{0.856593626387,0.942329876563,0.605305675198},{0.856593626387,0.942329876563,0.605305675198},{0.847520193633,0.938638988663,0.607151119148},{0.847520193633,0.938638988663,0.607151119148},{0.838446760879,0.934948100763,0.608996563098},{0.838446760879,0.934948100763,0.608996563098},{0.829373328125,0.931257212863,0.610842007048},{0.829373328125,0.931257212863,0.610842007048},{0.820299895371,0.927566324963,0.612687450998},{0.820299895371,0.927566324963,0.612687450998},{0.811226462617,0.923875437063,0.614532894948},{0.811226462617,0.923875437063,0.614532894948},{0.802153029863,0.920184549163,0.616378338898},{0.802153029863,0.920184549163,0.616378338898},{0.793079597109,0.916493661263,0.618223782848},{0.793079597109,0.916493661263,0.618223782848},{0.784006164354,0.912802773363,0.620069226798},{0.784006164354,0.912802773363,0.620069226798},{0.7749327316,0.909111885463,0.621914670748},{0.7749327316,0.909111885463,0.621914670748},{0.765859298846,0.905420997564,0.623760114698},{0.765859298846,0.905420997564,0.623760114698},{0.756785866092,0.901730109664,0.625605558648},{0.756785866092,0.901730109664,0.625605558648},{0.747712433338,0.898039221764,0.627451002598},{0.747712433338,0.898039221764,0.627451002598},{0.738639000584,0.894348333864,0.629296446548},{0.738639000584,0.894348333864,0.629296446548},{0.72956556783,0.890657445964,0.631141890498},{0.72956556783,0.890657445964,0.631141890498},{0.720492135076,0.886966558064,0.632987334448},{0.720492135076,0.886966558064,0.632987334448},{0.711418702322,0.883275670164,0.634832778398},{0.711418702322,0.883275670164,0.634832778398},{0.702345269568,0.879584782264,0.636678222348},{0.702345269568,0.879584782264,0.636678222348},{0.693271836814,0.875893894364,0.638523666298},{0.693271836814,0.875893894364,0.638523666298},{0.68419840406,0.872203006464,0.640369110248},{0.675124971306,0.868512118564,0.642214554198},{0.675124971306,0.868512118564,0.642214554198},{0.665282602988,0.86459055017,0.643214169671},{0.665282602988,0.86459055017,0.643214169671},{0.654671299107,0.860438301283,0.643367956666},{0.654671299107,0.860438301283,0.643367956666},{0.644059995226,0.856286052395,0.643521743662},{0.644059995226,0.856286052395,0.643521743662},{0.633448691345,0.852133803508,0.643675530658},{0.633448691345,0.852133803508,0.643675530658},{0.622837387464,0.84798155462,0.643829317654},{0.622837387464,0.84798155462,0.643829317654},{0.612226083583,0.843829305733,0.64398310465},{0.612226083583,0.843829305733,0.64398310465},{0.601614779701,0.839677056845,0.644136891646},{0.601614779701,0.839677056845,0.644136891646},{0.59100347582,0.835524807958,0.644290678641},{0.59100347582,0.835524807958,0.644290678641},{0.580392171939,0.831372559071,0.644444465637},{0.580392171939,0.831372559071,0.644444465637},{0.569780868058,0.827220310183,0.644598252633},{0.569780868058,0.827220310183,0.644598252633},{0.559169564177,0.823068061296,0.644752039629},{0.559169564177,0.823068061296,0.644752039629},{0.548558260296,0.818915812408,0.644905826625},{0.548558260296,0.818915812408,0.644905826625},{0.537946956415,0.814763563521,0.645059613621},{0.537946956415,0.814763563521,0.645059613621},{0.527335652534,0.810611314633,0.645213400616},{0.527335652534,0.810611314633,0.645213400616},{0.516724348653,0.806459065746,0.645367187612},{0.516724348653,0.806459065746,0.645367187612},{0.506113044771,0.802306816858,0.645520974608},{0.506113044771,0.802306816858,0.645520974608},{0.49550174089,0.798154567971,0.645674761604},{0.49550174089,0.798154567971,0.645674761604},{0.484890437009,0.794002319083,0.6458285486},{0.474279133128,0.789850070196,0.645982335596},{0.474279133128,0.789850070196,0.645982335596},{0.463667829247,0.785697821309,0.646136122591},{0.463667829247,0.785697821309,0.646136122591},{0.453056525366,0.781545572421,0.646289909587},{0.453056525366,0.781545572421,0.646289909587},{0.442445221485,0.777393323534,0.646443696583},{0.442445221485,0.777393323534,0.646443696583},{0.431833917604,0.773241074646,0.646597483579},{0.431833917604,0.773241074646,0.646597483579},{0.421222613723,0.769088825759,0.646751270575},{0.421222613723,0.769088825759,0.646751270575},{0.410611309842,0.764936576871,0.646905057571},{0.410611309842,0.764936576871,0.646905057571},{0.40000000596,0.760784327984,0.647058844566},{0.40000000596,0.760784327984,0.647058844566},{0.392003081593,0.751864682226,0.650749732466},{0.392003081593,0.751864682226,0.650749732466},{0.384006157225,0.742945036467,0.654440620366},{0.384006157225,0.742945036467,0.654440620366},{0.376009232858,0.734025390709,0.658131508266},{0.376009232858,0.734025390709,0.658131508266},{0.36801230849,0.725105744951,0.661822396166},{0.36801230849,0.725105744951,0.661822396166},{0.360015384122,0.716186099193,0.665513284066},{0.360015384122,0.716186099193,0.665513284066},{0.352018459755,0.707266453434,0.669204171966},{0.352018459755,0.707266453434,0.669204171966},{0.344021535387,0.698346807676,0.672895059866},{0.344021535387,0.698346807676,0.672895059866},{0.33602461102,0.689427161918,0.676585947766},{0.33602461102,0.689427161918,0.676585947766},{0.328027686652,0.68050751616,0.680276835666},{0.328027686652,0.68050751616,0.680276835666},{0.320030762284,0.671587870401,0.683967723566},{0.320030762284,0.671587870401,0.683967723566},{0.312033837917,0.662668224643,0.687658611466},{0.312033837917,0.662668224643,0.687658611466},{0.304036913549,0.653748578885,0.691349499366},{0.296039989182,0.644828933127,0.695040387266},{0.296039989182,0.644828933127,0.695040387266},{0.288043064814,0.635909287369,0.698731275166},{0.288043064814,0.635909287369,0.698731275166},{0.280046140446,0.62698964161,0.702422163066},{0.280046140446,0.62698964161,0.702422163066},{0.272049216079,0.618069995852,0.706113050966},{0.272049216079,0.618069995852,0.706113050966},{0.264052291711,0.609150350094,0.709803938866},{0.264052291711,0.609150350094,0.709803938866},{0.256055367344,0.600230704336,0.713494826766},{0.256055367344,0.600230704336,0.713494826766},{0.248058442976,0.591311058577,0.717185714666},{0.248058442976,0.591311058577,0.717185714666},{0.240061518608,0.582391412819,0.720876602566},{0.240061518608,0.582391412819,0.720876602566},{0.232064594241,0.573471767061,0.724567490466},{0.232064594241,0.573471767061,0.724567490466},{0.224067669873,0.564552121303,0.728258378365},{0.224067669873,0.564552121303,0.728258378365},{0.216070745506,0.555632475544,0.731949266265},{0.216070745506,0.555632475544,0.731949266265},{0.208073821138,0.546712829786,0.735640154165},{0.208073821138,0.546712829786,0.735640154165},{0.20007689677,0.537793184028,0.739331042065},{0.20007689677,0.537793184028,0.739331042065},{0.199461748787,0.528950431183,0.739100361572},{0.199461748787,0.528950431183,0.739100361572},{0.206228377188,0.520184571252,0.734948112684},{0.206228377188,0.520184571252,0.734948112684},{0.212995005589,0.511418711321,0.730795863797},{0.212995005589,0.511418711321,0.730795863797},{0.21976163399,0.50265285139,0.726643614909},{0.21976163399,0.50265285139,0.726643614909},{0.226528262391,0.493886991459,0.722491366022},{0.226528262391,0.493886991459,0.722491366022},{0.233294890792,0.485121131528,0.718339117134},{0.233294890792,0.485121131528,0.718339117134},{0.240061519193,0.476355271597,0.714186868247},{0.246828147594,0.467589411665,0.710034619359},{0.246828147594,0.467589411665,0.710034619359},{0.253594775995,0.458823551734,0.705882370472},{0.253594775995,0.458823551734,0.705882370472},{0.260361404396,0.450057691803,0.701730121584},{0.260361404396,0.450057691803,0.701730121584},{0.267128032797,0.441291831872,0.697577872697},{0.267128032797,0.441291831872,0.697577872697},{0.273894661197,0.432525971941,0.69342562381},{0.273894661197,0.432525971941,0.69342562381},{0.280661289598,0.42376011201,0.689273374922},{0.280661289598,0.42376011201,0.689273374922},{0.287427917999,0.414994252079,0.685121126035},{0.287427917999,0.414994252079,0.685121126035},{0.2941945464,0.406228392148,0.680968877147},{0.2941945464,0.406228392148,0.680968877147},{0.300961174801,0.397462532216,0.67681662826},{0.300961174801,0.397462532216,0.67681662826},{0.307727803202,0.388696672285,0.672664379372},{0.307727803202,0.388696672285,0.672664379372},{0.314494431603,0.379930812354,0.668512130485},{0.314494431603,0.379930812354,0.668512130485},{0.321261060004,0.371164952423,0.664359881597},{0.321261060004,0.371164952423,0.664359881597},{0.328027688405,0.362399092492,0.66020763271},{0.328027688405,0.362399092492,0.66020763271},{0.334794316806,0.353633232561,0.656055383822},{0.334794316806,0.353633232561,0.656055383822},{0.341560945207,0.34486737263,0.651903134935},{0.341560945207,0.34486737263,0.651903134935},{0.348327573608,0.336101512699,0.647750886048},{0.348327573608,0.336101512699,0.647750886048},{0.355094202009,0.327335652767,0.64359863716},{0.355094202009,0.327335652767,0.64359863716},{0.36186083041,0.318569792836,0.639446388273},{0.36186083041,0.318569792836,0.639446388273},{0.368627458811,0.309803932905,0.635294139385},{0.368627458811,0.309803932905,0.635294139385},};

const std::vector<glm::vec3> CM_RAINBOW =
{{0.5,0.0,1.0},{0.5,0.0,1.0},{0.492156862745,0.0123196595352,0.999981027349},{0.492156862745,0.0123196595352,0.999981027349},{0.48431372549,0.0246374491954,0.999924110115},{0.48431372549,0.0246374491954,0.999924110115},{0.476470588235,0.0369514993891,0.999829250458},{0.476470588235,0.0369514993891,0.999829250458},{0.46862745098,0.0492599410928,0.999696451978},{0.46862745098,0.0492599410928,0.999696451978},{0.460784313725,0.0615609061339,0.999525719713},{0.460784313725,0.0615609061339,0.999525719713},{0.452941176471,0.0738525274749,0.999317060143},{0.452941176471,0.0738525274749,0.999317060143},{0.445098039216,0.0861329394961,0.999070481184},{0.445098039216,0.0861329394961,0.999070481184},{0.437254901961,0.0984002782796,0.998785992194},{0.437254901961,0.0984002782796,0.998785992194},{0.429411764706,0.110652681892,0.998463603967},{0.429411764706,0.110652681892,0.998463603967},{0.421568627451,0.122888290665,0.998103328737},{0.421568627451,0.122888290665,0.998103328737},{0.413725490196,0.135105247481,0.997705180174},{0.413725490196,0.135105247481,0.997705180174},{0.405882352941,0.147301698055,0.997269173386},{0.405882352941,0.147301698055,0.997269173386},{0.398039215686,0.15947579121,0.996795324917},{0.398039215686,0.15947579121,0.996795324917},{0.390196078431,0.171625679166,0.996283652748},{0.390196078431,0.171625679166,0.996283652748},{0.382352941176,0.183749517817,0.995734176295},{0.382352941176,0.183749517817,0.995734176295},{0.374509803922,0.195845467007,0.995146916407},{0.374509803922,0.195845467007,0.995146916407},{0.366666666667,0.207911690818,0.994521895368},{0.366666666667,0.207911690818,0.994521895368},{0.358823529412,0.21994635784,0.993859136895},{0.358823529412,0.21994635784,0.993859136895},{0.350980392157,0.231947641454,0.993158666137},{0.343137254902,0.243913720108,0.992420509672},{0.343137254902,0.243913720108,0.992420509672},{0.335294117647,0.255842777594,0.991644695511},{0.335294117647,0.255842777594,0.991644695511},{0.327450980392,0.267733003322,0.990831253092},{0.327450980392,0.267733003322,0.990831253092},{0.319607843137,0.279582592597,0.989980213281},{0.319607843137,0.279582592597,0.989980213281},{0.311764705882,0.291389746889,0.989091608371},{0.311764705882,0.291389746889,0.989091608371},{0.303921568627,0.303152674113,0.988165472081},{0.303921568627,0.303152674113,0.988165472081},{0.296078431373,0.314869588894,0.987201839554},{0.296078431373,0.314869588894,0.987201839554},{0.288235294118,0.32653871284,0.986200747353},{0.288235294118,0.32653871284,0.986200747353},{0.280392156863,0.338158274816,0.985162233468},{0.280392156863,0.338158274816,0.985162233468},{0.272549019608,0.349726511206,0.984086337303},{0.272549019608,0.349726511206,0.984086337303},{0.264705882353,0.361241666187,0.982973099684},{0.264705882353,0.361241666187,0.982973099684},{0.256862745098,0.372701991991,0.981822562854},{0.256862745098,0.372701991991,0.981822562854},{0.249019607843,0.384105749172,0.980634770469},{0.249019607843,0.384105749172,0.980634770469},{0.241176470588,0.395451206871,0.979409767601},{0.241176470588,0.395451206871,0.979409767601},{0.233333333333,0.406736643076,0.978147600734},{0.233333333333,0.406736643076,0.978147600734},{0.225490196078,0.417960344887,0.97684831776},{0.225490196078,0.417960344887,0.97684831776},{0.217647058824,0.429120608773,0.97551196798},{0.217647058824,0.429120608773,0.97551196798},{0.209803921569,0.440215740831,0.974138602105},{0.209803921569,0.440215740831,0.974138602105},{0.201960784314,0.451244057045,0.972728272245},{0.201960784314,0.451244057045,0.972728272245},{0.194117647059,0.46220388354,0.971281031916},{0.186274509804,0.473093556836,0.969796936035},{0.186274509804,0.473093556836,0.969796936035},{0.178431372549,0.4839114241,0.968276040916},{0.178431372549,0.4839114241,0.968276040916},{0.170588235294,0.4946558434,0.966718404269},{0.170588235294,0.4946558434,0.966718404269},{0.162745098039,0.505325183949,0.9651240852},{0.162745098039,0.505325183949,0.9651240852},{0.154901960784,0.515917826358,0.963493144206},{0.154901960784,0.515917826358,0.963493144206},{0.147058823529,0.526432162877,0.961825643173},{0.147058823529,0.526432162877,0.961825643173},{0.139215686275,0.536866597644,0.960121645375},{0.139215686275,0.536866597644,0.960121645375},{0.13137254902,0.547219546922,0.95838121547},{0.13137254902,0.547219546922,0.95838121547},{0.123529411765,0.557489439343,0.9566044195},{0.123529411765,0.557489439343,0.9566044195},{0.11568627451,0.567674716145,0.954791324887},{0.11568627451,0.567674716145,0.954791324887},{0.107843137255,0.577773831408,0.952942000427},{0.107843137255,0.577773831408,0.952942000427},{0.1,0.587785252292,0.951056516295},{0.1,0.587785252292,0.951056516295},{0.0921568627451,0.597707459266,0.949134944036},{0.0921568627451,0.597707459266,0.949134944036},{0.0843137254902,0.607538946339,0.947177356564},{0.0843137254902,0.607538946339,0.947177356564},{0.0764705882353,0.61727822129,0.945183828161},{0.0764705882353,0.61727822129,0.945183828161},{0.0686274509804,0.626923805894,0.943154434471},{0.0686274509804,0.626923805894,0.943154434471},{0.0607843137255,0.636474236147,0.941089252501},{0.0607843137255,0.636474236147,0.941089252501},{0.0529411764706,0.645928062487,0.938988360615},{0.0529411764706,0.645928062487,0.938988360615},{0.0450980392157,0.655283850013,0.936851838531},{0.0450980392157,0.655283850013,0.936851838531},{0.0372549019608,0.664540178708,0.934679767321},{0.0294117647059,0.673695643647,0.932472229404},{0.0294117647059,0.673695643647,0.932472229404},{0.021568627451,0.682748855215,0.930229308547},{0.021568627451,0.682748855215,0.930229308547},{0.0137254901961,0.691698439319,0.927951089857},{0.0137254901961,0.691698439319,0.927951089857},{0.00588235294118,0.700543037593,0.925637659782},{0.00588235294118,0.700543037593,0.925637659782},{0.00196078431373,0.709281307606,0.923289106105},{0.00196078431373,0.709281307606,0.923289106105},{0.00980392156863,0.717911923064,0.920905517945},{0.00980392156863,0.717911923064,0.920905517945},{0.0176470588235,0.726433574016,0.918486985746},{0.0176470588235,0.726433574016,0.918486985746},{0.0254901960784,0.734844967047,0.91603360128},{0.0254901960784,0.734844967047,0.91603360128},{0.0333333333333,0.743144825477,0.913545457643},{0.0333333333333,0.743144825477,0.913545457643},{0.0411764705882,0.751331889557,0.911022649246},{0.0411764705882,0.751331889557,0.911022649246},{0.0490196078431,0.759404916655,0.90846527182},{0.0490196078431,0.759404916655,0.90846527182},{0.056862745098,0.767362681449,0.905873422403},{0.056862745098,0.767362681449,0.905873422403},{0.0647058823529,0.775203976111,0.903247199346},{0.0647058823529,0.775203976111,0.903247199346},{0.0725490196078,0.782927610492,0.900586702301},{0.0725490196078,0.782927610492,0.900586702301},{0.0803921568627,0.7905324123,0.89789203222},{0.0803921568627,0.7905324123,0.89789203222},{0.0882352941176,0.79801722728,0.895163291355},{0.0882352941176,0.79801722728,0.895163291355},{0.0960784313725,0.805380919389,0.892400583248},{0.0960784313725,0.805380919389,0.892400583248},{0.103921568627,0.812622370966,0.889604012731},{0.103921568627,0.812622370966,0.889604012731},{0.111764705882,0.819740482907,0.88677368592},{0.119607843137,0.826734174826,0.883909710214},{0.119607843137,0.826734174826,0.883909710214},{0.127450980392,0.833602385221,0.881012194286},{0.127450980392,0.833602385221,0.881012194286},{0.135294117647,0.840344071638,0.878081248084},{0.135294117647,0.840344071638,0.878081248084},{0.143137254902,0.846958210824,0.875116982823},{0.143137254902,0.846958210824,0.875116982823},{0.150980392157,0.853443798888,0.872119510984},{0.150980392157,0.853443798888,0.872119510984},{0.158823529412,0.859799851448,0.869088946306},{0.158823529412,0.859799851448,0.869088946306},{0.166666666667,0.866025403784,0.866025403784},{0.166666666667,0.866025403784,0.866025403784},{0.174509803922,0.872119510984,0.862928999667},{0.174509803922,0.872119510984,0.862928999667},{0.182352941176,0.878081248084,0.859799851448},{0.182352941176,0.878081248084,0.859799851448},{0.190196078431,0.883909710214,0.856638077864},{0.190196078431,0.883909710214,0.856638077864},{0.198039215686,0.889604012731,0.853443798888},{0.198039215686,0.889604012731,0.853443798888},{0.205882352941,0.895163291355,0.85021713573},{0.205882352941,0.895163291355,0.85021713573},{0.213725490196,0.900586702301,0.846958210824},{0.213725490196,0.900586702301,0.846958210824},{0.221568627451,0.905873422403,0.843667147834},{0.221568627451,0.905873422403,0.843667147834},{0.229411764706,0.911022649246,0.840344071638},{0.229411764706,0.911022649246,0.840344071638},{0.237254901961,0.91603360128,0.836989108332},{0.237254901961,0.91603360128,0.836989108332},{0.245098039216,0.920905517945,0.833602385221},{0.245098039216,0.920905517945,0.833602385221},{0.252941176471,0.925637659782,0.830184030816},{0.252941176471,0.925637659782,0.830184030816},{0.260784313725,0.930229308547,0.826734174826},{0.260784313725,0.930229308547,0.826734174826},{0.26862745098,0.934679767321,0.823252948158},{0.276470588235,0.938988360615,0.819740482907},{0.276470588235,0.938988360615,0.819740482907},{0.28431372549,0.943154434471,0.816196912356},{0.28431372549,0.943154434471,0.816196912356},{0.292156862745,0.947177356564,0.812622370966},{0.292156862745,0.947177356564,0.812622370966},{0.3,0.951056516295,0.809016994375},{0.3,0.951056516295,0.809016994375},{0.307843137255,0.954791324887,0.805380919389},{0.307843137255,0.954791324887,0.805380919389},{0.31568627451,0.95838121547,0.80171428398},{0.31568627451,0.95838121547,0.80171428398},{0.323529411765,0.961825643173,0.79801722728},{0.323529411765,0.961825643173,0.79801722728},{0.33137254902,0.9651240852,0.794289889575},{0.33137254902,0.9651240852,0.794289889575},{0.339215686275,0.968276040916,0.7905324123},{0.339215686275,0.968276040916,0.7905324123},{0.347058823529,0.971281031916,0.786744938033},{0.347058823529,0.971281031916,0.786744938033},{0.354901960784,0.974138602105,0.782927610492},{0.354901960784,0.974138602105,0.782927610492},{0.362745098039,0.97684831776,0.779080574526},{0.362745098039,0.97684831776,0.779080574526},{0.370588235294,0.979409767601,0.775203976111},{0.370588235294,0.979409767601,0.775203976111},{0.378431372549,0.981822562854,0.771297962347},{0.378431372549,0.981822562854,0.771297962347},{0.386274509804,0.984086337303,0.767362681449},{0.386274509804,0.984086337303,0.767362681449},{0.394117647059,0.986200747353,0.763398282741},{0.394117647059,0.986200747353,0.763398282741},{0.401960784314,0.988165472081,0.759404916655},{0.401960784314,0.988165472081,0.759404916655},{0.409803921569,0.989980213281,0.755382734719},{0.409803921569,0.989980213281,0.755382734719},{0.417647058824,0.991644695511,0.751331889557},{0.417647058824,0.991644695511,0.751331889557},{0.425490196078,0.993158666137,0.747252534879},{0.433333333333,0.994521895368,0.743144825477},{0.433333333333,0.994521895368,0.743144825477},{0.441176470588,0.995734176295,0.739008917221},{0.441176470588,0.995734176295,0.739008917221},{0.449019607843,0.996795324917,0.734844967047},{0.449019607843,0.996795324917,0.734844967047},{0.456862745098,0.997705180174,0.730653132959},{0.456862745098,0.997705180174,0.730653132959},{0.464705882353,0.998463603967,0.726433574016},{0.464705882353,0.998463603967,0.726433574016},{0.472549019608,0.999070481184,0.722186450332},{0.472549019608,0.999070481184,0.722186450332},{0.480392156863,0.999525719713,0.717911923064},{0.480392156863,0.999525719713,0.717911923064},{0.488235294118,0.999829250458,0.713610154412},{0.488235294118,0.999829250458,0.713610154412},{0.496078431373,0.999981027349,0.709281307606},{0.496078431373,0.999981027349,0.709281307606},{0.503921568627,0.999981027349,0.704925546906},{0.503921568627,0.999981027349,0.704925546906},{0.511764705882,0.999829250458,0.700543037593},{0.511764705882,0.999829250458,0.700543037593},{0.519607843137,0.999525719713,0.696133945963},{0.519607843137,0.999525719713,0.696133945963},{0.527450980392,0.999070481184,0.691698439319},{0.527450980392,0.999070481184,0.691698439319},{0.535294117647,0.998463603967,0.687236685969},{0.535294117647,0.998463603967,0.687236685969},{0.543137254902,0.997705180174,0.682748855215},{0.543137254902,0.997705180174,0.682748855215},{0.550980392157,0.996795324917,0.678235117349},{0.550980392157,0.996795324917,0.678235117349},{0.558823529412,0.995734176295,0.673695643647},{0.558823529412,0.995734176295,0.673695643647},{0.566666666667,0.994521895368,0.669130606359},{0.566666666667,0.994521895368,0.669130606359},{0.574509803922,0.993158666137,0.664540178708},{0.582352941176,0.991644695511,0.659924534879},{0.582352941176,0.991644695511,0.659924534879},{0.590196078431,0.989980213281,0.655283850013},{0.590196078431,0.989980213281,0.655283850013},{0.598039215686,0.988165472081,0.650618300204},{0.598039215686,0.988165472081,0.650618300204},{0.605882352941,0.986200747353,0.645928062487},{0.605882352941,0.986200747353,0.645928062487},{0.613725490196,0.984086337303,0.641213314834},{0.613725490196,0.984086337303,0.641213314834},{0.621568627451,0.981822562854,0.636474236147},{0.621568627451,0.981822562854,0.636474236147},{0.629411764706,0.979409767601,0.631711006253},{0.629411764706,0.979409767601,0.631711006253},{0.637254901961,0.97684831776,0.626923805894},{0.637254901961,0.97684831776,0.626923805894},{0.645098039216,0.974138602105,0.622112816721},{0.645098039216,0.974138602105,0.622112816721},{0.652941176471,0.971281031916,0.61727822129},{0.652941176471,0.971281031916,0.61727822129},{0.660784313725,0.968276040916,0.612420203049},{0.660784313725,0.968276040916,0.612420203049},{0.66862745098,0.9651240852,0.607538946339},{0.66862745098,0.9651240852,0.607538946339},{0.676470588235,0.961825643173,0.602634636379},{0.676470588235,0.961825643173,0.602634636379},{0.68431372549,0.95838121547,0.597707459266},{0.68431372549,0.95838121547,0.597707459266},{0.692156862745,0.954791324887,0.592757601963},{0.692156862745,0.954791324887,0.592757601963},{0.7,0.951056516295,0.587785252292},{0.7,0.951056516295,0.587785252292},{0.707843137255,0.947177356564,0.582790598933},{0.707843137255,0.947177356564,0.582790598933},{0.71568627451,0.943154434471,0.577773831408},{0.71568627451,0.943154434471,0.577773831408},{0.723529411765,0.938988360615,0.572735140081},{0.723529411765,0.938988360615,0.572735140081},{0.73137254902,0.934679767321,0.567674716145},{0.739215686275,0.930229308547,0.56259275162},{0.739215686275,0.930229308547,0.56259275162},{0.747058823529,0.925637659782,0.557489439343},{0.747058823529,0.925637659782,0.557489439343},{0.754901960784,0.920905517945,0.552364972961},{0.754901960784,0.920905517945,0.552364972961},{0.762745098039,0.91603360128,0.547219546922},{0.762745098039,0.91603360128,0.547219546922},{0.770588235294,0.911022649246,0.542053356472},{0.770588235294,0.911022649246,0.542053356472},{0.778431372549,0.905873422403,0.536866597644},{0.778431372549,0.905873422403,0.536866597644},{0.786274509804,0.900586702301,0.53165946725},{0.786274509804,0.900586702301,0.53165946725},{0.794117647059,0.895163291355,0.526432162877},{0.794117647059,0.895163291355,0.526432162877},{0.801960784314,0.889604012731,0.521184882877},{0.801960784314,0.889604012731,0.521184882877},{0.809803921569,0.883909710214,0.515917826358},{0.809803921569,0.883909710214,0.515917826358},{0.817647058824,0.878081248084,0.510631193181},{0.817647058824,0.878081248084,0.510631193181},{0.825490196078,0.872119510984,0.505325183949},{0.825490196078,0.872119510984,0.505325183949},{0.833333333333,0.866025403784,0.5},{0.833333333333,0.866025403784,0.5},{0.841176470588,0.859799851448,0.4946558434},{0.841176470588,0.859799851448,0.4946558434},{0.849019607843,0.853443798888,0.489292916934},{0.849019607843,0.853443798888,0.489292916934},{0.856862745098,0.846958210824,0.4839114241},{0.856862745098,0.846958210824,0.4839114241},{0.864705882353,0.840344071638,0.478511569101},{0.864705882353,0.840344071638,0.478511569101},{0.872549019608,0.833602385221,0.473093556836},{0.872549019608,0.833602385221,0.473093556836},{0.880392156863,0.826734174826,0.467657592893},{0.880392156863,0.826734174826,0.467657592893},{0.888235294118,0.819740482907,0.46220388354},{0.896078431373,0.812622370966,0.456732635722},{0.896078431373,0.812622370966,0.456732635722},{0.903921568627,0.805380919389,0.451244057045},{0.903921568627,0.805380919389,0.451244057045},{0.911764705882,0.79801722728,0.445738355777},{0.911764705882,0.79801722728,0.445738355777},{0.919607843137,0.7905324123,0.440215740831},{0.919607843137,0.7905324123,0.440215740831},{0.927450980392,0.782927610492,0.434676421766},{0.927450980392,0.782927610492,0.434676421766},{0.935294117647,0.775203976111,0.429120608773},{0.935294117647,0.775203976111,0.429120608773},{0.943137254902,0.767362681449,0.423548512668},{0.943137254902,0.767362681449,0.423548512668},{0.950980392157,0.759404916655,0.417960344887},{0.950980392157,0.759404916655,0.417960344887},{0.958823529412,0.751331889557,0.412356317474},{0.958823529412,0.751331889557,0.412356317474},{0.966666666667,0.743144825477,0.406736643076},{0.966666666667,0.743144825477,0.406736643076},{0.974509803922,0.734844967047,0.401101534933},{0.974509803922,0.734844967047,0.401101534933},{0.982352941176,0.726433574016,0.395451206871},{0.982352941176,0.726433574016,0.395451206871},{0.990196078431,0.717911923064,0.389785873293},{0.990196078431,0.717911923064,0.389785873293},{0.998039215686,0.709281307606,0.384105749172},{0.998039215686,0.709281307606,0.384105749172},{1.0,0.700543037593,0.378411050042},{1.0,0.700543037593,0.378411050042},{1.0,0.691698439319,0.372701991991},{1.0,0.691698439319,0.372701991991},{1.0,0.682748855215,0.36697879165},{1.0,0.682748855215,0.36697879165},{1.0,0.673695643647,0.361241666187},{1.0,0.673695643647,0.361241666187},{1.0,0.664540178708,0.3554908333},{1.0,0.655283850013,0.349726511206},{1.0,0.655283850013,0.349726511206},{1.0,0.645928062487,0.343948918634},{1.0,0.645928062487,0.343948918634},{1.0,0.636474236147,0.338158274816},{1.0,0.636474236147,0.338158274816},{1.0,0.626923805894,0.33235479948},{1.0,0.626923805894,0.33235479948},{1.0,0.61727822129,0.32653871284},{1.0,0.61727822129,0.32653871284},{1.0,0.607538946339,0.32071023559},{1.0,0.607538946339,0.32071023559},{1.0,0.597707459266,0.314869588894},{1.0,0.597707459266,0.314869588894},{1.0,0.587785252292,0.309016994375},{1.0,0.587785252292,0.309016994375},{1.0,0.577773831408,0.303152674113},{1.0,0.577773831408,0.303152674113},{1.0,0.567674716145,0.297276850631},{1.0,0.567674716145,0.297276850631},{1.0,0.557489439343,0.291389746889},{1.0,0.557489439343,0.291389746889},{1.0,0.547219546922,0.285491586275},{1.0,0.547219546922,0.285491586275},{1.0,0.536866597644,0.279582592597},{1.0,0.536866597644,0.279582592597},{1.0,0.526432162877,0.273662990072},{1.0,0.526432162877,0.273662990072},{1.0,0.515917826358,0.267733003322},{1.0,0.515917826358,0.267733003322},{1.0,0.505325183949,0.261792857363},{1.0,0.505325183949,0.261792857363},{1.0,0.4946558434,0.255842777594},{1.0,0.4946558434,0.255842777594},{1.0,0.4839114241,0.249882989794},{1.0,0.4839114241,0.249882989794},{1.0,0.473093556836,0.243913720108},{1.0,0.473093556836,0.243913720108},{1.0,0.46220388354,0.237935195043},{1.0,0.451244057045,0.231947641454},{1.0,0.451244057045,0.231947641454},{1.0,0.440215740831,0.225951286542},{1.0,0.440215740831,0.225951286542},{1.0,0.429120608773,0.21994635784},{1.0,0.429120608773,0.21994635784},{1.0,0.417960344887,0.213933083206},{1.0,0.417960344887,0.213933083206},{1.0,0.406736643076,0.207911690818},{1.0,0.406736643076,0.207911690818},{1.0,0.395451206871,0.201882409157},{1.0,0.395451206871,0.201882409157},{1.0,0.384105749172,0.195845467007},{1.0,0.384105749172,0.195845467007},{1.0,0.372701991991,0.189801093442},{1.0,0.372701991991,0.189801093442},{1.0,0.361241666187,0.183749517817},{1.0,0.361241666187,0.183749517817},{1.0,0.349726511206,0.17769096976},{1.0,0.349726511206,0.17769096976},{1.0,0.338158274816,0.171625679166},{1.0,0.338158274816,0.171625679166},{1.0,0.32653871284,0.165553876184},{1.0,0.32653871284,0.165553876184},{1.0,0.314869588894,0.15947579121},{1.0,0.314869588894,0.15947579121},{1.0,0.303152674113,0.153391654879},{1.0,0.303152674113,0.153391654879},{1.0,0.291389746889,0.147301698055},{1.0,0.291389746889,0.147301698055},{1.0,0.279582592597,0.141206151823},{1.0,0.279582592597,0.141206151823},{1.0,0.267733003322,0.135105247481},{1.0,0.267733003322,0.135105247481},{1.0,0.255842777594,0.12899921653},{1.0,0.255842777594,0.12899921653},{1.0,0.243913720108,0.122888290665},{1.0,0.243913720108,0.122888290665},{1.0,0.231947641454,0.116772701766},{1.0,0.21994635784,0.110652681892},{1.0,0.21994635784,0.110652681892},{1.0,0.207911690818,0.104528463268},{1.0,0.207911690818,0.104528463268},{1.0,0.195845467007,0.0984002782796},{1.0,0.195845467007,0.0984002782796},{1.0,0.183749517817,0.0922683594633},{1.0,0.183749517817,0.0922683594633},{1.0,0.171625679166,0.0861329394961},{1.0,0.171625679166,0.0861329394961},{1.0,0.15947579121,0.0799942511885},{1.0,0.15947579121,0.0799942511885},{1.0,0.147301698055,0.0738525274749},{1.0,0.147301698055,0.0738525274749},{1.0,0.135105247481,0.0677080014047},{1.0,0.135105247481,0.0677080014047},{1.0,0.122888290665,0.0615609061339},{1.0,0.122888290665,0.0615609061339},{1.0,0.110652681892,0.055411474916},{1.0,0.110652681892,0.055411474916},{1.0,0.0984002782796,0.0492599410928},{1.0,0.0984002782796,0.0492599410928},{1.0,0.0861329394961,0.0431065380863},{1.0,0.0861329394961,0.0431065380863},{1.0,0.0738525274749,0.0369514993891},{1.0,0.0738525274749,0.0369514993891},{1.0,0.0615609061339,0.0307950585562},{1.0,0.0615609061339,0.0307950585562},{1.0,0.0492599410928,0.0246374491954},{1.0,0.0492599410928,0.0246374491954},{1.0,0.0369514993891,0.0184789049591},{1.0,0.0369514993891,0.0184789049591},{1.0,0.0246374491954,0.0123196595352},{1.0,0.0246374491954,0.0123196595352},{1.0,0.0123196595352,0.00615994663814},{1.0,0.0123196595352,0.00615994663814},{1.0,1.22464679915e-16,6.12323399574e-17},{1.0,1.22464679915e-16,6.12323399574e-17},};


const std::vector<glm::vec3> CM_JET = 
{{0.0,0.0,0.5},{0.0,0.0,0.5},{0.0,0.0,0.517825311943},{0.0,0.0,0.517825311943},{0.0,0.0,0.535650623886},{0.0,0.0,0.535650623886},{0.0,0.0,0.553475935829},{0.0,0.0,0.553475935829},{0.0,0.0,0.571301247772},{0.0,0.0,0.571301247772},{0.0,0.0,0.589126559715},{0.0,0.0,0.589126559715},{0.0,0.0,0.606951871658},{0.0,0.0,0.606951871658},{0.0,0.0,0.624777183601},{0.0,0.0,0.624777183601},{0.0,0.0,0.642602495544},{0.0,0.0,0.642602495544},{0.0,0.0,0.660427807487},{0.0,0.0,0.660427807487},{0.0,0.0,0.67825311943},{0.0,0.0,0.67825311943},{0.0,0.0,0.696078431373},{0.0,0.0,0.696078431373},{0.0,0.0,0.713903743316},{0.0,0.0,0.713903743316},{0.0,0.0,0.731729055258},{0.0,0.0,0.731729055258},{0.0,0.0,0.749554367201},{0.0,0.0,0.749554367201},{0.0,0.0,0.767379679144},{0.0,0.0,0.767379679144},{0.0,0.0,0.785204991087},{0.0,0.0,0.785204991087},{0.0,0.0,0.80303030303},{0.0,0.0,0.80303030303},{0.0,0.0,0.820855614973},{0.0,0.0,0.820855614973},{0.0,0.0,0.838680926916},{0.0,0.0,0.856506238859},{0.0,0.0,0.856506238859},{0.0,0.0,0.874331550802},{0.0,0.0,0.874331550802},{0.0,0.0,0.892156862745},{0.0,0.0,0.892156862745},{0.0,0.0,0.909982174688},{0.0,0.0,0.909982174688},{0.0,0.0,0.927807486631},{0.0,0.0,0.927807486631},{0.0,0.0,0.945632798574},{0.0,0.0,0.945632798574},{0.0,0.0,0.963458110517},{0.0,0.0,0.963458110517},{0.0,0.0,0.98128342246},{0.0,0.0,0.98128342246},{0.0,0.0,0.999108734403},{0.0,0.0,0.999108734403},{0.0,0.0,1.0},{0.0,0.0,1.0},{0.0,0.0,1.0},{0.0,0.0,1.0},{0.0,0.0,1.0},{0.0,0.0,1.0},{0.0,0.00196078431373,1.0},{0.0,0.00196078431373,1.0},{0.0,0.0176470588235,1.0},{0.0,0.0176470588235,1.0},{0.0,0.0333333333333,1.0},{0.0,0.0333333333333,1.0},{0.0,0.0490196078431,1.0},{0.0,0.0490196078431,1.0},{0.0,0.0647058823529,1.0},{0.0,0.0647058823529,1.0},{0.0,0.0803921568627,1.0},{0.0,0.0803921568627,1.0},{0.0,0.0960784313725,1.0},{0.0,0.0960784313725,1.0},{0.0,0.111764705882,1.0},{0.0,0.127450980392,1.0},{0.0,0.127450980392,1.0},{0.0,0.143137254902,1.0},{0.0,0.143137254902,1.0},{0.0,0.158823529412,1.0},{0.0,0.158823529412,1.0},{0.0,0.174509803922,1.0},{0.0,0.174509803922,1.0},{0.0,0.190196078431,1.0},{0.0,0.190196078431,1.0},{0.0,0.205882352941,1.0},{0.0,0.205882352941,1.0},{0.0,0.221568627451,1.0},{0.0,0.221568627451,1.0},{0.0,0.237254901961,1.0},{0.0,0.237254901961,1.0},{0.0,0.252941176471,1.0},{0.0,0.252941176471,1.0},{0.0,0.26862745098,1.0},{0.0,0.26862745098,1.0},{0.0,0.28431372549,1.0},{0.0,0.28431372549,1.0},{0.0,0.3,1.0},{0.0,0.3,1.0},{0.0,0.31568627451,1.0},{0.0,0.31568627451,1.0},{0.0,0.33137254902,1.0},{0.0,0.33137254902,1.0},{0.0,0.347058823529,1.0},{0.0,0.347058823529,1.0},{0.0,0.362745098039,1.0},{0.0,0.362745098039,1.0},{0.0,0.378431372549,1.0},{0.0,0.378431372549,1.0},{0.0,0.394117647059,1.0},{0.0,0.394117647059,1.0},{0.0,0.409803921569,1.0},{0.0,0.409803921569,1.0},{0.0,0.425490196078,1.0},{0.0,0.441176470588,1.0},{0.0,0.441176470588,1.0},{0.0,0.456862745098,1.0},{0.0,0.456862745098,1.0},{0.0,0.472549019608,1.0},{0.0,0.472549019608,1.0},{0.0,0.488235294118,1.0},{0.0,0.488235294118,1.0},{0.0,0.503921568627,1.0},{0.0,0.503921568627,1.0},{0.0,0.519607843137,1.0},{0.0,0.519607843137,1.0},{0.0,0.535294117647,1.0},{0.0,0.535294117647,1.0},{0.0,0.550980392157,1.0},{0.0,0.550980392157,1.0},{0.0,0.566666666667,1.0},{0.0,0.566666666667,1.0},{0.0,0.582352941176,1.0},{0.0,0.582352941176,1.0},{0.0,0.598039215686,1.0},{0.0,0.598039215686,1.0},{0.0,0.613725490196,1.0},{0.0,0.613725490196,1.0},{0.0,0.629411764706,1.0},{0.0,0.629411764706,1.0},{0.0,0.645098039216,1.0},{0.0,0.645098039216,1.0},{0.0,0.660784313725,1.0},{0.0,0.660784313725,1.0},{0.0,0.676470588235,1.0},{0.0,0.676470588235,1.0},{0.0,0.692156862745,1.0},{0.0,0.692156862745,1.0},{0.0,0.707843137255,1.0},{0.0,0.707843137255,1.0},{0.0,0.723529411765,1.0},{0.0,0.739215686275,1.0},{0.0,0.739215686275,1.0},{0.0,0.754901960784,1.0},{0.0,0.754901960784,1.0},{0.0,0.770588235294,1.0},{0.0,0.770588235294,1.0},{0.0,0.786274509804,1.0},{0.0,0.786274509804,1.0},{0.0,0.801960784314,1.0},{0.0,0.801960784314,1.0},{0.0,0.817647058824,1.0},{0.0,0.817647058824,1.0},{0.0,0.833333333333,1.0},{0.0,0.833333333333,1.0},{0.0,0.849019607843,1.0},{0.0,0.849019607843,1.0},{0.0,0.864705882353,0.996204933586},{0.0,0.864705882353,0.996204933586},{0.0,0.880392156863,0.983554712207},{0.0,0.880392156863,0.983554712207},{0.0,0.896078431373,0.970904490829},{0.0,0.896078431373,0.970904490829},{0.00948766603416,0.911764705882,0.95825426945},{0.00948766603416,0.911764705882,0.95825426945},{0.022137887413,0.927450980392,0.945604048071},{0.022137887413,0.927450980392,0.945604048071},{0.0347881087919,0.943137254902,0.932953826692},{0.0347881087919,0.943137254902,0.932953826692},{0.0474383301708,0.958823529412,0.920303605313},{0.0474383301708,0.958823529412,0.920303605313},{0.0600885515497,0.974509803922,0.907653383934},{0.0600885515497,0.974509803922,0.907653383934},{0.0727387729285,0.990196078431,0.895003162555},{0.0727387729285,0.990196078431,0.895003162555},{0.0853889943074,1.0,0.882352941176},{0.0853889943074,1.0,0.882352941176},{0.0980392156863,1.0,0.869702719798},{0.0980392156863,1.0,0.869702719798},{0.110689437065,1.0,0.857052498419},{0.123339658444,1.0,0.84440227704},{0.123339658444,1.0,0.84440227704},{0.135989879823,1.0,0.831752055661},{0.135989879823,1.0,0.831752055661},{0.148640101202,1.0,0.819101834282},{0.148640101202,1.0,0.819101834282},{0.161290322581,1.0,0.806451612903},{0.161290322581,1.0,0.806451612903},{0.17394054396,1.0,0.793801391524},{0.17394054396,1.0,0.793801391524},{0.186590765338,1.0,0.781151170145},{0.186590765338,1.0,0.781151170145},{0.199240986717,1.0,0.768500948767},{0.199240986717,1.0,0.768500948767},{0.211891208096,1.0,0.755850727388},{0.211891208096,1.0,0.755850727388},{0.224541429475,1.0,0.743200506009},{0.224541429475,1.0,0.743200506009},{0.237191650854,1.0,0.73055028463},{0.237191650854,1.0,0.73055028463},{0.249841872233,1.0,0.717900063251},{0.249841872233,1.0,0.717900063251},{0.262492093612,1.0,0.705249841872},{0.262492093612,1.0,0.705249841872},{0.275142314991,1.0,0.692599620493},{0.275142314991,1.0,0.692599620493},{0.287792536369,1.0,0.679949399114},{0.287792536369,1.0,0.679949399114},{0.300442757748,1.0,0.667299177736},{0.300442757748,1.0,0.667299177736},{0.313092979127,1.0,0.654648956357},{0.313092979127,1.0,0.654648956357},{0.325743200506,1.0,0.641998734978},{0.325743200506,1.0,0.641998734978},{0.338393421885,1.0,0.629348513599},{0.338393421885,1.0,0.629348513599},{0.351043643264,1.0,0.61669829222},{0.351043643264,1.0,0.61669829222},{0.363693864643,1.0,0.604048070841},{0.376344086022,1.0,0.591397849462},{0.376344086022,1.0,0.591397849462},{0.3889943074,1.0,0.578747628083},{0.3889943074,1.0,0.578747628083},{0.401644528779,1.0,0.566097406705},{0.401644528779,1.0,0.566097406705},{0.414294750158,1.0,0.553447185326},{0.414294750158,1.0,0.553447185326},{0.426944971537,1.0,0.540796963947},{0.426944971537,1.0,0.540796963947},{0.439595192916,1.0,0.528146742568},{0.439595192916,1.0,0.528146742568},{0.452245414295,1.0,0.515496521189},{0.452245414295,1.0,0.515496521189},{0.464895635674,1.0,0.50284629981},{0.464895635674,1.0,0.50284629981},{0.477545857052,1.0,0.490196078431},{0.477545857052,1.0,0.490196078431},{0.490196078431,1.0,0.477545857052},{0.490196078431,1.0,0.477545857052},{0.50284629981,1.0,0.464895635674},{0.50284629981,1.0,0.464895635674},{0.515496521189,1.0,0.452245414295},{0.515496521189,1.0,0.452245414295},{0.528146742568,1.0,0.439595192916},{0.528146742568,1.0,0.439595192916},{0.540796963947,1.0,0.426944971537},{0.540796963947,1.0,0.426944971537},{0.553447185326,1.0,0.414294750158},{0.553447185326,1.0,0.414294750158},{0.566097406705,1.0,0.401644528779},{0.566097406705,1.0,0.401644528779},{0.578747628083,1.0,0.3889943074},{0.578747628083,1.0,0.3889943074},{0.591397849462,1.0,0.376344086022},{0.591397849462,1.0,0.376344086022},{0.604048070841,1.0,0.363693864643},{0.61669829222,1.0,0.351043643264},{0.61669829222,1.0,0.351043643264},{0.629348513599,1.0,0.338393421885},{0.629348513599,1.0,0.338393421885},{0.641998734978,1.0,0.325743200506},{0.641998734978,1.0,0.325743200506},{0.654648956357,1.0,0.313092979127},{0.654648956357,1.0,0.313092979127},{0.667299177736,1.0,0.300442757748},{0.667299177736,1.0,0.300442757748},{0.679949399114,1.0,0.287792536369},{0.679949399114,1.0,0.287792536369},{0.692599620493,1.0,0.275142314991},{0.692599620493,1.0,0.275142314991},{0.705249841872,1.0,0.262492093612},{0.705249841872,1.0,0.262492093612},{0.717900063251,1.0,0.249841872233},{0.717900063251,1.0,0.249841872233},{0.73055028463,1.0,0.237191650854},{0.73055028463,1.0,0.237191650854},{0.743200506009,1.0,0.224541429475},{0.743200506009,1.0,0.224541429475},{0.755850727388,1.0,0.211891208096},{0.755850727388,1.0,0.211891208096},{0.768500948767,1.0,0.199240986717},{0.768500948767,1.0,0.199240986717},{0.781151170145,1.0,0.186590765338},{0.781151170145,1.0,0.186590765338},{0.793801391524,1.0,0.17394054396},{0.793801391524,1.0,0.17394054396},{0.806451612903,1.0,0.161290322581},{0.806451612903,1.0,0.161290322581},{0.819101834282,1.0,0.148640101202},{0.819101834282,1.0,0.148640101202},{0.831752055661,1.0,0.135989879823},{0.831752055661,1.0,0.135989879823},{0.84440227704,1.0,0.123339658444},{0.84440227704,1.0,0.123339658444},{0.857052498419,1.0,0.110689437065},{0.869702719798,1.0,0.0980392156863},{0.869702719798,1.0,0.0980392156863},{0.882352941176,1.0,0.0853889943074},{0.882352941176,1.0,0.0853889943074},{0.895003162555,1.0,0.0727387729285},{0.895003162555,1.0,0.0727387729285},{0.907653383934,1.0,0.0600885515497},{0.907653383934,1.0,0.0600885515497},{0.920303605313,1.0,0.0474383301708},{0.920303605313,1.0,0.0474383301708},{0.932953826692,1.0,0.0347881087919},{0.932953826692,1.0,0.0347881087919},{0.945604048071,0.9883805374,0.022137887413},{0.945604048071,0.9883805374,0.022137887413},{0.95825426945,0.97385620915,0.00948766603416},{0.95825426945,0.97385620915,0.00948766603416},{0.970904490829,0.959331880901,0.0},{0.970904490829,0.959331880901,0.0},{0.983554712207,0.944807552651,0.0},{0.983554712207,0.944807552651,0.0},{0.996204933586,0.930283224401,0.0},{0.996204933586,0.930283224401,0.0},{1.0,0.915758896151,0.0},{1.0,0.915758896151,0.0},{1.0,0.901234567901,0.0},{1.0,0.901234567901,0.0},{1.0,0.886710239651,0.0},{1.0,0.886710239651,0.0},{1.0,0.872185911402,0.0},{1.0,0.872185911402,0.0},{1.0,0.857661583152,0.0},{1.0,0.857661583152,0.0},{1.0,0.843137254902,0.0},{1.0,0.843137254902,0.0},{1.0,0.828612926652,0.0},{1.0,0.828612926652,0.0},{1.0,0.814088598402,0.0},{1.0,0.814088598402,0.0},{1.0,0.799564270153,0.0},{1.0,0.785039941903,0.0},{1.0,0.785039941903,0.0},{1.0,0.770515613653,0.0},{1.0,0.770515613653,0.0},{1.0,0.755991285403,0.0},{1.0,0.755991285403,0.0},{1.0,0.741466957153,0.0},{1.0,0.741466957153,0.0},{1.0,0.726942628903,0.0},{1.0,0.726942628903,0.0},{1.0,0.712418300654,0.0},{1.0,0.712418300654,0.0},{1.0,0.697893972404,0.0},{1.0,0.697893972404,0.0},{1.0,0.683369644154,0.0},{1.0,0.683369644154,0.0},{1.0,0.668845315904,0.0},{1.0,0.668845315904,0.0},{1.0,0.654320987654,0.0},{1.0,0.654320987654,0.0},{1.0,0.639796659405,0.0},{1.0,0.639796659405,0.0},{1.0,0.625272331155,0.0},{1.0,0.625272331155,0.0},{1.0,0.610748002905,0.0},{1.0,0.610748002905,0.0},{1.0,0.596223674655,0.0},{1.0,0.596223674655,0.0},{1.0,0.581699346405,0.0},{1.0,0.581699346405,0.0},{1.0,0.567175018155,0.0},{1.0,0.567175018155,0.0},{1.0,0.552650689906,0.0},{1.0,0.552650689906,0.0},{1.0,0.538126361656,0.0},{1.0,0.538126361656,0.0},{1.0,0.523602033406,0.0},{1.0,0.509077705156,0.0},{1.0,0.509077705156,0.0},{1.0,0.494553376906,0.0},{1.0,0.494553376906,0.0},{1.0,0.480029048656,0.0},{1.0,0.480029048656,0.0},{1.0,0.465504720407,0.0},{1.0,0.465504720407,0.0},{1.0,0.450980392157,0.0},{1.0,0.450980392157,0.0},{1.0,0.436456063907,0.0},{1.0,0.436456063907,0.0},{1.0,0.421931735657,0.0},{1.0,0.421931735657,0.0},{1.0,0.407407407407,0.0},{1.0,0.407407407407,0.0},{1.0,0.392883079158,0.0},{1.0,0.392883079158,0.0},{1.0,0.378358750908,0.0},{1.0,0.378358750908,0.0},{1.0,0.363834422658,0.0},{1.0,0.363834422658,0.0},{1.0,0.349310094408,0.0},{1.0,0.349310094408,0.0},{1.0,0.334785766158,0.0},{1.0,0.334785766158,0.0},{1.0,0.320261437908,0.0},{1.0,0.320261437908,0.0},{1.0,0.305737109659,0.0},{1.0,0.305737109659,0.0},{1.0,0.291212781409,0.0},{1.0,0.291212781409,0.0},{1.0,0.276688453159,0.0},{1.0,0.276688453159,0.0},{1.0,0.262164124909,0.0},{1.0,0.262164124909,0.0},{1.0,0.247639796659,0.0},{1.0,0.247639796659,0.0},{1.0,0.23311546841,0.0},{1.0,0.21859114016,0.0},{1.0,0.21859114016,0.0},{1.0,0.20406681191,0.0},{1.0,0.20406681191,0.0},{1.0,0.18954248366,0.0},{1.0,0.18954248366,0.0},{1.0,0.17501815541,0.0},{1.0,0.17501815541,0.0},{1.0,0.16049382716,0.0},{1.0,0.16049382716,0.0},{1.0,0.145969498911,0.0},{1.0,0.145969498911,0.0},{1.0,0.131445170661,0.0},{1.0,0.131445170661,0.0},{1.0,0.116920842411,0.0},{1.0,0.116920842411,0.0},{1.0,0.102396514161,0.0},{1.0,0.102396514161,0.0},{1.0,0.0878721859114,0.0},{1.0,0.0878721859114,0.0},{0.999108734403,0.0733478576616,0.0},{0.999108734403,0.0733478576616,0.0},{0.98128342246,0.0588235294118,0.0},{0.98128342246,0.0588235294118,0.0},{0.963458110517,0.0442992011619,0.0},{0.963458110517,0.0442992011619,0.0},{0.945632798574,0.0297748729121,0.0},{0.945632798574,0.0297748729121,0.0},{0.927807486631,0.0152505446623,0.0},{0.927807486631,0.0152505446623,0.0},{0.909982174688,0.000726216412491,0.0},{0.909982174688,0.000726216412491,0.0},{0.892156862745,0.0,0.0},{0.892156862745,0.0,0.0},{0.874331550802,0.0,0.0},{0.874331550802,0.0,0.0},{0.856506238859,0.0,0.0},{0.856506238859,0.0,0.0},{0.838680926916,0.0,0.0},{0.820855614973,0.0,0.0},{0.820855614973,0.0,0.0},{0.80303030303,0.0,0.0},{0.80303030303,0.0,0.0},{0.785204991087,0.0,0.0},{0.785204991087,0.0,0.0},{0.767379679144,0.0,0.0},{0.767379679144,0.0,0.0},{0.749554367201,0.0,0.0},{0.749554367201,0.0,0.0},{0.731729055258,0.0,0.0},{0.731729055258,0.0,0.0},{0.713903743316,0.0,0.0},{0.713903743316,0.0,0.0},{0.696078431373,0.0,0.0},{0.696078431373,0.0,0.0},{0.67825311943,0.0,0.0},{0.67825311943,0.0,0.0},{0.660427807487,0.0,0.0},{0.660427807487,0.0,0.0},{0.642602495544,0.0,0.0},{0.642602495544,0.0,0.0},{0.624777183601,0.0,0.0},{0.624777183601,0.0,0.0},{0.606951871658,0.0,0.0},{0.606951871658,0.0,0.0},{0.589126559715,0.0,0.0},{0.589126559715,0.0,0.0},{0.571301247772,0.0,0.0},{0.571301247772,0.0,0.0},{0.553475935829,0.0,0.0},{0.553475935829,0.0,0.0},{0.535650623886,0.0,0.0},{0.535650623886,0.0,0.0},{0.517825311943,0.0,0.0},{0.517825311943,0.0,0.0},{0.5,0.0,0.0},{0.5,0.0,0.0},};

const std::vector<glm::vec3> CM_TURBO = 
{{0.18995,0.07176,0.23217},{0.19483,0.08339,0.26149},{0.19956,0.09498,0.29024},{0.20415,0.10652,0.31844},{0.20860,0.11802,0.34607},{0.21291,0.12947,0.37314},{0.21708,0.14087,0.39964},{0.22111,0.15223,0.42558},{0.22500,0.16354,0.45096},{0.22875,0.17481,0.47578},{0.23236,0.18603,0.50004},{0.23582,0.19720,0.52373},{0.23915,0.20833,0.54686},{0.24234,0.21941,0.56942},{0.24539,0.23044,0.59142},{0.24830,0.24143,0.61286},{0.25107,0.25237,0.63374},{0.25369,0.26327,0.65406},{0.25618,0.27412,0.67381},{0.25853,0.28492,0.69300},{0.26074,0.29568,0.71162},{0.26280,0.30639,0.72968},{0.26473,0.31706,0.74718},{0.26652,0.32768,0.76412},{0.26816,0.33825,0.78050},{0.26967,0.34878,0.79631},{0.27103,0.35926,0.81156},{0.27226,0.36970,0.82624},{0.27334,0.38008,0.84037},{0.27429,0.39043,0.85393},{0.27509,0.40072,0.86692},{0.27576,0.41097,0.87936},{0.27628,0.42118,0.89123},{0.27667,0.43134,0.90254},{0.27691,0.44145,0.91328},{0.27701,0.45152,0.92347},{0.27698,0.46153,0.93309},{0.27680,0.47151,0.94214},{0.27648,0.48144,0.95064},{0.27603,0.49132,0.95857},{0.27543,0.50115,0.96594},{0.27469,0.51094,0.97275},{0.27381,0.52069,0.97899},{0.27273,0.53040,0.98461},{0.27106,0.54015,0.98930},{0.26878,0.54995,0.99303},{0.26592,0.55979,0.99583},{0.26252,0.56967,0.99773},{0.25862,0.57958,0.99876},{0.25425,0.58950,0.99896},{0.24946,0.59943,0.99835},{0.24427,0.60937,0.99697},{0.23874,0.61931,0.99485},{0.23288,0.62923,0.99202},{0.22676,0.63913,0.98851},{0.22039,0.64901,0.98436},{0.21382,0.65886,0.97959},{0.20708,0.66866,0.97423},{0.20021,0.67842,0.96833},{0.19326,0.68812,0.96190},{0.18625,0.69775,0.95498},{0.17923,0.70732,0.94761},{0.17223,0.71680,0.93981},{0.16529,0.72620,0.93161},{0.15844,0.73551,0.92305},{0.15173,0.74472,0.91416},{0.14519,0.75381,0.90496},{0.13886,0.76279,0.89550},{0.13278,0.77165,0.88580},{0.12698,0.78037,0.87590},{0.12151,0.78896,0.86581},{0.11639,0.79740,0.85559},{0.11167,0.80569,0.84525},{0.10738,0.81381,0.83484},{0.10357,0.82177,0.82437},{0.10026,0.82955,0.81389},{0.09750,0.83714,0.80342},{0.09532,0.84455,0.79299},{0.09377,0.85175,0.78264},{0.09287,0.85875,0.77240},{0.09267,0.86554,0.76230},{0.09320,0.87211,0.75237},{0.09451,0.87844,0.74265},{0.09662,0.88454,0.73316},{0.09958,0.89040,0.72393},{0.10342,0.89600,0.71500},{0.10815,0.90142,0.70599},{0.11374,0.90673,0.69651},{0.12014,0.91193,0.68660},{0.12733,0.91701,0.67627},{0.13526,0.92197,0.66556},{0.14391,0.92680,0.65448},{0.15323,0.93151,0.64308},{0.16319,0.93609,0.63137},{0.17377,0.94053,0.61938},{0.18491,0.94484,0.60713},{0.19659,0.94901,0.59466},{0.20877,0.95304,0.58199},{0.22142,0.95692,0.56914},{0.23449,0.96065,0.55614},{0.24797,0.96423,0.54303},{0.26180,0.96765,0.52981},{0.27597,0.97092,0.51653},{0.29042,0.97403,0.50321},{0.30513,0.97697,0.48987},{0.32006,0.97974,0.47654},{0.33517,0.98234,0.46325},{0.35043,0.98477,0.45002},{0.36581,0.98702,0.43688},{0.38127,0.98909,0.42386},{0.39678,0.99098,0.41098},{0.41229,0.99268,0.39826},{0.42778,0.99419,0.38575},{0.44321,0.99551,0.37345},{0.45854,0.99663,0.36140},{0.47375,0.99755,0.34963},{0.48879,0.99828,0.33816},{0.50362,0.99879,0.32701},{0.51822,0.99910,0.31622},{0.53255,0.99919,0.30581},{0.54658,0.99907,0.29581},{0.56026,0.99873,0.28623},{0.57357,0.99817,0.27712},{0.58646,0.99739,0.26849},{0.59891,0.99638,0.26038},{0.61088,0.99514,0.25280},{0.62233,0.99366,0.24579},{0.63323,0.99195,0.23937},{0.64362,0.98999,0.23356},{0.65394,0.98775,0.22835},{0.66428,0.98524,0.22370},{0.67462,0.98246,0.21960},{0.68494,0.97941,0.21602},{0.69525,0.97610,0.21294},{0.70553,0.97255,0.21032},{0.71577,0.96875,0.20815},{0.72596,0.96470,0.20640},{0.73610,0.96043,0.20504},{0.74617,0.95593,0.20406},{0.75617,0.95121,0.20343},{0.76608,0.94627,0.20311},{0.77591,0.94113,0.20310},{0.78563,0.93579,0.20336},{0.79524,0.93025,0.20386},{0.80473,0.92452,0.20459},{0.81410,0.91861,0.20552},{0.82333,0.91253,0.20663},{0.83241,0.90627,0.20788},{0.84133,0.89986,0.20926},{0.85010,0.89328,0.21074},{0.85868,0.88655,0.21230},{0.86709,0.87968,0.21391},{0.87530,0.87267,0.21555},{0.88331,0.86553,0.21719},{0.89112,0.85826,0.21880},{0.89870,0.85087,0.22038},{0.90605,0.84337,0.22188},{0.91317,0.83576,0.22328},{0.92004,0.82806,0.22456},{0.92666,0.82025,0.22570},{0.93301,0.81236,0.22667},{0.93909,0.80439,0.22744},{0.94489,0.79634,0.22800},{0.95039,0.78823,0.22831},{0.95560,0.78005,0.22836},{0.96049,0.77181,0.22811},{0.96507,0.76352,0.22754},{0.96931,0.75519,0.22663},{0.97323,0.74682,0.22536},{0.97679,0.73842,0.22369},{0.98000,0.73000,0.22161},{0.98289,0.72140,0.21918},{0.98549,0.71250,0.21650},{0.98781,0.70330,0.21358},{0.98986,0.69382,0.21043},{0.99163,0.68408,0.20706},{0.99314,0.67408,0.20348},{0.99438,0.66386,0.19971},{0.99535,0.65341,0.19577},{0.99607,0.64277,0.19165},{0.99654,0.63193,0.18738},{0.99675,0.62093,0.18297},{0.99672,0.60977,0.17842},{0.99644,0.59846,0.17376},{0.99593,0.58703,0.16899},{0.99517,0.57549,0.16412},{0.99419,0.56386,0.15918},{0.99297,0.55214,0.15417},{0.99153,0.54036,0.14910},{0.98987,0.52854,0.14398},{0.98799,0.51667,0.13883},{0.98590,0.50479,0.13367},{0.98360,0.49291,0.12849},{0.98108,0.48104,0.12332},{0.97837,0.46920,0.11817},{0.97545,0.45740,0.11305},{0.97234,0.44565,0.10797},{0.96904,0.43399,0.10294},{0.96555,0.42241,0.09798},{0.96187,0.41093,0.09310},{0.95801,0.39958,0.08831},{0.95398,0.38836,0.08362},{0.94977,0.37729,0.07905},{0.94538,0.36638,0.07461},{0.94084,0.35566,0.07031},{0.93612,0.34513,0.06616},{0.93125,0.33482,0.06218},{0.92623,0.32473,0.05837},{0.92105,0.31489,0.05475},{0.91572,0.30530,0.05134},{0.91024,0.29599,0.04814},{0.90463,0.28696,0.04516},{0.89888,0.27824,0.04243},{0.89298,0.26981,0.03993},{0.88691,0.26152,0.03753},{0.88066,0.25334,0.03521},{0.87422,0.24526,0.03297},{0.86760,0.23730,0.03082},{0.86079,0.22945,0.02875},{0.85380,0.22170,0.02677},{0.84662,0.21407,0.02487},{0.83926,0.20654,0.02305},{0.83172,0.19912,0.02131},{0.82399,0.19182,0.01966},{0.81608,0.18462,0.01809},{0.80799,0.17753,0.01660},{0.79971,0.17055,0.01520},{0.79125,0.16368,0.01387},{0.78260,0.15693,0.01264},{0.77377,0.15028,0.01148},{0.76476,0.14374,0.01041},{0.75556,0.13731,0.00942},{0.74617,0.13098,0.00851},{0.73661,0.12477,0.00769},{0.72686,0.11867,0.00695},{0.71692,0.11268,0.00629},{0.70680,0.10680,0.00571},{0.69650,0.10102,0.00522},{0.68602,0.09536,0.00481},{0.67535,0.08980,0.00449},{0.66449,0.08436,0.00424},{0.65345,0.07902,0.00408},{0.64223,0.07380,0.00401},{0.63082,0.06868,0.00401},{0.61923,0.06367,0.00410},{0.60746,0.05878,0.00427},{0.59550,0.05399,0.00453},{0.58336,0.04931,0.00486},{0.57103,0.04474,0.00529},{0.55852,0.04028,0.00579},{0.54583,0.03593,0.00638},{0.53295,0.03169,0.00705},{0.51989,0.02756,0.00780},{0.50664,0.02354,0.00863},{0.49321,0.01963,0.00955},{0.47960,0.01583,0.01055}};

const std::vector<glm::vec3> CM_REDS =
{{1.0,0.960784316063,0.941176474094},{1.0,0.960784316063,0.941176474094},{0.999876970403,0.958200694533,0.937485586194},{0.999876970403,0.958200694533,0.937485586194},{0.999753940807,0.955617073003,0.933794698294},{0.999753940807,0.955617073003,0.933794698294},{0.99963091121,0.953033451473,0.930103810395},{0.99963091121,0.953033451473,0.930103810395},{0.999507881613,0.950449829943,0.926412922495},{0.999507881613,0.950449829943,0.926412922495},{0.999384852017,0.947866208413,0.922722034595},{0.999384852017,0.947866208413,0.922722034595},{0.99926182242,0.945282586883,0.919031146695},{0.99926182242,0.945282586883,0.919031146695},{0.999138792823,0.942698965353,0.915340258795},{0.999138792823,0.942698965353,0.915340258795},{0.999015763227,0.940115343823,0.911649370895},{0.999015763227,0.940115343823,0.911649370895},{0.99889273363,0.937531722293,0.907958482995},{0.99889273363,0.937531722293,0.907958482995},{0.998769704033,0.934948100763,0.904267595095},{0.998769704033,0.934948100763,0.904267595095},{0.998646674437,0.932364479233,0.900576707195},{0.998646674437,0.932364479233,0.900576707195},{0.99852364484,0.929780857703,0.896885819295},{0.99852364484,0.929780857703,0.896885819295},{0.998400615243,0.927197236173,0.893194931395},{0.998400615243,0.927197236173,0.893194931395},{0.998277585647,0.924613614643,0.889504043495},{0.998277585647,0.924613614643,0.889504043495},{0.99815455605,0.922029993113,0.885813155595},{0.99815455605,0.922029993113,0.885813155595},{0.998031526453,0.919446371583,0.882122267695},{0.998031526453,0.919446371583,0.882122267695},{0.997908496857,0.916862750053,0.878431379795},{0.997908496857,0.916862750053,0.878431379795},{0.99778546726,0.914279128523,0.874740491895},{0.99778546726,0.914279128523,0.874740491895},{0.997662437663,0.911695506993,0.871049603995},{0.997539408067,0.909111885463,0.867358716095},{0.997539408067,0.909111885463,0.867358716095},{0.99741637847,0.906528263934,0.863667828195},{0.99741637847,0.906528263934,0.863667828195},{0.997293348873,0.903944642404,0.859976940295},{0.997293348873,0.903944642404,0.859976940295},{0.997170319277,0.901361020874,0.856286052395},{0.997170319277,0.901361020874,0.856286052395},{0.99704728968,0.898777399344,0.852595164495},{0.99704728968,0.898777399344,0.852595164495},{0.996924260083,0.896193777814,0.848904276595},{0.996924260083,0.896193777814,0.848904276595},{0.996801230487,0.893610156284,0.845213388695},{0.996801230487,0.893610156284,0.845213388695},{0.99667820089,0.891026534754,0.841522500795},{0.99667820089,0.891026534754,0.841522500795},{0.996555171293,0.888442913224,0.837831612896},{0.996555171293,0.888442913224,0.837831612896},{0.996432141697,0.885859291694,0.834140724996},{0.996432141697,0.885859291694,0.834140724996},{0.9963091121,0.883275670164,0.830449837096},{0.9963091121,0.883275670164,0.830449837096},{0.996186082503,0.880692048634,0.826758949196},{0.996186082503,0.880692048634,0.826758949196},{0.996047674207,0.87786236791,0.822775866004},{0.996047674207,0.87786236791,0.822775866004},{0.995801615014,0.873310272834,0.816747415767},{0.995801615014,0.873310272834,0.816747415767},{0.99555555582,0.868758177757,0.81071896553},{0.99555555582,0.868758177757,0.81071896553},{0.995309496627,0.864206082681,0.804690515294},{0.995309496627,0.864206082681,0.804690515294},{0.995063437434,0.859653987604,0.798662065057},{0.995063437434,0.859653987604,0.798662065057},{0.99481737824,0.855101892527,0.792633614821},{0.99481737824,0.855101892527,0.792633614821},{0.994571319047,0.850549797451,0.786605164584},{0.994571319047,0.850549797451,0.786605164584},{0.994325259854,0.845997702374,0.780576714347},{0.99407920066,0.841445607298,0.774548264111},{0.99407920066,0.841445607298,0.774548264111},{0.993833141467,0.836893512221,0.768519813874},{0.993833141467,0.836893512221,0.768519813874},{0.993587082274,0.832341417144,0.762491363638},{0.993587082274,0.832341417144,0.762491363638},{0.99334102308,0.827789322068,0.756462913401},{0.99334102308,0.827789322068,0.756462913401},{0.993094963887,0.823237226991,0.750434463164},{0.993094963887,0.823237226991,0.750434463164},{0.992848904694,0.818685131914,0.744406012928},{0.992848904694,0.818685131914,0.744406012928},{0.9926028455,0.814133036838,0.738377562691},{0.9926028455,0.814133036838,0.738377562691},{0.992356786307,0.809580941761,0.732349112455},{0.992356786307,0.809580941761,0.732349112455},{0.992110727114,0.805028846685,0.726320662218},{0.992110727114,0.805028846685,0.726320662218},{0.991864667921,0.800476751608,0.720292211981},{0.991864667921,0.800476751608,0.720292211981},{0.991618608727,0.795924656531,0.714263761745},{0.991618608727,0.795924656531,0.714263761745},{0.991372549534,0.791372561455,0.708235311508},{0.991372549534,0.791372561455,0.708235311508},{0.991126490341,0.786820466378,0.702206861272},{0.991126490341,0.786820466378,0.702206861272},{0.990880431147,0.782268371302,0.696178411035},{0.990880431147,0.782268371302,0.696178411035},{0.990634371954,0.777716276225,0.690149960798},{0.990634371954,0.777716276225,0.690149960798},{0.990388312761,0.773164181148,0.684121510562},{0.990388312761,0.773164181148,0.684121510562},{0.990142253567,0.768612086072,0.678093060325},{0.990142253567,0.768612086072,0.678093060325},{0.989896194374,0.764059990995,0.672064610089},{0.989896194374,0.764059990995,0.672064610089},{0.989650135181,0.759507895918,0.666036159852},{0.989650135181,0.759507895918,0.666036159852},{0.989404075987,0.754955800842,0.660007709615},{0.989158016794,0.750403705765,0.653979259379},{0.989158016794,0.750403705765,0.653979259379},{0.988911957601,0.745851610689,0.647950809142},{0.988911957601,0.745851610689,0.647950809142},{0.988665898407,0.741299515612,0.641922358906},{0.988665898407,0.741299515612,0.641922358906},{0.988419839214,0.736747420535,0.635893908669},{0.988419839214,0.736747420535,0.635893908669},{0.988235294819,0.732072295862,0.629926972997},{0.988235294819,0.732072295862,0.629926972997},{0.988235294819,0.727028082399,0.624144581019},{0.988235294819,0.727028082399,0.624144581019},{0.988235294819,0.721983868936,0.61836218904},{0.988235294819,0.721983868936,0.61836218904},{0.988235294819,0.716939655472,0.612579797062},{0.988235294819,0.716939655472,0.612579797062},{0.988235294819,0.711895442009,0.606797405084},{0.988235294819,0.711895442009,0.606797405084},{0.988235294819,0.706851228546,0.601015013106},{0.988235294819,0.706851228546,0.601015013106},{0.988235294819,0.701807015082,0.595232621127},{0.988235294819,0.701807015082,0.595232621127},{0.988235294819,0.696762801619,0.589450229149},{0.988235294819,0.696762801619,0.589450229149},{0.988235294819,0.691718588156,0.583667837171},{0.988235294819,0.691718588156,0.583667837171},{0.988235294819,0.686674374693,0.577885445193},{0.988235294819,0.686674374693,0.577885445193},{0.988235294819,0.681630161229,0.572103053215},{0.988235294819,0.681630161229,0.572103053215},{0.988235294819,0.676585947766,0.566320661236},{0.988235294819,0.676585947766,0.566320661236},{0.988235294819,0.671541734303,0.560538269258},{0.988235294819,0.671541734303,0.560538269258},{0.988235294819,0.666497520839,0.55475587728},{0.988235294819,0.666497520839,0.55475587728},{0.988235294819,0.661453307376,0.548973485302},{0.988235294819,0.656409093913,0.543191093323},{0.988235294819,0.656409093913,0.543191093323},{0.988235294819,0.65136488045,0.537408701345},{0.988235294819,0.65136488045,0.537408701345},{0.988235294819,0.646320666986,0.531626309367},{0.988235294819,0.646320666986,0.531626309367},{0.988235294819,0.641276453523,0.525843917389},{0.988235294819,0.641276453523,0.525843917389},{0.988235294819,0.63623224006,0.52006152541},{0.988235294819,0.63623224006,0.52006152541},{0.988235294819,0.631188026597,0.514279133432},{0.988235294819,0.631188026597,0.514279133432},{0.988235294819,0.626143813133,0.508496741454},{0.988235294819,0.626143813133,0.508496741454},{0.988235294819,0.62109959967,0.502714349476},{0.988235294819,0.62109959967,0.502714349476},{0.988235294819,0.616055386207,0.496931957497},{0.988235294819,0.616055386207,0.496931957497},{0.988235294819,0.611011172743,0.491149565519},{0.988235294819,0.611011172743,0.491149565519},{0.988235294819,0.60596695928,0.485367173541},{0.988235294819,0.60596695928,0.485367173541},{0.988235294819,0.600922745817,0.479584781563},{0.988235294819,0.600922745817,0.479584781563},{0.988235294819,0.595878532354,0.473802389584},{0.988235294819,0.595878532354,0.473802389584},{0.988235294819,0.59083431889,0.468019997606},{0.988235294819,0.59083431889,0.468019997606},{0.988235294819,0.585790105427,0.462237605628},{0.988235294819,0.585790105427,0.462237605628},{0.988235294819,0.580745891964,0.45645521365},{0.988235294819,0.580745891964,0.45645521365},{0.988235294819,0.5757016785,0.450672821671},{0.988235294819,0.5757016785,0.450672821671},{0.98818915872,0.570703600785,0.445213382735},{0.98818915872,0.570703600785,0.445213382735},{0.988066129123,0.565782415984,0.440292198868},{0.988066129123,0.565782415984,0.440292198868},{0.987943099527,0.560861231182,0.435371015002},{0.98782006993,0.555940046381,0.430449831135},{0.98782006993,0.555940046381,0.430449831135},{0.987697040333,0.551018861579,0.425528647269},{0.987697040333,0.551018861579,0.425528647269},{0.987574010737,0.546097676777,0.420607463402},{0.987574010737,0.546097676777,0.420607463402},{0.98745098114,0.541176491976,0.415686279535},{0.98745098114,0.541176491976,0.415686279535},{0.987327951543,0.536255307174,0.410765095669},{0.987327951543,0.536255307174,0.410765095669},{0.987204921947,0.531334122373,0.405843911802},{0.987204921947,0.531334122373,0.405843911802},{0.98708189235,0.526412937571,0.400922727935},{0.98708189235,0.526412937571,0.400922727935},{0.986958862753,0.521491752769,0.396001544069},{0.986958862753,0.521491752769,0.396001544069},{0.986835833157,0.516570567968,0.391080360202},{0.986835833157,0.516570567968,0.391080360202},{0.98671280356,0.511649383166,0.386159176336},{0.98671280356,0.511649383166,0.386159176336},{0.986589773963,0.506728198365,0.381237992469},{0.986589773963,0.506728198365,0.381237992469},{0.986466744367,0.501807013563,0.376316808602},{0.986466744367,0.501807013563,0.376316808602},{0.98634371477,0.496885828761,0.371395624736},{0.98634371477,0.496885828761,0.371395624736},{0.986220685173,0.49196464396,0.366474440869},{0.986220685173,0.49196464396,0.366474440869},{0.986097655577,0.487043459158,0.361553257003},{0.986097655577,0.487043459158,0.361553257003},{0.98597462598,0.482122274357,0.356632073136},{0.98597462598,0.482122274357,0.356632073136},{0.985851596383,0.477201089555,0.351710889269},{0.985851596383,0.477201089555,0.351710889269},{0.985728566787,0.472279904754,0.346789705403},{0.985728566787,0.472279904754,0.346789705403},{0.98560553719,0.467358719952,0.341868521536},{0.98560553719,0.467358719952,0.341868521536},{0.985482507593,0.46243753515,0.336947337669},{0.985359477997,0.457516350349,0.332026153803},{0.985359477997,0.457516350349,0.332026153803},{0.9852364484,0.452595165547,0.327104969936},{0.9852364484,0.452595165547,0.327104969936},{0.985113418803,0.447673980746,0.32218378607},{0.985113418803,0.447673980746,0.32218378607},{0.984990389207,0.442752795944,0.317262602203},{0.984990389207,0.442752795944,0.317262602203},{0.98486735961,0.437831611142,0.312341418336},{0.98486735961,0.437831611142,0.312341418336},{0.984744330013,0.432910426341,0.30742023447},{0.984744330013,0.432910426341,0.30742023447},{0.984621300417,0.427989241539,0.302499050603},{0.984621300417,0.427989241539,0.302499050603},{0.98449827082,0.423068056738,0.297577866737},{0.98449827082,0.423068056738,0.297577866737},{0.984375241224,0.418146871936,0.29265668287},{0.984375241224,0.418146871936,0.29265668287},{0.983575548845,0.41279508378,0.288350646753},{0.983575548845,0.41279508378,0.288350646753},{0.982099193685,0.407012692269,0.284659758385},{0.982099193685,0.407012692269,0.284659758385},{0.980622838525,0.401230300758,0.280968870018},{0.980622838525,0.401230300758,0.280968870018},{0.979146483365,0.395447909248,0.277277981651},{0.979146483365,0.395447909248,0.277277981651},{0.977670128205,0.389665517737,0.273587093283},{0.977670128205,0.389665517737,0.273587093283},{0.976193773045,0.383883126226,0.269896204916},{0.976193773045,0.383883126226,0.269896204916},{0.974717417885,0.378100734715,0.266205316548},{0.974717417885,0.378100734715,0.266205316548},{0.973241062725,0.372318343205,0.262514428181},{0.973241062725,0.372318343205,0.262514428181},{0.971764707565,0.366535951694,0.258823539813},{0.971764707565,0.366535951694,0.258823539813},{0.970288352405,0.360753560183,0.255132651446},{0.968811997245,0.354971168672,0.251441763078},{0.968811997245,0.354971168672,0.251441763078},{0.967335642085,0.349188777162,0.247750874711},{0.967335642085,0.349188777162,0.247750874711},{0.965859286925,0.343406385651,0.244059986344},{0.965859286925,0.343406385651,0.244059986344},{0.964382931765,0.33762399414,0.240369097976},{0.964382931765,0.33762399414,0.240369097976},{0.962906576605,0.331841602629,0.236678209609},{0.962906576605,0.331841602629,0.236678209609},{0.961430221445,0.326059211119,0.232987321241},{0.961430221445,0.326059211119,0.232987321241},{0.959953866285,0.320276819608,0.229296432874},{0.959953866285,0.320276819608,0.229296432874},{0.958477511125,0.314494428097,0.225605544506},{0.958477511125,0.314494428097,0.225605544506},{0.957001155965,0.308712036586,0.221914656139},{0.957001155965,0.308712036586,0.221914656139},{0.955524800805,0.302929645076,0.218223767771},{0.955524800805,0.302929645076,0.218223767771},{0.954048445646,0.297147253565,0.214532879404},{0.954048445646,0.297147253565,0.214532879404},{0.952572090486,0.291364862054,0.210841991037},{0.952572090486,0.291364862054,0.210841991037},{0.951095735326,0.285582470543,0.207151102669},{0.951095735326,0.285582470543,0.207151102669},{0.949619380166,0.279800079032,0.203460214302},{0.949619380166,0.279800079032,0.203460214302},{0.948143025006,0.274017687522,0.199769325934},{0.948143025006,0.274017687522,0.199769325934},{0.946666669846,0.268235296011,0.196078437567},{0.946666669846,0.268235296011,0.196078437567},{0.945190314686,0.2624529045,0.192387549199},{0.945190314686,0.2624529045,0.192387549199},{0.943713959526,0.256670512989,0.188696660832},{0.943713959526,0.256670512989,0.188696660832},{0.942237604366,0.250888121479,0.185005772464},{0.942237604366,0.250888121479,0.185005772464},{0.940761249206,0.245105729968,0.181314884097},{0.939284894046,0.239323338457,0.17762399573},{0.939284894046,0.239323338457,0.17762399573},{0.937808538886,0.233540946946,0.173933107362},{0.937808538886,0.233540946946,0.173933107362},{0.934486739776,0.228681277557,0.171395621609},{0.934486739776,0.228681277557,0.171395621609},{0.930057674296,0.22437524144,0.169550177426},{0.930057674296,0.22437524144,0.169550177426},{0.925628608816,0.220069205323,0.167704733242},{0.925628608816,0.220069205323,0.167704733242},{0.921199543336,0.215763169206,0.165859289058},{0.921199543336,0.215763169206,0.165859289058},{0.916770477856,0.211457133089,0.164013844875},{0.916770477856,0.211457133089,0.164013844875},{0.912341412376,0.207151096972,0.162168400691},{0.912341412376,0.207151096972,0.162168400691},{0.907912346896,0.202845060855,0.160322956507},{0.907912346896,0.202845060855,0.160322956507},{0.903483281416,0.198539024738,0.158477512323},{0.903483281416,0.198539024738,0.158477512323},{0.899054215936,0.194232988621,0.15663206814},{0.899054215936,0.194232988621,0.15663206814},{0.894625150456,0.189926952503,0.154786623956},{0.894625150456,0.189926952503,0.154786623956},{0.890196084976,0.185620916386,0.152941179772},{0.890196084976,0.185620916386,0.152941179772},{0.885767019496,0.181314880269,0.151095735588},{0.885767019496,0.181314880269,0.151095735588},{0.881337954016,0.177008844152,0.149250291405},{0.881337954016,0.177008844152,0.149250291405},{0.876908888536,0.172702808035,0.147404847221},{0.876908888536,0.172702808035,0.147404847221},{0.872479823056,0.168396771918,0.145559403037},{0.872479823056,0.168396771918,0.145559403037},{0.868050757576,0.164090735801,0.143713958854},{0.868050757576,0.164090735801,0.143713958854},{0.863621692096,0.159784699684,0.14186851467},{0.863621692096,0.159784699684,0.14186851467},{0.859192626617,0.155478663567,0.140023070486},{0.854763561137,0.15117262745,0.138177626302},{0.854763561137,0.15117262745,0.138177626302},{0.850334495657,0.146866591333,0.136332182119},{0.850334495657,0.146866591333,0.136332182119},{0.845905430177,0.142560555216,0.134486737935},{0.845905430177,0.142560555216,0.134486737935},{0.841476364697,0.138254519099,0.132641293751},{0.841476364697,0.138254519099,0.132641293751},{0.837047299217,0.133948482982,0.130795849568},{0.837047299217,0.133948482982,0.130795849568},{0.832618233737,0.129642446865,0.128950405384},{0.832618233737,0.129642446865,0.128950405384},{0.828189168257,0.125336410748,0.1271049612},{0.828189168257,0.125336410748,0.1271049612},{0.823760102777,0.121030374631,0.125259517016},{0.823760102777,0.121030374631,0.125259517016},{0.819331037297,0.116724338514,0.123414072833},{0.819331037297,0.116724338514,0.123414072833},{0.814901971817,0.112418302397,0.121568628649},{0.814901971817,0.112418302397,0.121568628649},{0.810472906337,0.10811226628,0.119723184465},{0.810472906337,0.10811226628,0.119723184465},{0.806043840857,0.103806230163,0.117877740281},{0.806043840857,0.103806230163,0.117877740281},{0.801614775377,0.0995001940458,0.116032296098},{0.801614775377,0.0995001940458,0.116032296098},{0.797185709897,0.0951941579288,0.114186851914},{0.797185709897,0.0951941579288,0.114186851914},{0.792572100022,0.0932871990344,0.112987313288},{0.792572100022,0.0932871990344,0.112987313288},{0.787896975349,0.0921799325476,0.112003076515},{0.787896975349,0.0921799325476,0.112003076515},{0.783221850676,0.0910726660607,0.111018839741},{0.783221850676,0.0910726660607,0.111018839741},{0.778546726002,0.0899653995738,0.110034602968},{0.778546726002,0.0899653995738,0.110034602968},{0.773871601329,0.088858133087,0.109050366195},{0.769196476656,0.0877508666001,0.108066129421},{0.769196476656,0.0877508666001,0.108066129421},{0.764521351983,0.0866436001133,0.107081892648},{0.764521351983,0.0866436001133,0.107081892648},{0.759846227309,0.0855363336264,0.106097655875},{0.759846227309,0.0855363336264,0.106097655875},{0.755171102636,0.0844290671395,0.105113419102},{0.755171102636,0.0844290671395,0.105113419102},{0.750495977963,0.0833218006527,0.104129182328},{0.750495977963,0.0833218006527,0.104129182328},{0.745820853289,0.0822145341658,0.103144945555},{0.745820853289,0.0822145341658,0.103144945555},{0.741145728616,0.081107267679,0.102160708782},{0.741145728616,0.081107267679,0.102160708782},{0.736470603943,0.0800000011921,0.101176472008},{0.736470603943,0.0800000011921,0.101176472008},{0.73179547927,0.0788927347052,0.100192235235},{0.73179547927,0.0788927347052,0.100192235235},{0.727120354596,0.0777854682184,0.0992079984616},{0.727120354596,0.0777854682184,0.0992079984616},{0.722445229923,0.0766782017315,0.0982237616883},{0.722445229923,0.0766782017315,0.0982237616883},{0.71777010525,0.0755709352447,0.0972395249149},{0.71777010525,0.0755709352447,0.0972395249149},{0.713094980576,0.0744636687578,0.0962552881416},{0.713094980576,0.0744636687578,0.0962552881416},{0.708419855903,0.0733564022709,0.0952710513683},{0.708419855903,0.0733564022709,0.0952710513683},{0.70374473123,0.0722491357841,0.094286814595},{0.70374473123,0.0722491357841,0.094286814595},{0.699069606557,0.0711418692972,0.0933025778216},{0.699069606557,0.0711418692972,0.0933025778216},{0.694394481883,0.0700346028104,0.0923183410483},{0.694394481883,0.0700346028104,0.0923183410483},{0.68971935721,0.0689273363235,0.091334104275},{0.68971935721,0.0689273363235,0.091334104275},{0.685044232537,0.0678200698366,0.0903498675017},{0.685044232537,0.0678200698366,0.0903498675017},{0.680369107863,0.0667128033498,0.0893656307284},{0.67569398319,0.0656055368629,0.088381393955},{0.67569398319,0.0656055368629,0.088381393955},{0.671018858517,0.0644982703761,0.0873971571817},{0.671018858517,0.0644982703761,0.0873971571817},{0.666343733844,0.0633910038892,0.0864129204084},{0.666343733844,0.0633910038892,0.0864129204084},{0.66166860917,0.0622837374023,0.0854286836351},{0.66166860917,0.0622837374023,0.0854286836351},{0.656993484497,0.0611764709155,0.0844444468617},{0.656993484497,0.0611764709155,0.0844444468617},{0.652318359824,0.0600692044286,0.0834602100884},{0.652318359824,0.0600692044286,0.0834602100884},{0.647643235151,0.0589619379418,0.0824759733151},{0.647643235151,0.0589619379418,0.0824759733151},{0.640384488129,0.0572087660724,0.0814917364395},{0.640384488129,0.0572087660724,0.0814917364395},{0.632756652201,0.0553633220056,0.0805074995493},{0.632756652201,0.0553633220056,0.0805074995493},{0.625128816273,0.0535178779387,0.0795232626591},{0.625128816273,0.0535178779387,0.0795232626591},{0.617500980344,0.0516724338718,0.0785390257689},{0.617500980344,0.0516724338718,0.0785390257689},{0.609873144416,0.049826989805,0.0775547888787},{0.609873144416,0.049826989805,0.0775547888787},{0.602245308488,0.0479815457381,0.0765705519885},{0.602245308488,0.0479815457381,0.0765705519885},{0.59461747256,0.0461361016713,0.0755863150983},{0.59461747256,0.0461361016713,0.0755863150983},{0.586989636632,0.0442906576044,0.0746020782082},{0.586989636632,0.0442906576044,0.0746020782082},{0.579361800703,0.0424452135376,0.073617841318},{0.579361800703,0.0424452135376,0.073617841318},{0.571733964775,0.0405997694707,0.0726336044278},{0.571733964775,0.0405997694707,0.0726336044278},{0.564106128847,0.0387543254039,0.0716493675376},{0.564106128847,0.0387543254039,0.0716493675376},{0.556478292919,0.036908881337,0.0706651306474},{0.556478292919,0.036908881337,0.0706651306474},{0.54885045699,0.0350634372702,0.0696808937572},{0.541222621062,0.0332179932033,0.068696656867},{0.541222621062,0.0332179932033,0.068696656867},{0.533594785134,0.0313725491365,0.0677124199768},{0.533594785134,0.0313725491365,0.0677124199768},{0.525966949206,0.0295271050696,0.0667281830866},{0.525966949206,0.0295271050696,0.0667281830866},{0.518339113278,0.0276816610028,0.0657439461964},{0.518339113278,0.0276816610028,0.0657439461964},{0.510711277349,0.0258362169359,0.0647597093062},{0.510711277349,0.0258362169359,0.0647597093062},{0.503083441421,0.0239907728691,0.063775472416},{0.503083441421,0.0239907728691,0.063775472416},{0.495455605493,0.0221453288022,0.0627912355258},{0.495455605493,0.0221453288022,0.0627912355258},{0.487827769565,0.0202998847354,0.0618069986356},{0.487827769565,0.0202998847354,0.0618069986356},{0.480199933636,0.0184544406685,0.0608227617454},{0.480199933636,0.0184544406685,0.0608227617454},{0.472572097708,0.0166089966017,0.0598385248552},{0.472572097708,0.0166089966017,0.0598385248552},{0.46494426178,0.0147635525348,0.058854287965},{0.46494426178,0.0147635525348,0.058854287965},{0.457316425852,0.012918108468,0.0578700510748},{0.457316425852,0.012918108468,0.0578700510748},{0.449688589924,0.0110726644011,0.0568858141847},{0.449688589924,0.0110726644011,0.0568858141847},{0.442060753995,0.00922722033426,0.0559015772945},{0.442060753995,0.00922722033426,0.0559015772945},{0.434432918067,0.00738177626741,0.0549173404043},{0.434432918067,0.00738177626741,0.0549173404043},{0.426805082139,0.00553633220056,0.0539331035141},{0.426805082139,0.00553633220056,0.0539331035141},{0.419177246211,0.0036908881337,0.0529488666239},{0.419177246211,0.0036908881337,0.0529488666239},{0.411549410282,0.00184544406685,0.0519646297337},{0.411549410282,0.00184544406685,0.0519646297337},{0.403921574354,0.0,0.0509803928435},{0.403921574354,0.0,0.0509803928435},};


const std::vector<glm::vec3> CM_PHASE =
{{0.658308392892,0.469939169032,0.0494128820399},{0.658308392892,0.469939169032,0.0494128820399},{0.664337418937,0.466201900857,0.057664734504},{0.664337418937,0.466201900857,0.057664734504},{0.670208692505,0.462480138122,0.0653456030954},{0.670208692505,0.462480138122,0.0653456030954},{0.676042990533,0.458698375996,0.0727317432221},{0.676042990533,0.458698375996,0.0727317432221},{0.681752284628,0.454914065184,0.0797926195668},{0.681752284628,0.454914065184,0.0797926195668},{0.687402804728,0.451084166991,0.0866710295087},{0.687402804728,0.451084166991,0.0866710295087},{0.692950498095,0.447238929651,0.0933586896015},{0.692950498095,0.447238929651,0.0933586896015},{0.698426191209,0.443357678593,0.0999283926833},{0.698426191209,0.443357678593,0.0999283926833},{0.703812298104,0.439453276235,0.106387104564},{0.703812298104,0.439453276235,0.106387104564},{0.709120692319,0.435517652511,0.112771744925},{0.709120692319,0.435517652511,0.112771744925},{0.714345244901,0.431555756203,0.119093478945},{0.714345244901,0.431555756203,0.119093478945},{0.719492886167,0.427562717169,0.125376059096},{0.719492886167,0.427562717169,0.125376059096},{0.724556192748,0.423544697127,0.13162325163},{0.724556192748,0.423544697127,0.13162325163},{0.72954894829,0.419490984923,0.137863048349},{0.72954894829,0.419490984923,0.137863048349},{0.734451724225,0.41541774051,0.144080393372},{0.734451724225,0.41541774051,0.144080393372},{0.739294955064,0.411299732702,0.150322173697},{0.739294955064,0.411299732702,0.150322173697},{0.744038335124,0.407171577287,0.156543346185},{0.744038335124,0.407171577287,0.156543346185},{0.748736952369,0.402985190782,0.16282281613},{0.748736952369,0.402985190782,0.16282281613},{0.753323193752,0.39880106902,0.169075663806},{0.753323193752,0.39880106902,0.169075663806},{0.757880829458,0.394542451134,0.175421792407},{0.762332602216,0.39028095672,0.181759153853},{0.762332602216,0.39028095672,0.181759153853},{0.7667320479,0.385965494354,0.188168187504},{0.7667320479,0.385965494354,0.188168187504},{0.771052472182,0.381621414822,0.194615319358},{0.771052472182,0.381621414822,0.194615319358},{0.775295277846,0.377247324652,0.201106520895},{0.775295277846,0.377247324652,0.201106520895},{0.779486659378,0.37281508501,0.207687295365},{0.779486659378,0.37281508501,0.207687295365},{0.783585336262,0.368367724773,0.214297364199},{0.783585336262,0.368367724773,0.214297364199},{0.787637631373,0.363853995029,0.221016475748},{0.787637631373,0.363853995029,0.221016475748},{0.79161133858,0.359308040423,0.22779740121},{0.79161133858,0.359308040423,0.22779740121},{0.795506055251,0.354729897918,0.234643526025},{0.795506055251,0.354729897918,0.234643526025},{0.799353983813,0.350079592948,0.241618319348},{0.799353983813,0.350079592948,0.241618319348},{0.803116708478,0.345401516392,0.248658916218},{0.803116708478,0.345401516392,0.248658916218},{0.806810326186,0.34067452251,0.25580075146},{0.806810326186,0.34067452251,0.25580075146},{0.8104452043,0.335882484131,0.263072216807},{0.8104452043,0.335882484131,0.263072216807},{0.813996801063,0.331055383196,0.270431826071},{0.813996801063,0.331055383196,0.270431826071},{0.817476894782,0.326175263767,0.277910961902},{0.817476894782,0.326175263767,0.277910961902},{0.820894148194,0.321226288958,0.285538460449},{0.820894148194,0.321226288958,0.285538460449},{0.824227126161,0.316236198585,0.293276167778},{0.824227126161,0.316236198585,0.293276167778},{0.827476613752,0.311201543345,0.301133878848},{0.827476613752,0.311201543345,0.301133878848},{0.830663985673,0.30608459405,0.309175785183},{0.830663985673,0.30608459405,0.309175785183},{0.833763065889,0.300922435954,0.317349208622},{0.836772858757,0.295713461259,0.325661993753},{0.836772858757,0.295713461259,0.325661993753},{0.839696930167,0.290447232868,0.334136649771},{0.839696930167,0.290447232868,0.334136649771},{0.842538733432,0.285111507699,0.342799619335},{0.842538733432,0.285111507699,0.342799619335},{0.845282969306,0.279729165834,0.351620780974},{0.845282969306,0.279729165834,0.351620780974},{0.847927041458,0.274300451817,0.360606806181},{0.847927041458,0.274300451817,0.360606806181},{0.850467925711,0.268826235069,0.369763947699},{0.850467925711,0.268826235069,0.369763947699},{0.852910557451,0.263288587404,0.379131161153},{0.852910557451,0.263288587404,0.379131161153},{0.855242004405,0.257708883964,0.388682169609},{0.855242004405,0.257708883964,0.388682169609},{0.857456726321,0.252093665886,0.398416010867},{0.857456726321,0.252093665886,0.398416010867},{0.859550232072,0.246447370855,0.408336253344},{0.859550232072,0.246447370855,0.408336253344},{0.861517669567,0.240775634951,0.418445571241},{0.861517669567,0.240775634951,0.418445571241},{0.863353924562,0.235085213863,0.428746057324},{0.863353924562,0.235085213863,0.428746057324},{0.865056845219,0.229372876811,0.43926007811},{0.865056845219,0.229372876811,0.43926007811},{0.866616063146,0.223663075912,0.449961266039},{0.866616063146,0.223663075912,0.449961266039},{0.86802577872,0.217967852481,0.46084758006},{0.86802577872,0.217967852481,0.46084758006},{0.86928002814,0.212301322732,0.471915544486},{0.86928002814,0.212301322732,0.471915544486},{0.870372741448,0.206679880751,0.483160154069},{0.870372741448,0.206679880751,0.483160154069},{0.871297806808,0.201122398412,0.49457479396},{0.871297806808,0.201122398412,0.49457479396},{0.872049140265,0.195650412874,0.506151179086},{0.872049140265,0.195650412874,0.506151179086},{0.87262075977,0.190288288901,0.517879317254},{0.873006861953,0.185063339211,0.529747499865},{0.873006861953,0.185063339211,0.529747499865},{0.873201899807,0.180005881391,0.541742323374},{0.873201899807,0.180005881391,0.541742323374},{0.87320065922,0.175149204974,0.553848743653},{0.87320065922,0.175149204974,0.553848743653},{0.872998332157,0.170529417732,0.566050164184},{0.872998332157,0.170529417732,0.566050164184},{0.872590584303,0.166185137096,0.578328557682},{0.872590584303,0.166185137096,0.578328557682},{0.871973612653,0.16215697962,0.590664660687},{0.871973612653,0.16215697962,0.590664660687},{0.87114414307,0.158486668076,0.603038808554},{0.87114414307,0.158486668076,0.603038808554},{0.870099660691,0.155216866248,0.615428437863},{0.870099660691,0.155216866248,0.615428437863},{0.868838231517,0.152388922857,0.627811754858},{0.868838231517,0.152388922857,0.627811754858},{0.867358580349,0.150041987043,0.640166505942},{0.867358580349,0.150041987043,0.640166505942},{0.865660095361,0.148211493519,0.65247021711},{0.865660095361,0.148211493519,0.65247021711},{0.863742820371,0.146927620733,0.664700433402},{0.863742820371,0.146927620733,0.664700433402},{0.861607435585,0.14621385657,0.676834951829},{0.861607435585,0.14621385657,0.676834951829},{0.85925522796,0.146085818174,0.688852041748},{0.85925522796,0.146085818174,0.688852041748},{0.856688052659,0.146550460138,0.700730647495},{0.856688052659,0.146550460138,0.700730647495},{0.853908294081,0.147605763482,0.712450541666},{0.853908294081,0.147605763482,0.712450541666},{0.850918813243,0.149240943921,0.723992497409},{0.850918813243,0.149240943921,0.723992497409},{0.847722873216,0.151437167202,0.735338489692},{0.847722873216,0.151437167202,0.735338489692},{0.844324093567,0.154168650668,0.746471739322},{0.840726393963,0.157404031455,0.757376782904},{0.840726393963,0.157404031455,0.757376782904},{0.836933938672,0.161107855552,0.76803951842},{0.836933938672,0.161107855552,0.76803951842},{0.832951083227,0.165242052784,0.778447227341},{0.832951083227,0.165242052784,0.778447227341},{0.828782324029,0.169767291005,0.788588575589},{0.828782324029,0.169767291005,0.788588575589},{0.824432251451,0.174644139132,0.798453595936},{0.824432251451,0.174644139132,0.798453595936},{0.819905506793,0.179834004693,0.808033654553},{0.819905506793,0.179834004693,0.808033654553},{0.81520674324,0.185299841441,0.817321404386},{0.81520674324,0.185299841441,0.817321404386},{0.810340590837,0.191006643757,0.826310727942},{0.810340590837,0.191006643757,0.826310727942},{0.805311757905,0.196921601061,0.834996450489},{0.805311757905,0.196921601061,0.834996450489},{0.800124669532,0.20301465242,0.843374862107},{0.800124669532,0.20301465242,0.843374862107},{0.794783673009,0.209258261464,0.851443195646},{0.794783673009,0.209258261464,0.851443195646},{0.789293018039,0.215627373716,0.859199565535},{0.789293018039,0.215627373716,0.859199565535},{0.783656806972,0.222099355044,0.866642944436},{0.783656806972,0.222099355044,0.866642944436},{0.777878979612,0.228653855109,0.87377308288},{0.777878979612,0.228653855109,0.87377308288},{0.771963300247,0.235272649647,0.880590429922},{0.771963300247,0.235272649647,0.880590429922},{0.765913346604,0.241939473532,0.887096055527},{0.765913346604,0.241939473532,0.887096055527},{0.759732500465,0.248639853039,0.893291575193},{0.759732500465,0.248639853039,0.893291575193},{0.753423939683,0.255360942964,0.899179077196},{0.753423939683,0.255360942964,0.899179077196},{0.74699063142,0.262091372137,0.904761052687},{0.74699063142,0.262091372137,0.904761052687},{0.740435326435,0.268821099342,0.910040328779},{0.733760554319,0.275541280537,0.915020004719},{0.733760554319,0.275541280537,0.915020004719},{0.726968619585,0.282244147513,0.919703391141},{0.726968619585,0.282244147513,0.919703391141},{0.720061598583,0.288922897644,0.924093952388},{0.720061598583,0.288922897644,0.924093952388},{0.713041337231,0.295571594063,0.92819525188},{0.713041337231,0.295571594063,0.92819525188},{0.705909449591,0.302185075438,0.932010900454},{0.705909449591,0.302185075438,0.932010900454},{0.698667317349,0.308758874406,0.935544507658},{0.698667317349,0.308758874406,0.935544507658},{0.691316090279,0.315289143767,0.938799635947},{0.691316090279,0.315289143767,0.938799635947},{0.683856687822,0.321772589498,0.941779757756},{0.683856687822,0.321772589498,0.941779757756},{0.676289801898,0.328206409748,0.944488215474},{0.676289801898,0.328206409748,0.944488215474},{0.66861590113,0.334588239008,0.946928184318},{0.66861590113,0.334588239008,0.946928184318},{0.660835236665,0.340916096734,0.949102638181},{0.660835236665,0.340916096734,0.949102638181},{0.652947849787,0.347188339785,0.951014318531},{0.652947849787,0.347188339785,0.951014318531},{0.644953581573,0.35340361808,0.952665706495},{0.644953581573,0.35340361808,0.952665706495},{0.636852084815,0.359560832988,0.954058998276},{0.636852084815,0.359560832988,0.954058998276},{0.628642838505,0.365659098003,0.955196084109},{0.628642838505,0.365659098003,0.955196084109},{0.620325165144,0.371697701338,0.956078530991},{0.620325165144,0.371697701338,0.956078530991},{0.611898251184,0.377676070126,0.956707569474},{0.611898251184,0.377676070126,0.956707569474},{0.603361170918,0.38359373599,0.957084084833},{0.603361170918,0.38359373599,0.957084084833},{0.594712914127,0.389450301774,0.957208612975},{0.594712914127,0.389450301774,0.957208612975},{0.585952417808,0.395245409322,0.957081341492},{0.577078602298,0.400978708233,0.956702116283},{0.577078602298,0.400978708233,0.956702116283},{0.568090412076,0.406649825569,0.956070454204},{0.568090412076,0.406649825569,0.956070454204},{0.55898686152,0.412258336579,0.955185562223},{0.55898686152,0.412258336579,0.955185562223},{0.549767085846,0.417803736546,0.954046363531},{0.549767085846,0.417803736546,0.954046363531},{0.540430397369,0.42328541394,0.952651531091},{0.540430397369,0.42328541394,0.952651531091},{0.530976347181,0.428702625127,0.950999529007},{0.530976347181,0.428702625127,0.950999529007},{0.521404792215,0.43405447093,0.94908866207},{0.521404792215,0.43405447093,0.94908866207},{0.511715967538,0.439339875449,0.94691713371},{0.511715967538,0.439339875449,0.94691713371},{0.501910563544,0.444557567548,0.944483112445},{0.501910563544,0.444557567548,0.944483112445},{0.491989807571,0.449706065529,0.941784806747},{0.491989807571,0.449706065529,0.941784806747},{0.481955549211,0.454783665516,0.938820547987},{0.481955549211,0.454783665516,0.938820547987},{0.471810348375,0.45978843412,0.93558888087},{0.471810348375,0.45978843412,0.93558888087},{0.461557564917,0.464718205967,0.932088660443},{0.461557564917,0.464718205967,0.932088660443},{0.451198008225,0.469572177684,0.92831786369},{0.451198008225,0.469572177684,0.92831786369},{0.440738524632,0.47434688193,0.924276690668},{0.440738524632,0.47434688193,0.924276690668},{0.430187219196,0.479038640683,0.919966197002},{0.430187219196,0.479038640683,0.919966197002},{0.419551659042,0.483644396275,0.91538759064},{0.419551659042,0.483644396275,0.91538759064},{0.408840633101,0.488160938611,0.910542931565},{0.408840633101,0.488160938611,0.910542931565},{0.398064208302,0.492584942395,0.905435227216},{0.387233771755,0.496913010871,0.900068517037},{0.387233771755,0.496913010871,0.900068517037},{0.376362056499,0.501141725475,0.894447942754},{0.376362056499,0.501141725475,0.894447942754},{0.365461266376,0.505268403433,0.888578767472},{0.365461266376,0.505268403433,0.888578767472},{0.354546543979,0.509289796888,0.882468192304},{0.354546543979,0.509289796888,0.882468192304},{0.343637793185,0.513201579589,0.876126636093},{0.343637793185,0.513201579589,0.876126636093},{0.332753093914,0.51700081982,0.869564086482},{0.332753093914,0.51700081982,0.869564086482},{0.321911658393,0.520684873358,0.862791658562},{0.321911658393,0.520684873358,0.862791658562},{0.311133719862,0.524251441443,0.855821520667},{0.311133719862,0.524251441443,0.855821520667},{0.300440398609,0.527698623816,0.848666794877},{0.300440398609,0.527698623816,0.848666794877},{0.289853256631,0.531025054992,0.841341230552},{0.289853256631,0.531025054992,0.841341230552},{0.279396159424,0.534229314818,0.833860509465},{0.279396159424,0.534229314818,0.833860509465},{0.269091811976,0.537310987157,0.82623984187},{0.269091811976,0.537310987157,0.82623984187},{0.258963000993,0.540270201043,0.81849474753},{0.258963000993,0.540270201043,0.81849474753},{0.249032393162,0.543107629026,0.810640895215},{0.249032393162,0.543107629026,0.810640895215},{0.239322289958,0.54582448394,0.80269392237},{0.239322289958,0.54582448394,0.80269392237},{0.229856635182,0.548421889,0.794671215898},{0.229856635182,0.548421889,0.794671215898},{0.220655100294,0.550902414196,0.786587062692},{0.220655100294,0.550902414196,0.786587062692},{0.211736409442,0.553269012911,0.778455329698},{0.211736409442,0.553269012911,0.778455329698},{0.203118430649,0.555524891192,0.770289734711},{0.203118430649,0.555524891192,0.770289734711},{0.194817201504,0.557673646703,0.76210317763},{0.186846596629,0.559719222965,0.753907633934},{0.186846596629,0.559719222965,0.753907633934},{0.179217988478,0.561665861309,0.745714066205},{0.179217988478,0.561665861309,0.745714066205},{0.171942195902,0.56351747024,0.737534984317},{0.171942195902,0.56351747024,0.737534984317},{0.165022951267,0.565279153364,0.729377543987},{0.165022951267,0.565279153364,0.729377543987},{0.158461160281,0.56695600177,0.721248188626},{0.158461160281,0.56695600177,0.721248188626},{0.152254991821,0.568552967252,0.713153205188},{0.152254991821,0.568552967252,0.713153205188},{0.146398761868,0.570075056921,0.705097686656},{0.146398761868,0.570075056921,0.705097686656},{0.140882840591,0.571527291452,0.697085535335},{0.140882840591,0.571527291452,0.697085535335},{0.135693662537,0.572914665847,0.689119475827},{0.135693662537,0.572914665847,0.689119475827},{0.130813852542,0.574242112738,0.681201076313},{0.130813852542,0.574242112738,0.681201076313},{0.126222473429,0.575514468206,0.673330776933},{0.126222473429,0.575514468206,0.673330776933},{0.121895393804,0.576736439983,0.665507924233},{0.121895393804,0.576736439983,0.665507924233},{0.117806543972,0.577912352304,0.657732326014},{0.117806543972,0.577912352304,0.657732326014},{0.113926129801,0.57904679578,0.649999836645},{0.113926129801,0.57904679578,0.649999836645},{0.110223478239,0.580143980837,0.642306368114},{0.110223478239,0.580143980837,0.642306368114},{0.106667323518,0.581207821548,0.634647331489},{0.106667323518,0.581207821548,0.634647331489},{0.103226306764,0.582241975777,0.627017287995},{0.103226306764,0.582241975777,0.627017287995},{0.0998697011666,0.583249824863,0.619410007498},{0.0998697011666,0.583249824863,0.619410007498},{0.0965681326987,0.584234454901,0.611818530585},{0.0965681326987,0.584234454901,0.611818530585},{0.0932942920612,0.585198639665,0.604235234177},{0.0900236427693,0.586144825264,0.596651900546},{0.0900236427693,0.586144825264,0.596651900546},{0.0867351401365,0.587075116665,0.589059789486},{0.0867351401365,0.587075116665,0.589059789486},{0.0834119859822,0.587991266222,0.581449713302},{0.0834119859822,0.587991266222,0.581449713302},{0.0800424540035,0.588894664373,0.573812114106},{0.0800424540035,0.588894664373,0.573812114106},{0.0766208306059,0.589786332627,0.566137142784},{0.0766208306059,0.589786332627,0.566137142784},{0.073148524854,0.590666918973,0.558414738841},{0.073148524854,0.590666918973,0.558414738841},{0.0696354071465,0.591536695753,0.55063471023},{0.0696354071465,0.591536695753,0.55063471023},{0.0661014350115,0.592395560023,0.542786812151},{0.0661014350115,0.592395560023,0.542786812151},{0.0625786076009,0.593243036315,0.534860823801},{0.0625786076009,0.593243036315,0.534860823801},{0.0591130375998,0.594078331469,0.526846137602},{0.0591130375998,0.594078331469,0.526846137602},{0.0557676501029,0.5949002975,0.518732197987},{0.0557676501029,0.5949002975,0.518732197987},{0.0526251056542,0.595707320083,0.510509780826},{0.0526251056542,0.595707320083,0.510509780826},{0.0497888068094,0.596497503811,0.502169359532},{0.0497888068094,0.596497503811,0.502169359532},{0.0473831926939,0.597268621556,0.493701736171},{0.0473831926939,0.597268621556,0.493701736171},{0.0455506684766,0.598018125295,0.485098086772},{0.0455506684766,0.598018125295,0.485098086772},{0.044443961893,0.59874315667,0.476350000404},{0.044443961893,0.59874315667,0.476350000404},{0.044213229489,0.599440556688,0.46744951244},{0.044213229489,0.599440556688,0.46744951244},{0.0449891786771,0.600106874013,0.458389132783},{0.0449891786771,0.600106874013,0.458389132783},{0.0468660448507,0.600738371276,0.449161870185},{0.0498897902424,0.601331028898,0.439761254173},{0.0498897902424,0.601331028898,0.439761254173},{0.0540557300631,0.601880554394,0.430181201574},{0.0540557300631,0.601880554394,0.430181201574},{0.0593220854021,0.602382887631,0.420405432562},{0.0593220854021,0.602382887631,0.420405432562},{0.0656077388042,0.602832583137,0.410437715603},{0.0656077388042,0.602832583137,0.410437715603},{0.0728196236389,0.603224415597,0.400273626235},{0.0728196236389,0.603224415597,0.400273626235},{0.0808617678135,0.603552833505,0.389909414935},{0.0808617678135,0.603552833505,0.389909414935},{0.0896436576776,0.603811940421,0.379342077786},{0.0896436576776,0.603811940421,0.379342077786},{0.0990895248698,0.603995543502,0.368564118402},{0.0990895248698,0.603995543502,0.368564118402},{0.109146168639,0.604096946088,0.357557987688},{0.109146168639,0.604096946088,0.357557987688},{0.119741186824,0.604108584409,0.34634096055},{0.119741186824,0.604108584409,0.34634096055},{0.130827463571,0.604022803635,0.334914160517},{0.130827463571,0.604022803635,0.334914160517},{0.14238003476,0.60383119145,0.323266999746},{0.14238003476,0.60383119145,0.323266999746},{0.154384700863,0.603524253284,0.311388229549},{0.154384700863,0.603524253284,0.311388229549},{0.166790927844,0.603093006307,0.299310287856},{0.166790927844,0.603093006307,0.299310287856},{0.179597574193,0.602526680057,0.287023704448},{0.179597574193,0.602526680057,0.287023704448},{0.192799657016,0.601813638169,0.274529642485},{0.192799657016,0.601813638169,0.274529642485},{0.206344646186,0.600944661405,0.261879400204},{0.206344646186,0.600944661405,0.261879400204},{0.220272872973,0.599904299374,0.249042513665},{0.220272872973,0.599904299374,0.249042513665},{0.234498332925,0.598685911475,0.236110215698},{0.234498332925,0.598685911475,0.236110215698},{0.249044157731,0.597274598402,0.223077804617},{0.263820058885,0.595666560172,0.210046733237},{0.263820058885,0.595666560172,0.210046733237},{0.278810397244,0.593852096035,0.197054842687},{0.278810397244,0.593852096035,0.197054842687},{0.293914943602,0.591833478387,0.184216205574},{0.293914943602,0.591833478387,0.184216205574},{0.309063395803,0.589613019789,0.171619421326},{0.309063395803,0.589613019789,0.171619421326},{0.324155770115,0.587201319418,0.159377534847},{0.324155770115,0.587201319418,0.159377534847},{0.339105898793,0.584611641776,0.147590123966},{0.339105898793,0.584611641776,0.147590123966},{0.353796240305,0.581867930492,0.136377340519},{0.353796240305,0.581867930492,0.136377340519},{0.368179053993,0.57898610151,0.125800541591},{0.368179053993,0.57898610151,0.125800541591},{0.382159657379,0.575995119173,0.115950397799},{0.382159657379,0.575995119173,0.115950397799},{0.395728244523,0.572909280944,0.106850382067},{0.395728244523,0.572909280944,0.106850382067},{0.408819264891,0.569757269791,0.0985552120261},{0.408819264891,0.569757269791,0.0985552120261},{0.421481064904,0.566541586272,0.0910400224657},{0.421481064904,0.566541586272,0.0910400224657},{0.433649533456,0.56329295616,0.0843411623911},{0.433649533456,0.56329295616,0.0843411623911},{0.445389078352,0.560008590638,0.0784130544391},{0.445389078352,0.560008590638,0.0784130544391},{0.456724213085,0.556694295148,0.0732291301251},{0.456724213085,0.556694295148,0.0732291301251},{0.467650170768,0.553363726905,0.0687676213476},{0.467650170768,0.553363726905,0.0687676213476},{0.478191378182,0.550021300819,0.0649843572199},{0.478191378182,0.550021300819,0.0649843572199},{0.488396861238,0.546661950698,0.0618216283741},{0.488396861238,0.546661950698,0.0618216283741},{0.498289239879,0.54328739536,0.0592272577501},{0.498289239879,0.54328739536,0.0592272577501},{0.507891137442,0.539898266893,0.0571446599845},{0.517224745646,0.53649428722,0.055514762882},{0.517224745646,0.53649428722,0.055514762882},{0.526311502589,0.533074430075,0.0542779255664},{0.526311502589,0.533074430075,0.0542779255664},{0.535171864525,0.529637065882,0.0533756685344},{0.535171864525,0.529637065882,0.0533756685344},{0.543825153637,0.526180088704,0.0527520754738},{0.543825153637,0.526180088704,0.0527520754738},{0.552289466379,0.522701025565,0.0523547907545},{0.552289466379,0.522701025565,0.0523547907545},{0.560581629478,0.519197129018,0.0521355994447},{0.560581629478,0.519197129018,0.0521355994447},{0.568717193317,0.515665454088,0.0520506243945},{0.568717193317,0.515665454088,0.0520506243945},{0.576710454772,0.512102920631,0.0520602020436},{0.576710454772,0.512102920631,0.0520602020436},{0.58457450375,0.508506361973,0.052128506748},{0.58457450375,0.508506361973,0.052128506748},{0.592321289445,0.504872560293,0.052222987749},{0.592321289445,0.504872560293,0.052222987749},{0.599961703895,0.501198268844,0.0523136690356},{0.599961703895,0.501198268844,0.0523136690356},{0.607505681632,0.497480220579,0.0523723445572},{0.607505681632,0.497480220579,0.0523723445572},{0.614962315274,0.493715122292,0.0523716818241},{0.614962315274,0.493715122292,0.0523716818241},{0.622339987733,0.489899632822,0.0522842261313},{0.622339987733,0.489899632822,0.0522842261313},{0.629646522519,0.486030323314,0.0520812738141},{0.629646522519,0.486030323314,0.0520812738141},{0.636889354233,0.482103616943,0.0517315523255},{0.636889354233,0.482103616943,0.0517315523255},{0.644075722043,0.478115704908,0.0511996007682},{0.644075722043,0.478115704908,0.0511996007682},{0.651212889353,0.474062435013,0.0504436747876},{0.651212889353,0.474062435013,0.0504436747876},{0.658308392892,0.469939169032,0.049412882041},{0.658308392892,0.469939169032,0.049412882041},};

const std::vector<glm::vec3> CM_HSV =
{{0.85,0.0,0.0},{0.85,0.0,0.0},{0.85,0.019687519687519674,0.0},{0.85,0.019687519687519674,0.0},{0.85,0.03937503937503935,0.0},{0.85,0.03937503937503935,0.0},{0.85,0.059062559062559014,0.0},{0.85,0.059062559062559014,0.0},{0.85,0.07875007875007878,0.0},{0.85,0.07875007875007878,0.0},{0.85,0.09843759843759844,0.0},{0.85,0.09843759843759844,0.0},{0.85,0.11812511812511813,0.0},{0.85,0.11812511812511813,0.0},{0.85,0.1378126378126378,0.0},{0.85,0.1378126378126378,0.0},{0.85,0.15750015750015747,0.0},{0.85,0.15750015750015747,0.0},{0.85,0.17718767718767714,0.0},{0.85,0.17718767718767714,0.0},{0.85,0.1968751968751969,0.0},{0.85,0.1968751968751969,0.0},{0.85,0.21656271656271658,0.0},{0.85,0.21656271656271658,0.0},{0.85,0.23625023625023625,0.0},{0.85,0.23625023625023625,0.0},{0.85,0.2559377559377559,0.0},{0.85,0.2559377559377559,0.0},{0.85,0.2756252756252756,0.0},{0.85,0.2756252756252756,0.0},{0.85,0.29531279531279525,0.0},{0.85,0.29531279531279525,0.0},{0.85,0.31500031500031495,0.0},{0.85,0.31500031500031495,0.0},{0.85,0.3346878346878346,0.0},{0.85,0.3346878346878346,0.0},{0.85,0.3543753543753543,0.0},{0.85,0.3543753543753543,0.0},{0.85,0.37406287406287403,0.0},{0.85,0.3937503937503937,0.0},{0.85,0.3937503937503937,0.0},{0.85,0.41343791343791336,0.0},{0.85,0.41343791343791336,0.0},{0.85,0.43312543312543306,0.0},{0.85,0.43312543312543306,0.0},{0.85,0.45281295281295275,0.0},{0.85,0.45281295281295275,0.0},{0.85,0.47250047250047245,0.0},{0.85,0.47250047250047245,0.0},{0.85,0.49218799218799203,0.0},{0.85,0.49218799218799203,0.0},{0.85,0.5118755118755118,0.0},{0.85,0.5118755118755118,0.0},{0.85,0.5315630315630314,0.0},{0.85,0.5315630315630314,0.0},{0.85,0.5512505512505512,0.0},{0.85,0.5512505512505512,0.0},{0.85,0.5709380709380709,0.0},{0.85,0.5709380709380709,0.0},{0.85,0.5906255906255905,0.0},{0.85,0.5906255906255905,0.0},{0.85,0.6103131103131103,0.0},{0.85,0.6103131103131103,0.0},{0.85,0.63000063000063,0.0},{0.85,0.63000063000063,0.0},{0.85,0.6496881496881495,0.0},{0.85,0.6496881496881495,0.0},{0.85,0.6693756693756693,0.0},{0.85,0.6693756693756693,0.0},{0.85,0.6890631890631891,0.0},{0.85,0.6890631890631891,0.0},{0.85,0.7087507087507086,0.0},{0.85,0.7087507087507086,0.0},{0.85,0.7284382284382283,0.0},{0.85,0.7284382284382283,0.0},{0.85,0.7481257481257481,0.0},{0.85,0.7481257481257481,0.0},{0.85,0.7678132678132678,0.0},{0.85,0.7875007875007874,0.0},{0.85,0.7875007875007874,0.0},{0.846562230937231,0.8037505381255381,0.0},{0.846562230937231,0.8037505381255381,0.0},{0.8399997243747244,0.8168755512505512,0.0},{0.8399997243747244,0.8168755512505512,0.0},{0.8334372178122178,0.8300005643755644,0.0},{0.8334372178122178,0.8300005643755644,0.0},{0.8268747112497115,0.8431255775005774,0.0},{0.8268747112497115,0.8431255775005774,0.0},{0.8140616140616144,0.85,0.0},{0.8140616140616144,0.85,0.0},{0.7943740943740945,0.85,0.0},{0.7943740943740945,0.85,0.0},{0.7746865746865746,0.85,0.0},{0.7746865746865746,0.85,0.0},{0.7549990549990548,0.85,0.0},{0.7549990549990548,0.85,0.0},{0.7353115353115355,0.85,0.0},{0.7353115353115355,0.85,0.0},{0.7156240156240157,0.85,0.0},{0.7156240156240157,0.85,0.0},{0.6959364959364958,0.85,0.0},{0.6959364959364958,0.85,0.0},{0.6762489762489764,0.85,0.0},{0.6762489762489764,0.85,0.0},{0.6565614565614571,0.85,0.0},{0.6565614565614571,0.85,0.0},{0.6368739368739367,0.85,0.0},{0.6368739368739367,0.85,0.0},{0.6171864171864174,0.85,0.0},{0.6171864171864174,0.85,0.0},{0.5974988974988975,0.85,0.0},{0.5974988974988975,0.85,0.0},{0.5778113778113777,0.85,0.0},{0.5778113778113777,0.85,0.0},{0.5581238581238585,0.85,0.0},{0.5581238581238585,0.85,0.0},{0.5384363384363385,0.85,0.0},{0.5187488187488187,0.85,0.0},{0.5187488187488187,0.85,0.0},{0.4990612990612994,0.85,0.0},{0.4990612990612994,0.85,0.0},{0.47937377937377956,0.85,0.0},{0.47937377937377956,0.85,0.0},{0.4596862596862597,0.85,0.0},{0.4596862596862597,0.85,0.0},{0.4399987399987404,0.85,0.0},{0.4399987399987404,0.85,0.0},{0.42031122031122,0.85,0.0},{0.42031122031122,0.85,0.0},{0.4006237006237007,0.85,0.0},{0.4006237006237007,0.85,0.0},{0.38093618093618087,0.85,0.0},{0.38093618093618087,0.85,0.0},{0.36124866124866095,0.85,0.0},{0.36124866124866095,0.85,0.0},{0.34156114156114165,0.85,0.0},{0.34156114156114165,0.85,0.0},{0.3218736218736218,0.85,0.0},{0.3218736218736218,0.85,0.0},{0.3021861021861019,0.85,0.0},{0.3021861021861019,0.85,0.0},{0.2824985824985827,0.85,0.0},{0.2824985824985827,0.85,0.0},{0.2628110628110628,0.85,0.0},{0.2628110628110628,0.85,0.0},{0.24312354312354348,0.85,0.0},{0.24312354312354348,0.85,0.0},{0.22343602343602362,0.85,0.0},{0.22343602343602362,0.85,0.0},{0.20374850374850376,0.85,0.0},{0.20374850374850376,0.85,0.0},{0.18406098406098392,0.85,0.0},{0.18406098406098392,0.85,0.0},{0.16437346437346462,0.85,0.0},{0.14468594468594476,0.85,0.0},{0.14468594468594476,0.85,0.0},{0.1249984249984249,0.85,0.0},{0.1249984249984249,0.85,0.0},{0.10531090531090559,0.85,0.0},{0.10531090531090559,0.85,0.0},{0.08562338562338631,0.85,0.0},{0.08562338562338631,0.85,0.0},{0.06593586593586587,0.85,0.0},{0.06593586593586587,0.85,0.0},{0.04624834624834602,0.85,0.0},{0.04624834624834602,0.85,0.0},{0.02656089628315742,0.8500011156261156,0.0},{0.02656089628315742,0.8500011156261156,0.0},{0.007088918601823065,0.8631261287511287,0.0},{0.007088918601823065,0.8631261287511287,0.0},{0.0,0.8634369290619291,0.013225858210837826},{0.0,0.8634369290619291,0.013225858210837826},{0.0,0.8568744224994225,0.033031738107403674},{0.0,0.8568744224994225,0.033031738107403674},{0.0,0.850311915936916,0.05222756897819825},{0.0,0.850311915936916,0.05222756897819825},{0.0,0.85,0.07187665374123467},{0.0,0.85,0.07187665374123467},{0.0,0.85,0.09156404939803806},{0.0,0.85,0.09156404939803806},{0.0,0.85,0.11125144505484116},{0.0,0.85,0.11125144505484116},{0.0,0.85,0.1309388407116443},{0.0,0.85,0.1309388407116443},{0.0,0.85,0.15062623636844713},{0.0,0.85,0.15062623636844713},{0.0,0.85,0.17031363202525024},{0.0,0.85,0.17031363202525024},{0.0,0.85,0.19000102768205335},{0.0,0.85,0.19000102768205335},{0.0,0.85,0.20968842333885618},{0.0,0.85,0.20968842333885618},{0.0,0.85,0.229375818995659},{0.0,0.85,0.2490632146524624},{0.0,0.85,0.2490632146524624},{0.0,0.85,0.2687506103092652},{0.0,0.85,0.2687506103092652},{0.0,0.85,0.28843800596606833},{0.0,0.85,0.28843800596606833},{0.0,0.85,0.30812540162287144},{0.0,0.85,0.30812540162287144},{0.0,0.85,0.3278127972796743},{0.0,0.85,0.3278127972796743},{0.0,0.85,0.3475001929364774},{0.0,0.85,0.3475001929364774},{0.0,0.85,0.36718758859328055},{0.0,0.85,0.36718758859328055},{0.0,0.85,0.38687498425008304},{0.0,0.85,0.38687498425008304},{0.0,0.85,0.4065623799068865},{0.0,0.85,0.4065623799068865},{0.0,0.85,0.4262497755636896},{0.0,0.85,0.4262497755636896},{0.0,0.85,0.44593717122049237},{0.0,0.85,0.44593717122049237},{0.0,0.85,0.4656245668772958},{0.0,0.85,0.4656245668772958},{0.0,0.85,0.48531196253409864},{0.0,0.85,0.48531196253409864},{0.0,0.85,0.5049993581909015},{0.0,0.85,0.5049993581909015},{0.0,0.85,0.5246867538477046},{0.0,0.85,0.5246867538477046},{0.0,0.85,0.5443741495045071},{0.0,0.85,0.5443741495045071},{0.0,0.85,0.564061545161311},{0.0,0.85,0.564061545161311},{0.0,0.85,0.5837489408181139},{0.0,0.85,0.5837489408181139},{0.0,0.85,0.6034363364749167},{0.0,0.85,0.6034363364749167},{0.0,0.85,0.6231237321317196},{0.0,0.85,0.6428111277885227},{0.0,0.85,0.6428111277885227},{0.0,0.85,0.6624985234453261},{0.0,0.85,0.6624985234453261},{0.0,0.85,0.6821859191021292},{0.0,0.85,0.6821859191021292},{0.0,0.85,0.7018733147589318},{0.0,0.85,0.7018733147589318},{0.0,0.85,0.7215607104157351},{0.0,0.85,0.7215607104157351},{0.0,0.85,0.7412481060725382},{0.0,0.85,0.7412481060725382},{0.0,0.85,0.7609355017293411},{0.0,0.85,0.7609355017293411},{0.0,0.85,0.7806228973861442},{0.0,0.85,0.7806228973861442},{0.0,0.85,0.8003102930429473},{0.0,0.85,0.8003102930429473},{0.0,0.85,0.8199976886997501},{0.0,0.85,0.8199976886997501},{0.0,0.85,0.8396850843565533},{0.0,0.85,0.8396850843565533},{0.0,0.8406274609399609,0.85},{0.0,0.8406274609399609,0.85},{0.0,0.8209399412524405,0.85},{0.0,0.8209399412524405,0.85},{0.0,0.8012524215649223,0.85},{0.0,0.8012524215649223,0.85},{0.0,0.7815649018774012,0.85},{0.0,0.7815649018774012,0.85},{0.0,0.7618773821898821,0.85},{0.0,0.7618773821898821,0.85},{0.0,0.7421898625023622,0.85},{0.0,0.7421898625023622,0.85},{0.0,0.7225023428148429,0.85},{0.0,0.7225023428148429,0.85},{0.0,0.7028148231273225,0.85},{0.0,0.6831273034398032,0.85},{0.0,0.6831273034398032,0.85},{0.0,0.6634397837522833,0.85},{0.0,0.6634397837522833,0.85},{0.0,0.6437522640647639,0.85},{0.0,0.6437522640647639,0.85},{0.0,0.6240647443772441,0.85},{0.0,0.6240647443772441,0.85},{0.0,0.6043772246897248,0.85},{0.0,0.6043772246897248,0.85},{0.0,0.5846897050022044,0.85},{0.0,0.5846897050022044,0.85},{0.0,0.5650021853146852,0.85},{0.0,0.5650021853146852,0.85},{0.0,0.5453146656271652,0.85},{0.0,0.5453146656271652,0.85},{0.0,0.525627145939646,0.85},{0.0,0.525627145939646,0.85},{0.0,0.5059396262521261,0.85},{0.0,0.5059396262521261,0.85},{0.0,0.4862521065646068,0.85},{0.0,0.4862521065646068,0.85},{0.0,0.46656458687708635,0.85},{0.0,0.46656458687708635,0.85},{0.0,0.4468770671895671,0.85},{0.0,0.4468770671895671,0.85},{0.0,0.42718954750204724,0.85},{0.0,0.42718954750204724,0.85},{0.0,0.40750202781452793,0.85},{0.0,0.40750202781452793,0.85},{0.0,0.38781450812700813,0.85},{0.0,0.38781450812700813,0.85},{0.0,0.36812698843948877,0.85},{0.0,0.36812698843948877,0.85},{0.0,0.34843946875196835,0.85},{0.0,0.34843946875196835,0.85},{0.0,0.32875194906444904,0.85},{0.0,0.32875194906444904,0.85},{0.0,0.3090644293769292,0.85},{0.0,0.28937690968940993,0.85},{0.0,0.28937690968940993,0.85},{0.0,0.2696893900018901,0.85},{0.0,0.2696893900018901,0.85},{0.0,0.2500018703143707,0.85},{0.0,0.2500018703143707,0.85},{0.0,0.23031435062685035,0.85},{0.0,0.23031435062685035,0.85},{0.0,0.2106268309393305,0.85},{0.0,0.2106268309393305,0.85},{0.0,0.19093931125181116,0.85},{0.0,0.19093931125181116,0.85},{0.0,0.17125179156429188,0.85},{0.0,0.17125179156429188,0.85},{0.0,0.15156427187677202,0.85},{0.0,0.15156427187677202,0.85},{0.0,0.1318767521892527,0.85},{0.0,0.1318767521892527,0.85},{0.0,0.1121892325017323,0.85},{0.0,0.1121892325017323,0.85},{0.0,0.09250171281421299,0.85},{0.0,0.09250171281421299,0.85},{0.0,0.07281419312669313,0.85},{0.0,0.07281419312669313,0.85},{0.0,0.05312667343917327,0.85},{0.0,0.05312667343917327,0.85},{0.0,0.03395946704038469,0.8565619487494486},{0.0,0.03395946704038469,0.8565619487494486},{0.0,0.014182959146786335,0.8631244553119553},{0.0,0.014182959146786335,0.8631244553119553},{0.006131102175554736,0.0,0.8637510762510765},{0.006131102175554736,0.0,0.8637510762510765},{0.02566117870736248,0.0,0.8506260631260634},{0.02566117870736248,0.0,0.8506260631260634},{0.04531092499842433,0.0,0.85},{0.04531092499842433,0.0,0.85},{0.06499844468594362,0.0,0.85},{0.06499844468594362,0.0,0.85},{0.08468596437346404,0.0,0.85},{0.10437348406098447,0.0,0.85},{0.10437348406098447,0.0,0.85},{0.12406100374850376,0.0,0.85},{0.12406100374850376,0.0,0.85},{0.14374852343602307,0.0,0.85},{0.14374852343602307,0.0,0.85},{0.16343604312354237,0.0,0.85},{0.16343604312354237,0.0,0.85},{0.1831235628110628,0.0,0.85},{0.1831235628110628,0.0,0.85},{0.20281108249858207,0.0,0.85},{0.20281108249858207,0.0,0.85},{0.2224986021861025,0.0,0.85},{0.2224986021861025,0.0,0.85},{0.2421861218736218,0.0,0.85},{0.2421861218736218,0.0,0.85},{0.2618736415611411,0.0,0.85},{0.2618736415611411,0.0,0.85},{0.2815611612486604,0.0,0.85},{0.2815611612486604,0.0,0.85},{0.30124868093618085,0.0,0.85},{0.30124868093618085,0.0,0.85},{0.3209362006237012,0.0,0.85},{0.3209362006237012,0.0,0.85},{0.34062372031122057,0.0,0.85},{0.34062372031122057,0.0,0.85},{0.3603112399987398,0.0,0.85},{0.3603112399987398,0.0,0.85},{0.3799987596862592,0.0,0.85},{0.3799987596862592,0.0,0.85},{0.39968627937377954,0.0,0.85},{0.39968627937377954,0.0,0.85},{0.4193737990612989,0.0,0.85},{0.4193737990612989,0.0,0.85},{0.43906131874881926,0.0,0.85},{0.43906131874881926,0.0,0.85},{0.4587488384363386,0.0,0.85},{0.4784363581238579,0.0,0.85},{0.4784363581238579,0.0,0.85},{0.4981238778113771,0.0,0.85},{0.4981238778113771,0.0,0.85},{0.5178113974988976,0.0,0.85},{0.5178113974988976,0.0,0.85},{0.5374989171864181,0.0,0.85},{0.5374989171864181,0.0,0.85},{0.5571864368739373,0.0,0.85},{0.5571864368739373,0.0,0.85},{0.5768739565614566,0.0,0.85},{0.5768739565614566,0.0,0.85},{0.5965614762489759,0.0,0.85},{0.5965614762489759,0.0,0.85},{0.6162489959364963,0.0,0.85},{0.6162489959364963,0.0,0.85},{0.6359365156240155,0.0,0.85},{0.6359365156240155,0.0,0.85},{0.655624035311536,0.0,0.85},{0.655624035311536,0.0,0.85},{0.6753115549990554,0.0,0.85},{0.6753115549990554,0.0,0.85},{0.6949990746865746,0.0,0.85},{0.6949990746865746,0.0,0.85},{0.714686594374094,0.0,0.85},{0.714686594374094,0.0,0.85},{0.7343741140616143,0.0,0.85},{0.7343741140616143,0.0,0.85},{0.7540616337491348,0.0,0.85},{0.7540616337491348,0.0,0.85},{0.773749153436653,0.0,0.85},{0.773749153436653,0.0,0.85},{0.7934366731241733,0.0,0.85},{0.7934366731241733,0.0,0.85},{0.8131241928116927,0.0,0.85},{0.8131241928116927,0.0,0.85},{0.8265622374997377,0.0,0.8437505250005248},{0.8265622374997377,0.0,0.8437505250005248},{0.8331247440622441,0.0,0.8306255118755116},{0.8396872506247507,0.0,0.8175004987504988},{0.8396872506247507,0.0,0.8175004987504988},{0.8462497571872573,0.0,0.8043754856254854},{0.8462497571872573,0.0,0.8043754856254854},{0.85,0.0,0.7884382087507088},{0.85,0.0,0.7884382087507088},{0.85,0.0,0.7687506890631883},{0.85,0.0,0.7687506890631883},{0.85,0.0,0.749063169375669},{0.85,0.0,0.749063169375669},{0.85,0.0,0.7293756496881486},{0.85,0.0,0.7293756496881486},{0.85,0.0,0.7096881300006292},{0.85,0.0,0.7096881300006292},{0.85,0.0,0.69000061031311},{0.85,0.0,0.69000061031311},{0.85,0.0,0.6703130906255895},{0.85,0.0,0.6703130906255895},{0.85,0.0,0.6506255709380703},{0.85,0.0,0.6506255709380703},{0.85,0.0,0.630938051250551},{0.85,0.0,0.630938051250551},{0.85,0.0,0.6112505315630317},{0.85,0.0,0.6112505315630317},{0.85,0.0,0.5915630118755113},{0.85,0.0,0.5915630118755113},{0.85,0.0,0.571875492187992},{0.85,0.0,0.571875492187992},{0.85,0.0,0.5521879725004716},{0.85,0.0,0.5521879725004716},{0.85,0.0,0.5325004528129522},{0.85,0.0,0.5325004528129522},{0.85,0.0,0.512812933125433},{0.85,0.0,0.512812933125433},{0.85,0.0,0.49312541343791255},{0.85,0.0,0.49312541343791255},{0.85,0.0,0.47343789375039325},{0.85,0.0,0.47343789375039325},{0.85,0.0,0.4537503740628739},{0.85,0.0,0.4340628543753535},{0.85,0.0,0.4340628543753535},{0.85,0.0,0.41437533468783416},{0.85,0.0,0.41437533468783416},{0.85,0.0,0.3946878150003149},{0.85,0.0,0.3946878150003149},{0.85,0.0,0.37500029531279444},{0.85,0.0,0.37500029531279444},{0.85,0.0,0.35531277562527525},{0.85,0.0,0.35531277562527525},{0.85,0.0,0.33562525593775594},{0.85,0.0,0.33562525593775594},{0.85,0.0,0.3159377362502355},{0.85,0.0,0.3159377362502355},{0.85,0.0,0.2962502165627162},{0.85,0.0,0.2962502165627162},{0.85,0.0,0.2765626968751969},{0.85,0.0,0.2765626968751969},{0.85,0.0,0.2568751771876765},{0.85,0.0,0.2568751771876765},{0.85,0.0,0.2371876575001572},{0.85,0.0,0.2371876575001572},{0.85,0.0,0.21750013781263788},{0.85,0.0,0.21750013781263788},{0.85,0.0,0.19781261812511747},{0.85,0.0,0.19781261812511747},{0.85,0.0,0.17812509843759816},{0.85,0.0,0.17812509843759816},{0.85,0.0,0.15843757875007886},{0.85,0.0,0.15843757875007886},{0.85,0.0,0.13875005906255844},{0.85,0.0,0.13875005906255844},{0.85,0.0,0.11906253937503915},{0.85,0.0,0.11906253937503915},{0.85,0.0,0.09937501968751986},{0.85,0.0,0.09937501968751986},{0.85,0.0,0.07968749999999944},{0.85,0.0,0.07968749999999944},};

// clang-format on


} // namespace render
} // namespace polyscope
