#include "geometrycentral/surface/manifold_surface_mesh.h"
#include "geometrycentral/surface/meshio.h"
#include "geometrycentral/surface/vertex_position_geometry.h"

#include "geometrycentral/surface/direction_fields.h"

#include "polyscope/polyscope.h"
#include "polyscope/surface_mesh.h"
#include "polyscope/pick.h"
#include "polyscope/curve_network.h"
#include "polyscope/point_cloud.h"

#include "glm/gtx/string_cast.hpp"

#include "args/args.hxx"
#include "imgui.h"

#include "neck_model.hpp"
#include "utils.hpp"
#include "benchmarker.hpp"

#include <set>
#include <queue>
#include <thread>
#include <algorithm>
#include <memory>

using namespace geometrycentral;
using namespace geometrycentral::surface;

std::unique_ptr<NeckModel> nm;
std::vector<Halfedge> best_cycle;
std::set<Face> best_cut;

// Polyscope visualization handle, to quickly add data to the surface
polyscope::SurfaceMesh *psMesh = NULL;

std::string mesh_name;

//
int r_hops = 25;

// Some algorithm parameters
int param1 = 10;
int cutidxs = 0;

int cut_index = 0;

int cycles_made = 0;
int cycle_sel = 0;
int bone_sel = 0;

float gpos[3] = {0., 0., 0.};
float gla[3] = {0., 0., 0.};

void myCallback()
{


  ImGui::SliderInt("Bone Select", &bone_sel, 0, (nm->skeleton_cycles_output).size()-1);
  if (ImGui::Button("Update Bone Select")) {
    nm->salient_cycles_output = nm->skeleton_cycles_output[bone_sel];
  }
  ImGui::SliderInt("Cycle Select", &cycle_sel, 0, (nm->salient_cycles_output).size()-1);

  if (ImGui::Button("Visualize Area")) {
    auto selected_cycle = nm->salient_cycles_output[cycle_sel];

    FaceData<bool> visited_f(*(nm->mesh)); // Visited Set of faces
    std::vector<std::array<double, 3>> cyclefaces(nm->mesh->nFaces(), {0.0, 1.0, 0.0});
    std::queue<Face> bfs_q;

    std::unordered_set<Face> banned_faces; // Banned faces from enquing in one iteration
      double area_sum = 0.0;
      for (auto he : selected_cycle)
      {
        // enqueue all the faces induced by one side of the cycle
        if (visited_f[he.face()] == false)
        {
          bfs_q.push(he.face());
          visited_f[he.face()] = true;
          cyclefaces[he.face().getIndex()] = {0.0,0.0,1.0};
        }
        // ban all the faces induced by the other side (this should prevent all crossings automatically)
        banned_faces.insert(he.twin().face());
      }

    while (!bfs_q.empty())
    {
      Face f = bfs_q.front();
      bfs_q.pop();
      area_sum += nm->geometry->faceAreas[f];
      for (Face g : f.adjacentFaces())
      {
        if (visited_f[g] == false && banned_faces.find(g) == banned_faces.end())
        {
          bfs_q.push(g);
          visited_f[g] = true;
          cyclefaces[g.getIndex()] = {0.0,0.0,1.0};
        }
      }
    }

    double c_length = 0.0;
    for (auto he : selected_cycle) {
      c_length += nm->geometry->edgeLengths[he.edge()];
    }
  
    double total_area = 0.0;
    for (size_t i = 0; i < nm->mesh->nFaces(); i++)
    {
      total_area += nm->geometry->faceAreas[i];
    }
    psMesh->addFaceColorQuantity("Cycle Faces", cyclefaces);
    std::cout << "Total Area: " << total_area << std::endl;
    std::cout << "Partial Area (Blue): " << area_sum << " Partial Area (Green): " << total_area - area_sum << std::endl;
    std::cout << "Cycle Length: " << c_length << std::endl;
    std::cout << "Tightness:" << min(area_sum, total_area - area_sum) / (c_length*c_length) << std::endl;
  }

  if (ImGui::Button("See Cycle")) {
    // std::cout << cycle_sel << std::endl;

    auto test_cycle = nm->salient_cycles_output[cycle_sel];

    // for (auto he : test_cycle) {
    //   std::cout << "(" << he.tipVertex() << " -> " << he.tailVertex() << ") -> ";
    // }
    // std::cout << std::endl;
    std::vector<std::array<double, 3>> cyclefaces(nm->mesh->nFaces(), {0.0,0.0,0.0});

    for (auto he : test_cycle) {
      Face f = he.face();
      cyclefaces[f.getIndex()] = {0.0, 1.0, 0.0};
    }

    // for (auto face : Y.second.adjacentFaces()) {
    //   cyclefaces[face.getIndex()] = {1.0, 0.0, 0.0};
    // }

    // for (auto face : Z.second.adjacentFaces()) {
    //   cyclefaces[face.getIndex()] = {0.0, 0.0, 1.0};
    // }
    auto surf = polyscope::getSurfaceMesh(mesh_name);
    surf->addFaceColorQuantity("cyclefaces", cyclefaces);
    std::vector<glm::vec3> output_ve;
    std::vector<std::array<size_t, 2>>   output_ed;
    int base_count = 0;
    for (size_t j = 0 ; j < test_cycle.size(); j++) {
      Halfedge he = test_cycle[j];
      Vector3 vertdat = nm->geometry->vertexPositions[he.tailVertex()];
      output_ve.push_back({vertdat.x, vertdat.y, vertdat.z});
      // output_ve.push_back(nm->geometry .tailVertex().getIndex);
      // std::cout << j << ", " << (j+1) % 37 << std::endl;
      output_ed.push_back({base_count + j, base_count +((j+1) % (test_cycle.size()))});
      // ecolors[he.edge().getIndex()] = {1.0, 0.0, 0.0};
    }
    base_count += test_cycle.size();
    auto curve2 = polyscope::registerCurveNetwork("cyclecurve", output_ve, output_ed);
    curve2->setColor({1.0,0.0,0.0});
  }

}

int main(int argc, char **argv)
{

  // Configure the argument parser
  args::ArgumentParser parser("geometry-central & Polyscope example project");
  args::Positional<std::string> inputFilename(parser, "mesh", "A mesh file.");
  args::Positional<int> r_hop(parser, "rhops", "Number of hops for candidate filtering");
  args::Positional<std::string> shouldshow(parser, "show", "if you should show");
  // Parse args
  try
  {
    parser.ParseCLI(argc, argv);
  }
  catch (args::Help &h)
  {
    std::cout << parser;
    return 0;
  }
  catch (args::ParseError &e)
  {
    std::cerr << e.what() << std::endl;
    std::cerr << parser;
    return 1;
  }

  // Make sure a mesh name was given
  if (!inputFilename)
  {
    std::cerr << "Please specify a mesh file as argument" << std::endl;
    return EXIT_FAILURE;
  }

  try{
    r_hops = args::get(r_hop);
  } catch (...) {
    r_hops = 25;
  }
  if (r_hops == 0){
    r_hops = 25;
  }

  polyscope::options::allowHeadlessBackends = true;
  // Initialize polyscope
  polyscope::init();

  // Set the callback function
  polyscope::state::userCallback = myCallback;

  //
  NeckModel nmtemp = NeckModel(args::get(inputFilename));
  nm = std::unique_ptr<NeckModel>(std::move(&nmtemp));
  // TEMP
  nm->_source = nm->mesh->vertex(0);

  mesh_name = polyscope::guessNiceNameFromPath(args::get(inputFilename));
  psMesh = polyscope::registerSurfaceMesh(mesh_name,
      nm->geometry->inputVertexPositions, nm->mesh->getFaceVertexList());
      
  auto perms = polyscopePermutations(*(nm->mesh));
  psMesh->setAllPermutations(perms);

  psMesh->setPosition(glm::vec3{0.,0.,0.});


  // Register Overlay curve network
  auto nodes = nm->geometry->inputVertexPositions;
  // Create edge list of vector of array of size 2 of size_t values from nodes indices
  std::vector<std::array<size_t, 2>> edges;
  for (Edge e : nm->mesh->edges())
  {
    edges.push_back({e.halfedge().vertex().getIndex(), e.halfedge().twin().vertex().getIndex()});
  }
  auto curve = polyscope::registerCurveNetwork("curve", nodes, edges);
  curve->setRadius(0.0002);
  curve->setEnabled(false);

  std::cout << "Running Algo" << std::endl;
  nm->_source = nm->mesh->vertex(12756);

  computeSkeleton(nm, r_hops);

  std::cout << "Taking Photos" << std::endl;
  polyscope::options::groundPlaneMode = polyscope::GroundPlaneMode::None;
  {
    using namespace polyscope;
    
    // Give ourselves some valid view
    view::lookAt(glm::vec3{1., 0., 0.},glm::vec3{0., 0., 0.});
    
    // Make Polyscope compute + set the best home view
    view::resetCameraToHomeView();
    polyscope::screenshot();

  }
  // polyscope::view::lookAt(pos,la);
  if (shouldshow) {
    polyscope::show();
  }
  // polyscope::show();

  polyscope::removeAllStructures();
  nm.release();
  nm = NULL;
  return EXIT_SUCCESS;
}
